#include "sim.h"
#include "local/sim/SharedGlobals.h"
#include "local/sim/Main.h"
#include "local/sim/Motion.h"
#include "local/sim/SoundPlay.h"
#include "local/sim/SimConfig.h"
#include "IPC/RegionRegistry.h"
#include "local/sim/Simulator.h"
#include "IPC/SharedObject.h"
#include "Shared/Config.h"
#include "Shared/string_util.h"
#include "Shared/StackTrace.h"
#include "IPC/SemaphoreManager.h"
#include "Events/EventRouter.h"
#include <iostream>
#include <sstream>
#include <iomanip>
#include <sys/ipc.h>
#include <signal.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <errno.h>
#include <pwd.h>
	
using namespace std;
	
const char * const sim::usage=
"[-h|--help] [-c|--config config-file] [var=val ...]";
	
//only true for the first process -- children should set this to false when they fork
bool sim::original=true;
	
SimConfig sim::config;
vector<string> sim::assignments;
	
/* Although I generally dislike the "can't have main function without a class declaration" style of java,
 * sometimes it does come in handy.  See the class notes for sim for more information. */
int main(int argc, const char* argv[]) {
	
	//This should match the ID of the process sent to manage_process by sim::run()
	// *must* be done before we create any shared regions to provide proper reference counting
	ProcessID::setID(ProcessID::SimulatorProcess);
			
	//initialize some threading globals
	Thread::initMainThread();
	
#ifndef USE_UNBACKED_SHM
	//append username to shared memory root avoid permission denied
	struct passwd * pw=getpwuid(getuid());
	if(pw!=NULL) {
		if(RCRegion::shmRoot[RCRegion::shmRoot.size()-1]=='/') {
			RCRegion::shmRoot[RCRegion::shmRoot.size()-1]='-';
			RCRegion::shmRoot+=pw->pw_name;
			RCRegion::shmRoot+='/';
		} else {
			RCRegion::shmRoot+=pw->pw_name;
		}
	}
#endif

	{
		sim s;
		if(!s.processCommandLine(argc,argv))
			return 2;
		//now the real meat begins
		if(!s.run()) //when this eventually returns, we're done (and all our children forks as well)
			return 1;
		return 0;
	}
}
	
sim::sim() :
	mutexSemMgr(),
	glob(),
	subj(),
	child(static_cast<pid_t>(-1))
{
	//what's the granularity of usleep on this platform?
	MutexLockBase::usleep_granularity=measure_usleep_cost();
	//use our own custom get_time routine
	project_get_time::get_time_callback=&sim_get_time;
	project_get_time::get_timeScale_callback=&sim_getTimeScale;
	
	//setup signal handlers
	signal(SIGHUP, handle_signal);
	signal(SIGINT, handle_signal);
	signal(SIGQUIT, handle_signal);
	signal(SIGILL, handle_signal);
	signal(SIGABRT, handle_signal);
	signal(SIGFPE, handle_signal);
	signal(SIGBUS, handle_signal);
	signal(SIGSEGV, handle_signal);
	signal(SIGSYS, handle_signal);
	signal(SIGPIPE, handle_signal);
	signal(SIGTERM, handle_signal);
	atexit(handle_exit);
		
#ifndef USE_UNBACKED_SHM
	//add config item for file backed memory
	sim::config.addEntry("TempStorageRoot",RCRegion::shmRoot,"The directory to use for temporary storage of file-backed shared memory regions");
#endif
	
	//Set up mutex's semaphore manager
	MutexLockBase::setSemaphoreManager(&(*mutexSemMgr));
	//Set up MessageQueue's semaphore manager
	MessageQueueBase::setSemaphoreManager(&(*mutexSemMgr));
	
	//Set up shared global parameters -- e.g. clock and runlevel info
	globals = &(*glob);
	sim::config.addEntry("Speed",globals->timeScale,"The speed at which to run the simulation, as a multiple of \"real-time\".\nFor example, '1' is normal, '0.5' is half-speed, '0' is paused.\nAny negative value requests non-realtime mode, where the clock is moved as fast as processing (or manual control) allows.");
	globals->simulatorTime=sim::config.initSimTime;
	sim::config.addEntry("Motion",globals->motion,"Parameters for motion simulation");
		
	//Set up the subject registration area
	ipc_setup = &(*subj);
		
	//everyone uses config and erouter, might as well set it up here
	::config = new Config();
	::config->setFileSystemRoot("ms");
	::config->readConfig("config/tekkotsu.cfg");
	::config->readConfig("config/sim_ovrd.cfg");
	::erouter = new EventRouter;
	
	//we won't have sensor values yet, but the system clock is good enough
	if(::config->main.seed_rng) {
		double t=TimeET().Value(); //current time with nanosecond resolution
		unsigned int * tm=reinterpret_cast<unsigned int*>(&t);
		unsigned int seed=tm[0]+tm[1];
		cout << "RNG seed=" << seed << endl;;
		srand(seed);
	}
}
	
bool sim::processCommandLine(int argc, const char* argv[]) {
	int i=0;
	
	//try to load the default configuration file
	const char * const defaultconfigfile="simulator.plist";
	struct stat sb;
	if(stat(defaultconfigfile,&sb)==0) {
		if(!sim::config.loadFile(defaultconfigfile)) {
			cerr << "Error loading default configuration file, may be malformed." << endl;
			return false;
		}
	} else
		sim::config.setLastFile(defaultconfigfile);
	
	//set the prompt from the binary name
	config.cmdPrompt=argv[i];
	if(config.cmdPrompt.rfind('/')!=string::npos)
		config.cmdPrompt=config.cmdPrompt.substr(config.cmdPrompt.rfind('/')+1);
	config.cmdPrompt+="> ";
	
	//now run through the rest of the arguments
	for(i++; i<argc; i++) {
		if(!strcmp(argv[i],"-h") || !strcmp(argv[i],"--help")) {
			cerr << argv[0] << ": " << usage << endl;
			return false;
		} else if(!strcmp(argv[i],"-c") || !strcmp(argv[i],"--config")) {
			if(!sim::config.loadFile(argv[++i]))
				return false;
		} else if(strchr(argv[i],'=')!=NULL) {
			string value=string_util::trim(strchr(argv[i],'=')+1);
			string key=string_util::trim(string(argv[i],strchr(argv[i],'=')-argv[i]));
			plist::ObjectBase* ob=sim::config.getEntry(key);
			if(ob==NULL)
				assignments.push_back(argv[i]); //might be a key which is added by Simulator, we'll come back to it once Simulator has been launched
			else if(plist::PrimitiveBase* pbp=dynamic_cast<plist::PrimitiveBase*>(ob)) {
				try {
					pbp->set(value);
				} catch(const XMLLoadSave::bad_format& bf) {
					cout << "'" << value << "' is a bad value for '" << key << "'" << endl;
					cout << bf.what() << endl;
					return false;
				} catch(const std::exception& e) {
					cout << "An exception occured: " << e.what() << endl;
					return false;
				}
			} else {
				cout << "Cannot assign to a dictionary ("<<key<<")" << endl;
				return false;
			}
		} else {
			cerr << "Unknown argument '" << argv[i] << "': assignments must be of the form 'var=value'" << endl;
			return false;
		}
	}
		
	return true;
}
	
bool sim::run() {
	//this will force all of the processes to wait at the end of construction
	//until we're done spawning all of them
	globals->level_count[SharedGlobals::CREATED]++;
		
	cout << "Spawning processes..." << endl;
	cout.setf(ios::left);
	cout << "  Initializing runlevel " << setw(12) << SharedGlobals::runlevel_names[SharedGlobals::CONSTRUCTING] << endl;
	cout.unsetf(ios::left);
	if(fork_process<Main>()) ;
	else if(fork_process<Motion>()) ;
	else if(fork_process<SoundPlay>()) ;
	else manage_process<Simulator>();
		
	//every process is going to pass through here on their way out
	globals->level_count[SharedGlobals::DESTRUCTED]++;
		
	return true;
}
	
sim::~sim() {
	MutexLockBase::setSemaphoreManager(NULL);
	MessageQueueBase::setSemaphoreManager(NULL);
	globals=NULL;
	ipc_setup=NULL;
	
	if(child==static_cast<pid_t>(-1)) // never got to the fork (or had an error at the fork)
		return;
	
	if(original)
		cout << ProcessID::getIDStr() << ": Waiting for children to exit..." << endl;
	if(child!=0) {
		int status;
		int res=waitpid(child,&status,0);
		if(res<0 && errno!=ECHILD)
			perror("wait");
		if(!WIFEXITED(status))
			cout << ProcessID::getIDStr() << ": waiting for " << child << "..." << endl;
		while(!WIFEXITED(status)) {
			sleep(1);
			res=waitpid(child,&status,0);
			if(res<0 && errno!=ECHILD)
				perror("wait");
		}
	}
}
	
void sim::wait_runlevel(SharedGlobals::runlevel_t level) {
	globals->lock.lock(ProcessID::getID());
	globals->level_count[level]++;
	if(globals->level_count[level]==1) {
		cout.setf(ios::left);
		cout << "Collecting for runlevel " << setw(12) << SharedGlobals::runlevel_names[level] << "  |=" << flush;
		cout.unsetf(ios::left);
	}
	cout << Process::getName() << '=' << flush;
	if(globals->level_count[level]==globals->level_count[SharedGlobals::CREATED])
		cout << "|  done" << endl;
	globals->lock.unlock();
	while(globals->level_count[level]!=globals->level_count[SharedGlobals::CREATED])
		usleep(150*1000);
	globals->lock.lock(ProcessID::getID());
	globals->lock.unlock();
}
	
template<class T>
void sim::manage_process() {
	//initialize the first runlevel
	globals->lock.lock(T::getID());
	globals->level_count[SharedGlobals::CONSTRUCTING]++;
	cout << setw(35) << T::getClassName() << ":  ProcessID::getID()=" << T::getID() << "   pid=" << getpid() << endl;
	T t;
	ASSERT(T::getID()==ProcessID::getID(),"Process ID set incorrectly!");
	globals->lock.unlock();
	while(globals->level_count[SharedGlobals::CONSTRUCTING]!=globals->level_count[SharedGlobals::CREATED])
		usleep(150*1000);
	globals->lock.lock(ProcessID::getID());
	globals->lock.unlock();
	
	//now just walk through each of the other steps
	wait_runlevel(SharedGlobals::STARTING);
	t.DoStart();
	wait_runlevel(SharedGlobals::RUNNING);
	t.run(); //should return if/when SharedGlobals::shutdown flag is set
	wait_runlevel(SharedGlobals::STOPPING);
	t.DoStop();
	wait_runlevel(SharedGlobals::DESTRUCTING);
}
	
template<class T>
bool sim::fork_process() {
	RCRegion::aboutToFork(T::getID());
	MutexLockBase::aboutToFork();
	//increment this *before* the fork to guarantee everyone knows to wait for the new process
	globals->level_count[SharedGlobals::CREATED]++;
	child=fork();
	if(child==static_cast<pid_t>(-1)) {
		cerr << "Unable to spawn simulator process!" << endl;
		exit(1);
	}
	if(child!=0) {
		manage_process<T>();
		return true;
	}
	original=false;
	return false;
}
	
void sim::handle_signal(int sig) {
	//	sigset_t mask_set;  /* used to set a signal masking set. */
	//	sigset_t old_set;   /* used to store the old mask set.   */
		
	/* reset the signal handler for next time */
	//	signal(sig, handle_signal);
	/* mask any further signals while we're inside the handler. */
	//	sigfillset(&mask_set);
	//	sigprocmask(SIG_SETMASK, &mask_set, &old_set);
		
	char * name=NULL;
	char defBuf[30];
	switch(sig) {
	case SIGINT: name="SIGINT"; break;
	case SIGQUIT: name="SIGQUIT"; break;
	case SIGBUS: name="SIGBUS"; break;
	case SIGSEGV: name="SIGSEGV"; break;
	case SIGTERM: name="SIGTERM"; break;
	case SIGABRT: name="SIGABRT"; break;
	case SIGFPE: name="SIGFPE"; break;
	case SIGPIPE: name="SIGPIPE"; break;
	case SIGHUP: name="SIGHUP"; break;
	default:
		name=defBuf;
		snprintf(name,30,"signal %d",sig);
		break;
	}
	cout << "*** ERROR " << Process::getName() << ": Received " << name << endl;
	
	static bool firstCall=true;
	if(!firstCall) {
		cerr << "Signal handler was recursively called, may be leaked IPC resources :(" << endl;
		cerr << "The 'ipcs' tool can be used to manually free these, if it becomes a problem. " << endl;
		cerr << "However, simply re-running will generally reclaim the previous buffers for you." << endl;
		exit(EXIT_FAILURE);
		return;
	}
	firstCall=false;
	
	stacktrace::displayCurrentStackTrace(15);
	
	cout << "*** ERROR " << Process::getName() << ": Engaging fault shutdown..." << endl;
	if(globals!=NULL && !globals->hadFault()) {
		if(!MutexLockBase::getSemaphoreManager()->hadFault())
			globals->lock.lock(ProcessID::getID());
		if(globals->level_count[SharedGlobals::CREATED]>0)
			globals->level_count[SharedGlobals::CREATED]--;
		else
			cout << "*** ERROR " << Process::getName() << ": level_count[CREATED] underflow" << endl;
		globals->signalShutdown();
		if(!MutexLockBase::getSemaphoreManager()->hadFault())
			globals->lock.unlock();
		globals->faultShutdown();
	}
	if(MutexLockBase::getSemaphoreManager()!=NULL && !MutexLockBase::getSemaphoreManager()->hadFault()) {
		cout << "*** ERROR " << Process::getName() << ": Dereferencing mutex SemaphoreManager" << endl;
		MutexLockBase::setSemaphoreManager(NULL); //reset to preallocated instance
		MutexLockBase::getSemaphoreManager()->faultShutdown();
	}
	cout << "*** ERROR " << Process::getName() << ": Dereferencing shared memory regions" << endl;
	RCRegion::faultShutdown();
	cout << "*** ERROR " << Process::getName() << ": Exiting..." << endl;
	exit(EXIT_FAILURE);
}
	
void sim::handle_exit() {
	static bool firstCall=true;
	if(!firstCall) {
		cerr << "handle_exit was recursively called" << endl;
		return;
	}
	firstCall=false;
	if(Process::getName().size()>0)
		cout << Process::getName() << ": Exiting..." << endl;
	if(RCRegion::NumberOfAttach()==0) {
		if(original)
			cout << "Clean shutdown complete.  Have a nice day." << endl;
		return;
	}
	cout << "*** ERROR " << Process::getName() << ": Exit with attached memory regions, engaging fault shutdown..." << endl;
	if(globals!=NULL && !globals->hadFault()) {
		if(!MutexLockBase::getSemaphoreManager()->hadFault())
			globals->lock.lock(ProcessID::getID());
		if(globals->level_count[SharedGlobals::CREATED]>0)
			globals->level_count[SharedGlobals::CREATED]--;
		else
			cout << "*** ERROR " << Process::getName() << ": level_count[CREATED] underflow" << endl;
		globals->signalShutdown();
		if(!MutexLockBase::getSemaphoreManager()->hadFault())
			globals->lock.unlock();
		globals->faultShutdown();
	} else {
		cerr << "*** ERROR " << Process::getName() << ": exit with previous global fault" << endl;
	}
	if(MutexLockBase::getSemaphoreManager()!=NULL && !MutexLockBase::getSemaphoreManager()->hadFault()) {
		cout << "*** ERROR " << Process::getName() << ": Dereferencing mutex SemaphoreManager" << endl;
		MutexLockBase::setSemaphoreManager(NULL); //reset to preallocated instance
		MutexLockBase::getSemaphoreManager()->faultShutdown();
	} else {
		cerr << "*** ERROR " << Process::getName() << ": exit with previous mutex fault" << endl;
	}
	cout << "*** ERROR " << Process::getName() << ": Dereferencing shared memory regions" << endl;
	RCRegion::faultShutdown();
	cout << "*** ERROR " << Process::getName() << ": Exiting..." << endl;
}
	
unsigned int sim::measure_usleep_cost() {
	usleep(50000); //to hopefully clear out the scheduler for the duration of our test
	const unsigned int TRIALS=50;
	TimeET mintime(1.0); //should definitely be less than a second
	for(unsigned int i=0; i<TRIALS; i++) {
		//measure usleep (plus overhead)
		TimeET cur;
		usleep(1); // at least 1 to avoid being a no-op
		TimeET elapsed(cur.Age());
		if(elapsed<mintime)
			mintime=elapsed;
	}
	usleep(50000); //to hopefully clear out the scheduler for the duration of our test
	TimeET minover(1.0); //should definitely be less than a second
	for(unsigned int i=0; i<TRIALS; i++) {
		//measure overhead
		TimeET cur;
		TimeET elapsed(cur.Age());
		if(elapsed<minover)
			minover=elapsed;
	}
	if(mintime<minover) { // something went wrong -- overhead is greater than total
		mintime=0L;
		cout << "usleep granularity couldn't be measured, default to 10ms" << endl;
	} else {
		//subtract overhead
		mintime-=minover;
		cout << "usleep granularity is " << mintime.Value()*1.0e6 << "us";
		if(mintime<2L) {
			mintime=2L;
			cout << ", reset to 2ms";
		}
		cout << endl;
	}
	return static_cast<unsigned>(mintime.Value()*1.0e6);
}
