//-*-c++-*-
#ifndef INCLUDED_Simulator_h_
#define INCLUDED_Simulator_h_

#include "Process.h"
#include "sim.h"
#include "IPC/SharedObject.h"
#include "IPC/MessageQueueStatusThread.h"
#include "SharedGlobals.h"
#include "Shared/plist.h"
#include "Shared/debuget.h"
#include "local/LoadImageThread.h"
#include <set>

class Simulator : public Process,  public plist::PrimitiveListener, public MessageQueueStatusThread::StatusListener {
public:
	//! constructor
	Simulator();
	
	~Simulator();

	virtual void DoStart();
	virtual void DoStop();
	virtual void run();
	
	static const char * getClassName() { return "Simulator"; }
	static ProcessID::ProcessID_t getID() { return ProcessID::SimulatorProcess; }
	
	static const char * getCameraQueueID() { return "CameraData"; }
	static const char * getSensorQueueID() { return "SensorData"; }
	static const char * getTimerWakeupID() { return "TimerWakeup"; }
	static const char * getMotionWakeupID() { return "MotionWakeup"; }
	static const char * getStatusRequestID() { return "StatusRequest"; }
	static const char * getCommandQueueID() { return "CommandQueue"; }
	
	virtual void plistValueChanged(const plist::PrimitiveBase& pl);

	virtual void messagesRead(MessageQueueBase& mq, unsigned int n);
	
	static void sendCommand(const std::string& cmd);

protected:
	class : public MessageQueueBase::MessageFilter {
	public:
		virtual bool filterSendRequest(RCRegion* rcr) {
			globals->motion.frameNumber++;
			ASSERT(globals->motion.frameNumber==LoadFileThread::recoverSN(rcr),"globals->motion.frameNumber does not match frame number sent! (" << globals->motion.frameNumber << " vs " << LoadFileThread::recoverSN(rcr));
			return true;
		}
	} frameCounter;
	
	class CommandThread : public Thread {
	public:
		virtual void * run();
	};
	enum step_t {
		STEP_NONE,
		STEP_CAMERA,
		STEP_SENSOR,
		STEP_TIMER,
		STEP_MOTION
	};
	
	void incrementTime();
	void sendTimerWakeup();
	unsigned int getNextFrame();
	void resetSpeedMode();
	
	void processRunlevel(SharedGlobals::runlevel_t curRunLevel);
	void processCommand(const std::string& line, bool addToHistory); //!< process an individual command
	static bool gotCommand(RCRegion* msg); //!< for commands coming from other processes via #commandQueue and #commandrecv
	
	void cmdQuit(const std::vector<std::string>& args);
	void cmdLoad(const std::vector<std::string>& args);
	void cmdSave(const std::vector<std::string>& args);
	void cmdRunlevel(const std::vector<std::string>& args);
	bool cmdSet(const std::vector<std::string>& args);
	void cmdRun(const std::vector<std::string>& args, bool isRelative);
	void cmdRun(const std::vector<std::string>& args);
	void cmdPause(const std::vector<std::string>& args);
	void cmdHelp(const std::vector<std::string>& args);
	void cmdStep(const std::vector<std::string>& args);
	void cmdStatus(const std::vector<std::string>& args);
	void cmdAdvance(const std::vector<std::string>& args);
	void cmdFreeze(bool v, const std::vector<std::string>& args);
	void cmdReset(const std::vector<std::string>& args);
	
	SharedObject<sim::CameraQueue_t> cameraQueue;
	SharedObject<sim::SensorQueue_t> sensorQueue;
	SharedObject<sim::TimerWakeup_t> timerWakeup;
	SharedObject<sim::MotionWakeup_t> motionWakeup;
	SharedObject<sim::StatusRequest_t> statusRequest;
	SharedObject<sim::EventQueue_t> events;
	typedef MessageQueue<10> CommandQueue_t;
	SharedObject<CommandQueue_t> commandQueue;
	
	MessageQueueStatusThread cameraStatus;
	MessageQueueStatusThread sensorStatus;
	MessageQueueStatusThread timerStatus;
	MessageQueueStatusThread motionStatus;
	MessageQueueStatusThread eventsStatus;

	class MessageReceiver * commandrecv;
	
	LoadImageThread vision;
	LoadFileThread sensors;
	std::set<unsigned int> frameTimes;
	float runSpeed;
	step_t step;
	unsigned int waitingSteps;
	SharedGlobals::runlevel_t curLevel;
	
	TimeET fullspeedWallStart; //!< "real" wall-clock time that full-speed mode was entered
	unsigned int fullspeedSimStart; //!< simulator time at which full-speed mode was entered
	TimeET lastFrameWallStart; //!< "real" wall-clock time that processing started on last frame (only valid in full-speed mode)
	float avgWallTime; //!< running average of frame processing time
	float avgSimTime; //!< running average of frame increments
	static const float avgSpeedupGamma; //!< gamma parameter for calculating running average in #avgFrameTime
	
	ThreadNS::Lock simLock;

private:
	Simulator(const Simulator&); //!< don't call (copy constructor)
	Simulator& operator=(const Simulator&); //!< don't call (assignment operator)
};

/*! @file
 * @brief Defines Simulator, which DESCRIPTION
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.28 $
 * $State: Exp $
 * $Date: 2006/09/28 20:42:52 $
 */

#endif
