#include "Simulator.h"
#include "Main.h"
#include "IPC/RegionRegistry.h"
#include "Shared/string_util.h"
#include "Shared/RobotInfo.h"
#include <iostream>
#include "SimConfig.h"
#include "Shared/debuget.h"
#include "Shared/MarkScope.h"
#include "IPC/MessageReceiver.h"

#ifndef DISABLE_READLINE
#  include <readline/readline.h>
#  include <readline/history.h>
#endif

using namespace std;

const float Simulator::avgSpeedupGamma=.99;

Simulator::Simulator()
: Process(getID(),getClassName()), frameCounter(),
cameraQueue(ipc_setup->registerRegion(Simulator::getCameraQueueID(),sizeof(sim::CameraQueue_t))),
sensorQueue(ipc_setup->registerRegion(Simulator::getSensorQueueID(),sizeof(sim::SensorQueue_t))),
timerWakeup(ipc_setup->registerRegion(Simulator::getTimerWakeupID(),sizeof(sim::TimerWakeup_t))),
motionWakeup(ipc_setup->registerRegion(Simulator::getMotionWakeupID(),sizeof(sim::MotionWakeup_t))),
statusRequest(ipc_setup->registerRegion(Simulator::getStatusRequestID(),sizeof(sim::StatusRequest_t))),
events(ipc_setup->registerRegion(Main::getEventsID(),sizeof(sim::EventQueue_t))),
commandQueue(ipc_setup->registerRegion(Simulator::getCommandQueueID(),sizeof(CommandQueue_t))),
cameraStatus(*cameraQueue), sensorStatus(*sensorQueue), timerStatus(*timerWakeup), motionStatus(*motionWakeup), eventsStatus(), commandrecv(NULL),
vision("images", ".*\\.(jpg|jpeg|png)$", 30, *cameraQueue,false),
sensors("sensors", ".*\\.pos$", 1000.f/NumFrames/FrameTime, *sensorQueue,false),
frameTimes(), runSpeed(1), step(STEP_NONE), waitingSteps(0), curLevel(SharedGlobals::CONSTRUCTING), 
fullspeedWallStart(), fullspeedSimStart(), lastFrameWallStart(), avgWallTime(), avgSimTime(),
simLock()
{
	new (&(*cameraQueue)) sim::CameraQueue_t;
	new (&(*sensorQueue)) sim::SensorQueue_t;
	new (&(*timerWakeup)) sim::TimerWakeup_t;
	new (&(*motionWakeup)) sim::MotionWakeup_t;
	new (&(*statusRequest)) sim::StatusRequest_t;
	new (&(*commandQueue)) CommandQueue_t;
	statusRequest->setOverflowPolicy(MessageQueueBase::WAIT);
	commandQueue->setOverflowPolicy(MessageQueueBase::WAIT);
	
	sim::config.writeParseTree();
	sim::config.addEntry("Sensors",sensors,"Settings for the loading of sensor values");
	sim::config.addEntry("Vision",vision,"Settings for the loading of camera frames");
	sim::config.readParseTree();
	
	motionWakeup->setOverflowPolicy(MessageQueueBase::DROP_OLDEST);
	timerWakeup->setOverflowPolicy(MessageQueueBase::DROP_OLDEST);
	cameraQueue->setOverflowPolicy(MessageQueueBase::DROP_OLDEST);
	sensorQueue->setOverflowPolicy(MessageQueueBase::DROP_OLDEST);
	sensorQueue->addMessageFilter(frameCounter);
	
	globals->timeScale.addPrimitiveListener(this);
	sensors.src.addPrimitiveListener(this);
	vision.src.addPrimitiveListener(this);
	
	vector<vector<string> > delayed;
	bool visSet=false, senSet=false;
	for(unsigned int i=0; i<sim::assignments.size(); i++) {
		vector<string> setarg;
		setarg.push_back("set");
		setarg.push_back(sim::assignments[i]);
		if(sim::assignments[i].find(".Source=")==string::npos) {
			if(!cmdSet(setarg))
				cerr << "Occurred while processing " << sim::assignments[i] << endl;
		} else {
			delayed.push_back(setarg); // delay setting of source until after options like Frozen or Speed have been set
			if(sim::assignments[i].find("Vision.")==0)
				visSet=true;
			if(sim::assignments[i].find("Sensors.")==0)
				senSet=true;
		}
	}
	for(unsigned int i=0; i<delayed.size(); i++) {
		//this avoids dropping initial frame(s)
		if(!cmdSet(delayed[i]))
			cerr << "Occurred while processing " << delayed[i][1] << endl;
	}
	if(!visSet)
		vision.loadFileList();
	if(!senSet)
		sensors.loadFileList();
	commandrecv=new MessageReceiver(*commandQueue,gotCommand);
	processRunlevel(SharedGlobals::CONSTRUCTING);
}

Simulator::~Simulator() {
	globals->timeScale.removePrimitiveListener(this);
	globals->motion.feedbackDelay.removePrimitiveListener(this);
	processRunlevel(SharedGlobals::DESTRUCTING);
}

void Simulator::DoStart() {
	Process::DoStart();
	eventsStatus.setMessageQueue(*events);
	processRunlevel(SharedGlobals::STARTING);
}

void Simulator::run() {
	for(unsigned int i=0; i<ProcessID::NumProcesses; ++i)
		cout << globals->processNames[i] << " pid=" << globals->pids[i] << ";  ";
	cout << endl;
	if(globals->timeScale!=0)
		runSpeed=globals->timeScale;

	resetSpeedMode();
	if(globals->timeScale<0)
		incrementTime();

	CommandThread cmdThread;
	cmdThread.start();
	if(sim::config.tgtRunlevel==SharedGlobals::RUNNING)
		Process::run();
	if(cmdThread.isRunning()) {
		cout << "\n\nShutdown requested, press 'enter' to continue...\n\n" << endl;
		cmdThread.stop();
	}

	if(sensors.isRunning())
		sensors.stop();
	if(vision.isRunning())
		vision.stop();
	
	if(globals->timeScale<0) {
		motionStatus.removeStatusListener(this);
		timerStatus.removeStatusListener(this);
		sensorStatus.removeStatusListener(this);
		cameraStatus.removeStatusListener(this);
	}
	globals->signalShutdown();
}

void Simulator::plistValueChanged(const plist::PrimitiveBase& pl) {
	if(&pl==&globals->timeScale) {
		get_time(); // force SharedGlobals to notice the change and update its state
		resetSpeedMode();
		if(globals->timeScale<0)
			incrementTime();
		timerWakeup->sendMessage(NULL);
		motionWakeup->sendMessage(NULL);
	} else if(&pl==&sensors.src || &pl==&vision.src) {
		// if both stream are using the same index file, keep them in sync across loops, regardless of (or in case of) their framerate differing
		if(sensors.src==vision.src && sensors.usingIndexFile()) {
			float looptime=max(sensors.getLoopTime(),vision.getLoopTime());
			sensors.setLoopTime(looptime);
			vision.setLoopTime(looptime);
		}
	} else {
		cerr << "WARNING: Simulator got a plistValueChanged for an unknown plist primitive";
	}
}

void Simulator::messagesRead(MessageQueueBase& mq, unsigned int /*n*/) {
	MarkScope l(simLock);
	if(globals->timeScale<0) {
		//clear corresponding bit in waitingSteps
		 if(&mq==&(*cameraQueue)) {
			//cout << "Camera read, ";
			waitingSteps&=~(1<<STEP_CAMERA);
		} else if(&mq==&(*sensorQueue)) {
			//cout << "Sensor read, ";
			waitingSteps&=~(1<<STEP_SENSOR);
		} else if(&mq==&(*timerWakeup)) {
			//cout << "Timer read, ";
			waitingSteps&=~(1<<STEP_TIMER);
		} else if(&mq==&(*motionWakeup)) {
			//cout << "Motion read, ";
			waitingSteps&=~(1<<STEP_MOTION);
		} else if(&mq==&(*events)) {
			//cout << "Main read event queue (" << events->getMessagesUnread() << "remain), ";
			// nothing to do, just waiting for main to catch up before incrementing
		} else {
			cout << "Unknown message base read (either you meant to add some code to Simulator::messagesRead, or why did you bother to register a listener?)" << endl;
		}
		//cout << " waiting " << waitingSteps << " events " << events->getMessagesUnread << endl;
		
		if(waitingSteps==0 && events->getMessagesUnread()==0) //if that was the last one we were waiting for -- go for the next!
			incrementTime();
	}
}

void Simulator::sendCommand(const std::string& cmd) {
	static unsigned int cmdSN=0;
	char msgname[30];
	snprintf(msgname,30,"SimCommand.%d.%d",ProcessID::getID(),cmdSN++);
	RCRegion * msg = new RCRegion(msgname,cmd.size());
	strcpy(msg->Base(),cmd.c_str());
	SharedObject<CommandQueue_t> commandQ(ipc_setup->registerRegion(Simulator::getCommandQueueID(),sizeof(CommandQueue_t)));
	commandQ->sendMessage(msg,true);
}

void Simulator::DoStop() {
	if(sensors.isRunning())
		sensors.join();
	if(vision.isRunning())
		vision.join();
	processRunlevel(SharedGlobals::STOPPING);
	Process::DoStop();
}


void* Simulator::CommandThread::run() {
	Simulator * simp=dynamic_cast<Simulator*>(Process::getCurrent());
	ASSERTRETVAL(simp!=NULL,"CommandThread not in Simulator!",NULL);
	simp->processRunlevel(SharedGlobals::RUNNING);
	return NULL;
}


void Simulator::resetSpeedMode() {
	if(globals->timeScale<=0) {
		if(vision.isRunning()) {
			vision.stop();
			//vision.join(); // can't join because runfor/runto pause might be triggered within LoadFileThread's get_time() call
		}
		if(sensors.isRunning()) {
			sensors.stop();
			//sensors.join(); // can't join because runfor/runto pause might be triggered within LoadFileThread's get_time() call
		}
	} else {
		if(sim::config.tgtRunlevel==SharedGlobals::RUNNING) { 
			if(!vision.isRunning() && (!vision.frozen || vision.heartbeat))
				vision.start();
			if(!sensors.isRunning() && (!sensors.frozen || sensors.heartbeat))
				sensors.start();
		}
	}
	if(globals->timeScale<0) {
		cameraQueue->setOverflowPolicy(MessageQueueBase::WAIT);
		sensorQueue->setOverflowPolicy(MessageQueueBase::WAIT);
		cameraStatus.addStatusListener(this);
		sensorStatus.addStatusListener(this);
		timerStatus.addStatusListener(this);
		motionStatus.addStatusListener(this);
		eventsStatus.addStatusListener(this);
		fullspeedWallStart.Set();
		fullspeedSimStart=globals->simulatorTime;
		lastFrameWallStart.Set();
		avgWallTime=avgSimTime=0;
	} else {
		cameraQueue->setOverflowPolicy(MessageQueueBase::DROP_OLDEST);
		sensorQueue->setOverflowPolicy(MessageQueueBase::DROP_OLDEST);
		eventsStatus.removeStatusListener(this);
		motionStatus.removeStatusListener(this);
		timerStatus.removeStatusListener(this);
		sensorStatus.removeStatusListener(this);
		cameraStatus.removeStatusListener(this);
	}
	if(globals->timeScale==0)
		globals->setAutoPauseTime(-1U);
}

void Simulator::incrementTime() {
	MarkScope l(simLock);
	waitingSteps=getNextFrame();
	if(waitingSteps==0)
		return;
	unsigned int next=*frameTimes.begin();
	if(next>globals->simulatorTime) {
		unsigned int adv=next-globals->simulatorTime;
		avgWallTime=avgWallTime*avgSpeedupGamma + lastFrameWallStart.Age().Value()*(1-avgSpeedupGamma);
		avgSimTime=avgSimTime*avgSpeedupGamma + adv*(1-avgSpeedupGamma);
		lastFrameWallStart.Set();
		//cout << "inc " << (avgSimTime/avgWallTime/1000) << " " << waitingSteps << endl;
		globals->simulatorTime=next;
	}
	if(waitingSteps & (1<<STEP_CAMERA))
		vision.advanceFrame(false);
	if(waitingSteps & (1<<STEP_SENSOR))
		sensors.advanceFrame(false);
	if(waitingSteps & (1<<STEP_TIMER))
		timerWakeup->sendMessage(NULL);
	if(waitingSteps & (1<<STEP_MOTION))
		motionWakeup->sendMessage(NULL);
	if(globals->getAutoPauseTime()<=globals->simulatorTime || (1<<step) & waitingSteps) {
		//that was the step we were waiting for, pause sim
		globals->timeScale=0;
		step=STEP_NONE;
		globals->setAutoPauseTime(-1U);
	}
}

unsigned int Simulator::getNextFrame() {
	frameTimes.clear();
	/*set<unsigned int>::iterator past=frameTimes.begin();
	while(past!=frameTimes.end() && *past<=globals->simulatorTime)
	past++;
	frameTimes.erase(frameTimes.begin(),past);*/
	unsigned int vis = vision.frozen && !vision.heartbeat ? -1U : vision.nextTimestamp();
	frameTimes.insert(vis);
	unsigned int sen = sensors.frozen && !sensors.heartbeat ? -1U : sensors.nextTimestamp();
	frameTimes.insert(sen);
	unsigned int tim=globals->getNextTimer();
	frameTimes.insert(tim);
	unsigned int mot=globals->getNextMotion();
	frameTimes.insert(mot);
	unsigned int next=*frameTimes.begin();
	//cout << "Testing: " << globals->simulatorTime << " => next camera: "<< vis << " next sensor: " << sen << " next timer: " << tim << " next motion: " << mot << " => " << next << endl;
	unsigned int steps=0;
	if(next!=-1U) {
		if(next==vis) {
			steps |= 1<<STEP_CAMERA;
		}
		if(next==sen) {
			steps |= 1<<STEP_SENSOR;
		}
		if(next==tim) {
			steps |= 1<<STEP_TIMER;
		}
		if(next==mot) {
			steps |= 1<<STEP_MOTION;
		}
	}
	return steps;
}

void Simulator::processRunlevel(SharedGlobals::runlevel_t curRunLevel) {
	curLevel=curRunLevel;
	while(sim::config.tgtRunlevel==curLevel && (!globals->isShutdown() || curLevel>SharedGlobals::RUNNING)) {
		string line;
#ifndef DISABLE_READLINE
		char* readin=readline(sim::config.cmdPrompt.c_str());
		if(readin==NULL) {
			cmdQuit(vector<string>());
			continue;
		}
		line=readin;
		free(readin);
#else
		cout << sim::config.cmdPrompt << flush;
		getline(cin,line);
		if(!cin) {
			cmdQuit(vector<string>());
			continue;
		}
#endif
		processCommand(line,true);
	}
}

void Simulator::processCommand(const std::string& line, bool addToHistory) {
	vector<string> args;
	vector<unsigned int> offs;
	if(!string_util::parseArgs(line,args,offs)) {
		cerr << "Mismatched quotes" << endl;
		return;
	}
	if(args.size()==0)
		return;
#ifndef DISABLE_READLINE
	/*		if(current_history()==NULL)
		cout << "current_history()==NULL" << endl;
	else if(current_history()->line==NULL)
		cout << "line == NULL" << endl;
	else if(line!=current_history()->line)
		cout << "line is different" << endl;
	else {
		cout << "not added" << endl;
		cout << "new line: " << line << endl;
		cout << "old line: " << current_history()->line << endl;
	}
	if(history_get(-1)==NULL)
		cout << "history_get(0)==NULL" << endl;
	else if(history_get(-1)->line==NULL)
		cout << "line 0 == NULL" << endl;
	else {
		cout << "line 0: " << history_get(-1)->line << endl;
		if(line!=history_get(-1)->line)
			cout << "line 0 is different" << endl;
		else
			cout << "0 not added" << endl;
	}
	*/	
	if(addToHistory && (current_history()==NULL || current_history()->line==NULL || line!=current_history()->line))
		add_history(line.c_str());
#endif
	if(args[0]=="shutdown" || args[0]=="quit" || args[0]=="exit") {
		cmdQuit(args);
	} else if(args[0]=="load") {
		cmdLoad(args);
	} else if(args[0]=="save") {
		cmdSave(args);
	} else if(args[0]=="runlevel") {
		cmdRunlevel(args);
	} else if(args[0]=="get_time") {
		cout << "Current time is " << get_time() << endl;
	} else if(args[0]=="set") {
		cmdSet(args);
	} else if(args[0]=="runto") {
		cmdRun(args,false);
	} else if(args[0]=="runfor") {
		cmdRun(args,true);
	} else if(args[0]=="run" || args[0]=="r") {
		cmdRun(args);
	} else if(args[0]=="pause" || args[0]=="p") {
		cmdPause(args);
	} else if(args[0]=="help") {
		cmdHelp(args);
	} else if(args[0]=="step") {
		cmdStep(args);
	} else if(args[0]=="status") {
		cmdStatus(args);
	} else if(args[0]=="advance") {
		cmdAdvance(args);
	} else if(args[0]=="freeze") {
		cmdFreeze(true,args);
	} else if(args[0]=="unfreeze") {
		cmdFreeze(false,args);
	} else if(args[0]=="reset") {
		cmdReset(args);
	} else {
		cout << "Unknown command '" << args[0] << "'" << endl;
	}
}

bool Simulator::gotCommand(RCRegion* msg) {
	Simulator * simp=dynamic_cast<Simulator*>(Process::getCurrent());
	ASSERTRETVAL(simp!=NULL,"gotCommand, but not within Simulator process!",true);
	simp->processCommand(msg->Base(),false);
	return true;
}	

void Simulator::cmdQuit(const std::vector<std::string>& /*args*/) {
	sim::config.tgtRunlevel=SharedGlobals::DESTRUCTED;
	globals->signalShutdown();
}
void Simulator::cmdLoad(const std::vector<std::string>& args) {
	if(args.size()>1)
		for(unsigned int i=1; i<args.size(); i++)
			sim::config.loadFile(args[i].c_str());
	else
		sim::config.loadFile(sim::config.getLastFile().c_str());
}
void Simulator::cmdSave(const std::vector<std::string>& args) {
	if(args.size()>1)
		for(unsigned int i=1; i<args.size(); i++)
			sim::config.saveFile(args[i].c_str());
	else
		sim::config.saveFile(sim::config.getLastFile().c_str());
}
void Simulator::cmdRunlevel(const std::vector<std::string>& args) {
	if(args.size()<=1) {
		sim::config.tgtRunlevel=static_cast<SharedGlobals::runlevel_t>(sim::config.tgtRunlevel+1);
		cout << "Moving to next runlevel: " << SharedGlobals::runlevel_names[sim::config.tgtRunlevel] << endl;
	} else {
		try {
			sim::config.tgtRunlevel=string_util::makeUpper(args[1]);
		} catch(...) {
			cout << "Invalid runlevel specification.  Try one of:\n\t";
			for(unsigned int i=0; i<SharedGlobals::NUM_RUNLEVELS; i++)
				cout << i << ' ' << SharedGlobals::runlevel_names[i] << ", ";
			cout << "\nCurrently at " << SharedGlobals::runlevel_names[curLevel] << endl;
			return;
		}
		if(sim::config.tgtRunlevel<curLevel) {
			sim::config.tgtRunlevel=curLevel;
			cout << "Cannot reduce runlevel, currently at " << curLevel << ' ' << SharedGlobals::runlevel_names[curLevel] << "\n\t";
			for(unsigned int i=0; i<SharedGlobals::NUM_RUNLEVELS; i++)
				cout << i << ' ' << SharedGlobals::runlevel_names[i] << ", ";
			cout << endl;
			return;
		} else if(sim::config.tgtRunlevel==curLevel) {
			cout << "Already at " << curLevel << ' ' << SharedGlobals::runlevel_names[curLevel] << "\n\t";
			for(unsigned int i=0; i<SharedGlobals::NUM_RUNLEVELS; i++)
				cout << i << ' ' << SharedGlobals::runlevel_names[i] << ", ";
			cout << endl;
			return;
		}
	}
	if(sim::config.tgtRunlevel>SharedGlobals::RUNNING && curLevel<=SharedGlobals::RUNNING)
		globals->signalShutdown();
}
bool Simulator::cmdSet(const std::vector<std::string>& args) {
	if(args.size()<=1) {
		cout << sim::config << endl;
		return false;
	}
	string arg;
	for(unsigned int i=1; i<args.size(); i++) {
		arg+=args[i];
		if(i!=args.size()-1)
			arg+=" ";
	}
	if(arg.rfind("=")==string::npos) {
		plist::ObjectBase* ob=sim::config.getEntry(arg);
		if(ob==NULL) {
			cout << "'" << arg << "' is unknown" << endl;
			return false;
		}
		cout << *ob << endl;
	} else {
		string value=string_util::trim(arg.substr(arg.find("=")+1));
		string key=string_util::trim(arg.substr(0,arg.find("=")));
		plist::ObjectBase* ob=sim::config.getEntry(key);
		if(ob==NULL) {
			cout << "'" << key << "' is unknown" << endl;
			return false;
		}
		if(plist::PrimitiveBase* pbp=dynamic_cast<plist::PrimitiveBase*>(ob)) {
			try {
				pbp->set(value);
				return true;
			} catch(const XMLLoadSave::bad_format& bf) {
				cout << "'" << value << "' is a bad value for '" << key << "'" << endl;
				cout << bf.what() << endl;
			} catch(const std::exception& e) {
				cout << "An exception occured: " << e.what() << endl;
			}
		} else {
			cout << "Cannot assign to a dictionary" << endl;
			return false;
		}
	}
	return false; //exception occurred
}
void Simulator::cmdRun(const std::vector<std::string>& args, bool isRelative) {
	if(args.size()<=1) {
		cout << "runfor/runto requires an argument" << endl;
		return;
	}
	if(isRelative)
		globals->setAutoPauseTime(get_time()+atoi(args[1].c_str()));
	else
		globals->setAutoPauseTime(atoi(args[1].c_str()));
	if(globals->timeScale==0)
		globals->timeScale=runSpeed;
}
void Simulator::cmdRun(const std::vector<std::string>& args) {
	if(args.size()<=1) {
		if(globals->timeScale!=0) {
			cout << "Already running" << endl;
			return;
		}
		globals->timeScale=runSpeed;
	} else {
		float speed=atof(args[1].c_str());
		if(speed!=0)
			runSpeed=speed;
		globals->timeScale=speed;
	}
}
void Simulator::cmdPause(const std::vector<std::string>& args) {
	if(globals->timeScale==0) {
		if(find(args.begin(),args.end(),"quiet")==args.end())
			cout << "Already paused" << endl;
		return;
	}
	runSpeed=globals->timeScale;
	globals->timeScale=0;
}
void Simulator::cmdHelp(const std::vector<std::string>& args) {
	map<string,string> syntax;
	syntax["load"]="[file]";
	syntax["save"]="[file]";
	syntax["runlevel"]="[";
	for(unsigned int i=0; i<SharedGlobals::NUM_RUNLEVELS; i++) {
		stringstream ss;
		ss << i << "|" << SharedGlobals::runlevel_names[i];
		if(i!=SharedGlobals::NUM_RUNLEVELS-1)
			ss << " | ";
		syntax["runlevel"]+=ss.str();
	}
	syntax["runlevel"]+="]";
	syntax["get_time"]="";
	syntax["set"]="[var=value]";
	syntax["runto"]="time";
	syntax["runfor"]="time";
	syntax["run"]="[speed]";
	syntax["pause"]="";
	syntax["step"]="[camera|sensor|timer|motion]";
	syntax["status"]="[Main|Motion|SoundPlay|Simulator|all]*";
	syntax["advance"]=syntax["freeze"]=syntax["unfreeze"]="[camera|sensors|all]*";
	syntax["reset"]="[camera|sensors|all]";
	
	map<string,string> help;
	
	help["load"]="Load simulator configuration from file; if file unspecified, defaults to 'simulator.plist'.\n"
		"Note that these files are human-readable XML (with comments!), and you can remove values to specify only a subset of settings.";
	
	help["save"]="Save simulator configuration to file; if file unspecified, defaults to 'simulator.plist'.\n"
		"Note that these files are human-readable XML (with comments!), and you can remove values to specify only a subset of settings.";
	
	help["runlevel"]="You can specify a runlevel to move to, or if unspecified, the next one.\n"
		"You can only move forward runlevels, not backward.  Usually you'll only need RUNNING, "
		"unless you are debugging startup/shutdown code or the simulator itself.";
	
	help["get_time"]="Displays the simulator time.";
	
	help["set"]="Sets simulator configuration variables.  Without any arguments, displays all available variables and their current values.\n"
		"Type 'help set <variable>' to get more information about a particular variable.";
	
	help["runto"]="Triggers 'run' until the simulator time reaches the specified value and then pauses.";
	
	help["runfor"]="Triggers 'run' until the simulator time has moved by the specified number of milliseconds, then pauses.";
	
	help["run"]="Resets simulator speed to last non-zero value (i.e. value prior to last 'pause'), can override by passing a new value as argument.  Can be abbreviated 'r'.";
	
	help["pause"]="Equivalent to 'set Speed=0'.  Can be abbreviated 'p'.  Stops the flow of time within the simulator.";
	
	help["step"]="Runs at \"full\" speed until the next indicated time frame, or the next available frame if no type is specified.\n"
		"See 'status' for available frames.";
	
	help["status"]="Displays a status report regarding current time, upcoming keyframes, and semaphore usage.  Specify one or more processes to get more in-depth, per-process status reports.";
	
	help["advance"]="Sends the next frame for the specified queue(s) in their listed order (can be listed more than once).\n"
		"Disregards timestamp information, and doesn't advance time, unlike 'step' command.  No arguments and \"all\" is the same as \"sensors camera\".";
	
	help["freeze"]="Equivalent to 'set queue.Frozen=true'.\n"
		"Stops sending frames from the specified queue(s), but still allows time to move (unlike 'pause').  No arguments is the same as \"all\".  See 'advance' and 'unfreeze'.";
	
	help["unfreeze"]="Equivalent to 'set queue.Frozen=false'.\n"
		"Begin sending frames from the specified queue(s) again.  Timestamps for the file listing are offset by the time spent frozen minus frames advanced so the queue(s) will continue from their current position.  No arguments is the same as \"all\".";
	
	help["reset"]="Moves the specified data queue(s) back to the first entry in their list.";
	
	if(args.size()==1) {
		cout << "Available commands: " << endl;
		for(map<string,string>::const_iterator it=help.begin(); it!=help.end(); ++it) {
			cout << '\t' << it->first << " " << syntax[it->first] << endl;
		}
		cout << "type 'help <command>' for more information" << endl;
	} else {
		if(help.find(args[1])==help.end()) {
			cout << "The command '"<< args[1] << "' was not found" << endl;
			return;
		}
		if(args.size()==2) {
			cout << args[1] << " " << syntax[args[1]] << endl;
			cout << help[args[1]] << endl;
		} else {
			if(args[1]=="set") {
				plist::ObjectBase* ob=sim::config.getEntry(args[2]);
				if(ob==NULL) {
					cout << "'" << args[2] << "' is unknown" << endl;
					return;
				}
				size_t n=args[2].rfind('.');
				if(n==string::npos)
					cout << sim::config.getComment(args[2]) << endl;
				else {
					ob=sim::config.getEntry(args[2].substr(0,n));
					if(const plist::Dictionary * dict=dynamic_cast<const plist::Dictionary*>(ob))
						cout << dict->getComment(args[2].substr(n+1)) << endl;
					else
						cout << "'" << args[2].substr(0,n) << "' is not a dictionary" << endl;
				}
			} else {
				cout << args[1] << " " << syntax[args[1]] << endl;
				cout << help[args[1]] << endl;
			}
		}
	}
}
void Simulator::cmdStep(const std::vector<std::string>& args) {
	if(args.size()<=1) {
		if(globals->timeScale!=0)
			globals->timeScale=0;
		step=STEP_NONE;
		incrementTime();
		return;
	}
	if(args.size()>2) {
		cout << args[0] << " takes 0 or 1 arguments; " << args.size()-1 << " supplied" << endl;
		return;
	}
	if(args[1]=="camera")
		step=STEP_CAMERA;
	else if(args[1]=="sensor" || args[1]=="sensors")
		step=STEP_SENSOR;
	else if(args[1]=="timer")
		step=STEP_TIMER;
	else if(args[1]=="motion")
		step=STEP_MOTION;
	else {
		cout << args[1] << " is not a valid argument for 'step'.  Type 'help step'." << endl;
		return;
	}
	if(step==STEP_CAMERA && vision.frozen && !vision.heartbeat) {
		cout << "Camera queue is frozen and has no heartbeat, cannot step (use 'advance' instead)" << endl;
		step=STEP_NONE;
	} else if(step==STEP_SENSOR && sensors.frozen && !sensors.heartbeat) {
		cout << "Sensor queue is frozen and has no heartbeat, cannot step (use 'advance' instead)" << endl;
		step=STEP_NONE;
	} else {
		unsigned int steps=getNextFrame();
		if((1<<step) & steps) { // the desired step is the next step -- just increment
			if(globals->timeScale!=0)
				globals->timeScale=0;
			step=STEP_NONE;
			incrementTime();
		} else if(globals->timeScale!=-1)
			globals->timeScale=-1;
	}
}
void Simulator::cmdStatus(const std::vector<std::string>& args) {
	cout << "Speed is " << static_cast<float>(globals->timeScale);
	if(globals->timeScale<0)
		cout << " (full speed mode: avg speed=" << ((globals->simulatorTime-fullspeedSimStart)/fullspeedWallStart.Age().Value()/1000) << "x, "
			<< " current speed=" << (avgSimTime/avgWallTime/1000) << "x)";
	cout << endl;
	cout << "Current time is " << get_time() << endl;
	unsigned int vis=vision.nextTimestamp();
	unsigned int sen=sensors.nextTimestamp();
	unsigned int tim=globals->getNextTimer();
	unsigned int mot=globals->getNextMotion();
	cout << "Next camera: ";
	if(vision.numLoaded()==0) cout << "(none)"; else {
		if(vision.frozen)
			cout << "Frozen@";
		cout << vision.getNextFrame();
		if(!vision.frozen || vision.heartbeat) {
			if(vision.frozen && vision.heartbeat && vision.getNextFrame()!="heartbeat")
				cout << " heartbeat";
			cout << " scheduled at " << vis;
		}
	}
	cout << endl;
	cout << "Next sensor: ";
	if(sensors.numLoaded()==0) cout << "(none)"; else {
		if(sensors.frozen)
			cout << "Frozen@";
		cout << sensors.getNextFrame();
		if(!sensors.frozen || sensors.heartbeat) {
			if(sensors.frozen && sensors.heartbeat && sensors.getNextFrame()!="heartbeat")
				cout << " heartbeat";
			cout << " scheduled at " << sen;
		}
	}
	cout << endl;
	cout << "Next timer: ";
	if(tim==-1U) cout << "(none)"; else cout << tim;
	cout << endl;
	cout << "Next motion: ";
	if(mot==-1U) cout << "(none)"; else cout << mot;
	cout << endl;
	unsigned int semUsed=MutexLockBase::getSemaphoreManager()->used();
	unsigned int semMax=semUsed+MutexLockBase::getSemaphoreManager()->available();
	cout << "Semaphores used: " << semUsed << "/" << semMax << " (" << ((semUsed*10000+5)/semMax/10)/10.f << "%)" << endl;
	cout << endl;
	if(args.size()>1) {
		SemaphoreManager::semid_t sem=statusRequest->addReadStatusListener();
		for(unsigned int i=1; i<args.size(); ++i) {
			RCRegion * region=new RCRegion(args[i].size()+1);
			strncpy(region->Base(),args[i].c_str(),args[i].size()+1);
			statusRequest->sendMessage(region);
			region->RemoveReference();
		}
		//wait until they're done to put the prompt back up
		if(sem!=statusRequest->getSemaphoreManager()->invalid()) {
			statusRequest->getSemaphoreManager()->lower(sem,args.size()-1);
			statusRequest->removeReadStatusListener(sem);
		}
		//check to see if we're included:
		//haha, now I remember why I don't use functional programming
		/*if(find_if(args.begin()+1,args.end(),not1(__gnu_cxx::compose1(bind2nd(ptr_fun(strcasecmp),getName().c_str()),mem_fun_ref(&string::c_str))))!=args.end())
			statusReport(cout);*/
		for(unsigned int i=1; i<args.size(); ++i) {
			if(strcasecmp(args[i].c_str(),getName().c_str())==0 || strcasecmp(args[i].c_str(),"all")==0) {
				statusReport(cout);
				cout << endl;
			}
		}
	}
}
void Simulator::cmdAdvance(const std::vector<std::string>& args) {
	if(curLevel!=SharedGlobals::RUNNING) {
		cout << args[0] << " can only be used in the RUNNING runlevel" << endl;
		return;
	}
	std::string senstr="sensors";
	std::string camstr="camera";
	std::vector<std::string> queuenames;
	bool isAll=false;
	if(args.size()<=1) { // no arguments supplied, advance all
		queuenames.push_back(senstr);
		queuenames.push_back(camstr);
		isAll=true;
	}
	for(unsigned int i=1; i<args.size(); ++i) {
		if(args[i]==camstr)
			queuenames.push_back(camstr);
		else if(args[i]==senstr)
			queuenames.push_back(senstr);
		else if(args[i]=="all") {
			queuenames.push_back(senstr);
			queuenames.push_back(camstr);
			isAll=true;
		} else {
			cout << "invalid argument for advance command: " << args[i] << endl;
			return;
		}
	}
	for(std::vector<std::string>::iterator it=queuenames.begin(); it!=queuenames.end(); ++it) {
		LoadFileThread * lft=NULL;
		MessageQueueBase* q=NULL;
		if(*it==camstr) {
			lft=&vision;
			q=&(*cameraQueue);
		} else if(*it==senstr) {
			lft=&sensors;
			q=&(*sensorQueue);
		} else {
			cout << "Simulator: internal error, invalid queue " << *it << endl;
			return;
		}
		SemaphoreManager::semid_t sem=q->addReadStatusListener(); //register read status listener before sending!
		bool sent=lft->advanceFrame(true); // send frame
		if(!sent) { // no data in queue
			sent=lft->heartbeat; // but there may have been a heartbeat?
			if(!isAll) {
				// only report empty queue if the queue was explicitly specified
				cout << "No data in " << *it << " queue";
				if(lft->heartbeat)
					cout << ", sent heartbeat";
				cout << endl;
			}
		}
		if(sent)
			q->getSemaphoreManager()->lower(sem,true); //block until we know message was read
		q->removeReadStatusListener(sem);
	}
}
void Simulator::cmdFreeze(bool v, const std::vector<std::string>& args) {
	std::string senstr="sensors";
	std::string camstr="camera";
	std::set<std::string> queuenames;
	if(args.size()<=1) {
		queuenames.insert(camstr);
		queuenames.insert(senstr);
	}
	for(unsigned int i=1; i<args.size(); ++i) {
		if(args[i]==camstr)
			queuenames.insert(camstr);
		else if(args[i]==senstr)
			queuenames.insert(senstr);
		else if(args[i]=="all") {
			queuenames.insert(camstr);
			queuenames.insert(senstr);
		} else {
			cout << "invalid argument for freeze/unfreeze command: " << args[i] << endl;
			return;
		}
	}
	for(std::set<std::string>::iterator it=queuenames.begin(); it!=queuenames.end(); ++it) {
		if(*it==camstr)
			vision.frozen=v;
		else if(*it==senstr)
			sensors.frozen=v;
		else {
			cout << "Simulator: internal error, invalid queue " << *it << endl;
			return;
		}
	}
}
void Simulator::cmdReset(const std::vector<std::string>& args) {
	std::string senstr="sensors";
	std::string camstr="camera";
	for(unsigned int i=1; i<args.size(); ++i) {
		if(args[i]==camstr)
			vision.setFrame(0);
		else if(args[i]==senstr)
			sensors.setFrame(0);
		else if(args[i]=="all") {
			vision.setFrame(0);
			sensors.setFrame(0);
		} else
			cout << "invalid argument for reset command: " << args[i] << endl;
	}
}

/*! @file
 * @brief 
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.62 $
 * $State: Exp $
 * $Date: 2006/09/26 21:42:50 $
 */

