#include "Motion.h"
#include "Main.h"
#include "SoundPlay.h"
#include "Simulator.h"
#include "MotionExecThread.h"
#include "IPC/RegionRegistry.h"
#include "IPC/MessageReceiver.h"
#include "Motion/Kinematics.h"
#include "Wireless/Wireless.h"
#include "Events/EventRouter.h"
#include "Shared/MarkScope.h"

#include "Events/EventBase.h"
#include "Events/LocomotionEvent.h"
#include "Events/TextMsgEvent.h"
#include "Events/VisionObjectEvent.h"

Motion::Motion()
	: Process(getID(),getClassName()),
	sounds(ipc_setup->registerRegion(SoundPlay::getSoundPlayID(),sizeof(sim::SoundPlayQueue_t))),
	motions(ipc_setup->registerRegion(getMotionCommandID(),sizeof(sim::MotionCommandQueue_t))),
	events(ipc_setup->registerRegion(Main::getEventsID(),sizeof(sim::EventQueue_t))),
	sensorFrames(ipc_setup->registerRegion(Simulator::getSensorQueueID(),sizeof(sim::SensorQueue_t))),
	statusRequest(ipc_setup->registerRegion(Simulator::getStatusRequestID(),sizeof(sim::StatusRequest_t))),
	motionmanager(ipc_setup->registerRegion(getMotionManagerID(),sizeof(MotionManager))),
	soundmanager(ipc_setup->registerRegion(SoundPlay::getSoundManagerID(),sizeof(SoundManager))),
	worldstatepool(ipc_setup->registerRegion(Main::getWorldStatePoolID(),sizeof(WorldStatePool))),
	motionWakeup(ipc_setup->registerRegion(Simulator::getMotionWakeupID(),sizeof(sim::MotionWakeup_t))),
	motionProf(ipc_setup->registerRegion(getMotionProfilerID(),sizeof(motionProfiler_t))),
	etrans(NULL), sensrecv(NULL), statusrecv(NULL), wakeuprecv(NULL), motionExec(NULL),
	wireless_thread(), motionLock(*worldstatepool), lastSensorSN(-1U)
{
	new (&(*motions)) sim::MotionCommandQueue_t;
	new (&(*motionmanager)) MotionManager;
	new (&(*motionProf)) motionProfiler_t;
	motman=&(*motionmanager);
	motman->InitAccess(*motions,motionLock);
	sndman=&(*soundmanager);
	state=NULL;
	::motionProfiler=&(*motionProf);
	
	//Setup wireless
	wireless = new Wireless();
	sout=wireless->socket(SocketNS::SOCK_STREAM,Wireless::WIRELESS_DEF_RECV_SIZE,Wireless::WIRELESS_DEF_SEND_SIZE*12);
	serr=wireless->socket(SocketNS::SOCK_STREAM,Wireless::WIRELESS_DEF_RECV_SIZE,Wireless::WIRELESS_DEF_SEND_SIZE*4);
	wireless->setDaemon(sout);
	wireless->setDaemon(serr);
	serr->setFlushType(SocketNS::FLUSH_BLOCKING);
	sout->setTextForward();
	serr->setForward(sout);

	//Setup Kinematics
	kine=new Kinematics();
	
	//need to register any events which we might be sending or receiving
	EventTranslator::registerPrototype<EventBase>();
	EventTranslator::registerPrototype<LocomotionEvent>();
	EventTranslator::registerPrototype<TextMsgEvent>();
	EventTranslator::registerPrototype<VisionObjectEvent>();

	//EventRouter and Config are set up for all processes by main() before fork
}

Motion::~Motion() {
	delete etrans;
	etrans=NULL;
	MotionManager::setTranslator(NULL);
}

void Motion::DoStart() {
	Process::DoStart();
	//These are constructed by other processes, so need to wait
	//until the construction runlevel is complete before we access them
	sndman->InitAccess(*sounds);
	etrans=new IPCEventTranslator(*events);
	MotionManager::setTranslator(etrans);
	
	// Set up Event Translator to trap and send events to main process
	//send everything over except erouter events
	for(unsigned int i=0; i<EventBase::numEGIDs; i++)
		if(i!=EventBase::erouterEGID)
			erouter->addTrapper(etrans,static_cast<EventBase::EventGeneratorID_t>(i));
	
	wireless_thread.start();
	sensrecv=new MessageReceiver(*sensorFrames,gotSensors);
	statusrecv=new MessageReceiver(*statusRequest,statusReport);
	wakeuprecv=new MessageReceiver(*motionWakeup,gotWakeup,false);

	motionExec=new MotionExecThread(motionLock);
}


void Motion::run() {
	motionExec->reset(); //starts the thread if appropriate
	wakeuprecv->start();
	Process::run();
	wakeuprecv->stop();
}

void Motion::DoStop() {
	wakeuprecv->finish();
	statusrecv->finish();
	sensrecv->finish();

	delete motionExec;
	motionExec=NULL;
	delete wakeuprecv;
	wakeuprecv=NULL;
	delete statusrecv;
	statusrecv=NULL;
	delete sensrecv;
	sensrecv=NULL;
	
	erouter->removeTrapper(etrans);
	motman->RemoveAccess();

	wireless_thread.stop();
	wireless_thread.join();
	
	Process::DoStop();
}

bool Motion::gotWakeup(RCRegion* /*msg*/) {
	Motion * motion=dynamic_cast<Motion*>(Process::getCurrent());
	ASSERTRETVAL(motion!=NULL,"gotWakeup, but not within motion process!",true);
	ASSERTRETVAL(motion->motionExec!=NULL,"motionExec thread is NULL when motion wakeup received",true);
	
	MarkScope l(motion->motionLock);
	if(globals->timeScale<=0 && get_time()>=globals->getNextMotion())
		motion->motionExec->poll();
	motion->motionExec->reset(); //reset will set globals->getNextMotion(), so have to do this after the poll itself
	return true;
}

bool Motion::gotSensors(RCRegion* msg) {
	Motion * motion=dynamic_cast<Motion*>(Process::getCurrent());
	ASSERTRETVAL(motion!=NULL,"gotSensors, but not within Motion process!",true);
	
	PROFSECTION("GotSensorFrame()",*motionProfiler);
	MarkScope l(motion->motionLock);
	
	EntryPoint::WorldStateWrite wsw(-1U);
	try {
		WorldStatePool::UpdateInfo * info=motion->worldstatepool->isUnread(*msg,globals->motion.frameNumber,motion->lastSensorSN,globals->motion.feedbackDelay>=0,globals->motion.override);
		int dif=0;
		if(info!=NULL) {
			bool generateFeedback=((info->payloadSize==0 && !info->dataInQueue) || globals->motion.override) && globals->motion.feedbackDelay>=0;
			const PostureEngine* pose = generateFeedback ? &motion->motionExec->getPostureFeedback() : NULL;
			wsw.frame=info->frameNumber;
			MarkScope lw(motion->motionLock, wsw);
			if(state==NULL || wsw.frame<globals->motion.frameNumber) {
				//a newer one came in while we were waiting for the lock (we'll report the accumulated drops next time we get one we don't drop)
				return true;
			}
			ASSERT((wsw.getStatus()&WorldStatePool::FEEDBACK_APPLIED)==0,"motion feedback already applied?");
			bool completedPartial=!wsw.getComplete() && (wsw.getStatus()&WorldStatePool::SENSORS_APPLIED)!=0;
			if(generateFeedback && info->verbose)
				cout << "Motion filling in sensor feedback" << endl;
			ASSERT(!completedPartial || generateFeedback,"partially complete, yet no feedback?");
			if(motion->worldstatepool->read(*info,wsw,generateFeedback,globals->motion.zeroPIDFeedback,pose)) {
				if(wsw.frame-motion->lastSensorSN!=1 && info->verbose)
					cout << ProcessID::getIDStr() << " dropped " << (wsw.frame-motion->lastSensorSN-1) << " sensor frame(s)" << endl;
				motion->lastSensorSN=wsw.frame;
			}
			if(completedPartial && wsw.getComplete()) { // was partial before, now completed as well
				dif=state->lastSensorUpdateTime - wsw.src->lastSensorUpdateTime;
				ASSERT(dif>=0,"sensor update time is negative? " << dif << ' ' << state << ' ' << wsw.src);
			}
		}
		if(dif>0) {
			// need to notify main that the sensor update it started is now complete
			// do it outside the write lock so if main gets this instantly it can use the new data
			// some sleight of hand here -- we've previously registered an IPCEventTranslator as an event trapper.  By posting here, the event will be sent to the Main process
			erouter->postEvent(EventBase::sensorEGID,SensorSourceID::UpdatedSID,EventBase::statusETID,dif,"SensorSourceID::UpdatedSID",1);
		}
	} catch(const std::exception& ex) {
		if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during sensor processing",&ex))
			throw;
	} catch(...) {
		if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during sensor processing",NULL))
			throw;
	}
	
	return true;
}
	

/*! @file
 * @brief 
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.21 $
 * $State: Exp $
 * $Date: 2006/08/31 19:01:46 $
 */

