#include "LoadImageThread.h"

#include "Shared/debuget.h"

#include <png.h>
extern "C" {
#include <jpeglib.h>

	struct png_read_mem_status {
		png_bytep buf;
		size_t bufsize;
		size_t offset;
	};
	void user_read_png_data(png_structp png_ptr, png_bytep data, png_size_t length) {
		png_read_mem_status* status=(png_read_mem_status*)(png_get_io_ptr(png_ptr));
		size_t endoff=status->offset+length;
		if(endoff<=status->bufsize) {
			memcpy(data,status->buf+status->offset,length);
			status->offset=endoff;
		} else {
			png_error(png_ptr, "Read Error - ran out of file");
		}
	}
	
}
#ifndef LOADFILE_NO_MMAP
#  include "Shared/jpeg-6b/jpeg_mem_src.h"
#  include <fcntl.h>
#  include <errno.h>
#  include <sys/types.h>
#  include <sys/mman.h>
#  include <sys/stat.h>
#endif

using namespace std;

LoadImageThread::LoadImageThread(std::string source, std::string filter, float fps, MessageQueueBase& messages, bool doLoad/*=true*/)
	: LoadFileThread(source,filter,fps,messages,false),
	cinfo(new jpeg_decompress_struct), jerr(new jpeg_error_mgr), frameIndex(0), layer(0)
{
	cinfo->err = jpeg_std_error(jerr);
	jpeg_create_decompress(cinfo);
	addEntry("Layer",layer,"Controls the resolution layer at which the image should be processed.\n0 indicates \"automatic\" mode (picks layer closest to image's resolution), positive numbers indicate the resolution layer directly.\nNegative values are relative to the number of layers marked available by the vision setup, so that typically -1 would correspond to the \"double\" layer, and -2 would correspond to the \"full\" layer.");
	if(doLoad)
		loadFileList(false,false);
}

LoadImageThread::~LoadImageThread() {
	jpeg_destroy_decompress(cinfo);
	delete cinfo;
	delete jerr;
}

bool LoadImageThread::loadFile(const std::string& file, RCRegion*& data) {
	struct stat statbuf;
	if(stat(file.c_str(),&statbuf)!=0) {
		perror("LoadImageThread::loadFile");
		return false;
	}
#ifdef LOADFILE_NO_MMAP
	FILE * infile= fopen(file.c_str(), "rb");
	if (infile==NULL) {
		int err=errno;
		cerr << "LoadImageThread::loadFile(): Could not open '" << file << "' (fopen:" << strerror(err) << ")" << endl;
		return false;
	}
	JOCTET* inbuf=static_cast<JOCTET*>(new char[statbuf.st_size]);
	if(!fread(inbuf,statbuf.st_size,1,infile) {
		int err=errno;
		cerr << "LoadImageThread::loadFile(): Could not load '" << file << "' (fread:" << strerror(err) << ")" << endl;
		if(fclose(infile))
			perror("Warning: LoadImageThread::loadFile(), on fclose");
		delete [] inbuf;
		return false;
	}
	if(fclose(infile))
		 perror("Warning: LoadImageThread::loadFile(), on fclose");
#else /*use mmap to load file into memory*/
	int infd=open(file.c_str(),O_RDONLY,0666);
	if(infd<0) {
		int err=errno;
		cerr << "LoadImageThread::loadFile(): Could not open '" << file << "' (open:" << strerror(err) << ")" << endl;
		return false;
	}
	JOCTET* inbuf=static_cast<JOCTET*>(mmap(NULL,statbuf.st_size,PROT_READ,MAP_FILE|MAP_PRIVATE,infd,0));
	if (inbuf == MAP_FAILED) { /* MAP_FAILED generally defined as ((void*)-1) */
		int err=errno;
		cerr << "LoadImageThread::loadFile(): Could not load '" << file << "' (mmap:" << strerror(err) << ")" << endl;
		if(close(infd)<0)
			perror("Warning: LoadImageThread::loadFile(), on closing temporary file descriptor from open");
		return false;
	}
	if(close(infd)<0)
		perror("Warning: LoadImageThread::loadFile(), on closing temporary file descriptor from open");
#endif
	if(!png_sig_cmp(inbuf, 0, 8)) {
		//is PNG
		png_structp png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, (png_voidp)NULL, NULL, NULL);
		if (!png_ptr) {
#ifdef LOADFILE_NO_MMAP
			delete [] inbuf;
#else
			if(munmap(inbuf,statbuf.st_size))
				perror("Warning: LoadImageThread::loadFile(), on munmap");
#endif
			return false;
		}
		
		png_infop info_ptr = png_create_info_struct(png_ptr);
		if (!info_ptr) {
			png_destroy_read_struct(&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
#ifdef LOADFILE_NO_MMAP
			delete [] inbuf;
#else
			if(munmap(inbuf,statbuf.st_size))
				perror("Warning: LoadImageThread::loadFile(), on munmap");
#endif
			return false;
		}
		
		if (setjmp(png_jmpbuf(png_ptr))) {
			png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
#ifdef LOADFILE_NO_MMAP
			delete [] inbuf;
#else
			if(munmap(inbuf,statbuf.st_size))
				perror("Warning: LoadImageThread::loadFile(), on munmap");
#endif
			return false;
		}
		
		png_read_mem_status read_status;
		read_status.buf=(png_bytep)inbuf;
		read_status.bufsize=statbuf.st_size;
		read_status.offset=0;
		png_set_read_fn(png_ptr, &read_status, user_read_png_data);
		
		png_read_info(png_ptr, info_ptr);
		
		size_t width=png_get_image_width(png_ptr, info_ptr);
		size_t height=png_get_image_height(png_ptr, info_ptr);
		if(width==0 || height==0) {
			cerr << "Empty image '" << file << "'" << endl;
			return false;
		}
		
		size_t bit_depth=png_get_bit_depth(png_ptr, info_ptr);
		if(bit_depth == 16)
			png_set_strip_16(png_ptr);
		if (bit_depth < 8)
			png_set_packing(png_ptr);

		size_t color_type=png_get_color_type(png_ptr, info_ptr);
		if (color_type & PNG_COLOR_MASK_ALPHA)
			png_set_strip_alpha(png_ptr);
		if (color_type == PNG_COLOR_TYPE_GRAY || color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
			png_set_gray_to_rgb(png_ptr);
		
		png_color_16 my_background;
		my_background.index=0;
		my_background.red=1<<15;
		my_background.green=1<<15;
		my_background.blue=1<<15;
		my_background.gray=1<<15;
		png_color_16p image_background=NULL;
		if (png_get_bKGD(png_ptr, info_ptr, &image_background))
			png_set_background(png_ptr, image_background, PNG_BACKGROUND_GAMMA_FILE, 1, 1.0);
		else
			png_set_background(png_ptr, &my_background, PNG_BACKGROUND_GAMMA_SCREEN, 0, 1.0);
		
		png_read_update_info(png_ptr, info_ptr);
		
		size_t rowbytes=png_get_rowbytes(png_ptr, info_ptr);
		/*png_bytep row_pointers[height];
		row_pointers[0]=reinterpret_cast<png_bytep>(data->Base()+FULL_HEADER_SIZE);
		for(unsigned int h=1; h<height; ++h)
			row_pointers[h]=row_pointers[h-1]+rowbytes;
		png_read_image(png_ptr, row_pointers);*/
		png_bytep row=reinterpret_cast<png_bytep>(setupRegion(data,file,width,height,3));
		for(unsigned int h=0; h<height; ++h) {
			if(row+rowbytes>reinterpret_cast<png_bytep>(data->Base()+data->Size())) {
				cerr << "Ran out of PNG buffer space" << endl;
				break;
			}
			png_read_row(png_ptr, row, NULL);
			row+=rowbytes;
		}
		png_read_end(png_ptr, NULL);
		png_destroy_read_struct(&png_ptr, &info_ptr,(png_infopp)NULL);
		
	} else {
		//assume JPEG
		jpeg_mem_src(cinfo, inbuf, statbuf.st_size);
		jpeg_read_header(cinfo, true);
		cinfo->out_color_space=JCS_YCbCr;
		jpeg_calc_output_dimensions(cinfo);
		if(cinfo->output_width==0 || cinfo->output_height==0) {
			cerr << "Empty image '" << file << "'" << endl;
			return false;
		}
		unsigned int remain=cinfo->output_height;
		unsigned int row_stride = cinfo->output_width * cinfo->output_components;
		JSAMPROW rows[remain];
		rows[0]=reinterpret_cast<JSAMPLE*>(setupRegion(data,file,cinfo->output_width,cinfo->output_height,cinfo->output_components));
		if(rows[0]==NULL) {
			jpeg_finish_decompress(cinfo);
			return false;
		}
		for(unsigned int i=1; i<remain; i++)
			rows[i]=rows[i-1]+row_stride;
		JSAMPROW* curpos=rows;
		//JSAMPARRAY buffer = (*cinfo.mem->alloc_sarray)((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);
		jpeg_start_decompress(cinfo);
		while (remain>0) {
			unsigned int used=jpeg_read_scanlines(cinfo, curpos, remain);
			curpos+=used;
			remain-=used;
		}
		jpeg_finish_decompress(cinfo);
	}
#ifdef LOADFILE_NO_MMAP
	delete [] inbuf;
#else
	if(munmap(inbuf,statbuf.st_size))
		perror("Warning: LoadImageThread::loadFile(), on munmap");
#endif
	if(jerr->num_warnings>0) {
		cerr << "Warning: Decompression of '" << file << "' had warnings" << endl;
		jerr->num_warnings=0;
	}
	return true;
}

char* LoadImageThread::setupRegion(RCRegion*& region, const std::string& filename, unsigned int width, unsigned int height, unsigned int components) {
	const unsigned int IMAGE_HEADER_SIZE=sizeof(unsigned int)*5;
	const unsigned int payload=IMAGE_HEADER_SIZE+sizeof(JSAMPLE)*width*height*components;
	char * buf=LoadFileThread::setupRegion(region,filename,payload);
	if(buf==NULL || region==NULL)
		return NULL;
	unsigned int remain=IMAGE_HEADER_SIZE;
	if(!LoadSave::encodeInc(width,buf,remain)) return NULL;
	if(!LoadSave::encodeInc(height,buf,remain)) return NULL;
	if(!LoadSave::encodeInc(components,buf,remain)) return NULL;
	if(!LoadSave::encodeInc(frameIndex++,buf,remain)) return NULL;
	if(!LoadSave::encodeInc(*layer,buf,remain)) return NULL;
	ASSERTRETVAL(remain==0,"LoadImageThread::setupRegion(): Leftover bytes in header? IMAGE_HEADER_SIZE is wrong\n",NULL);
	return buf;
}



/*! @file
 * @brief 
 * @author Ethan Tira-Thompson (ejt) (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.1 $
 * $State: Exp $
 * $Date: 2006/09/28 20:42:51 $
 */
