//-*-c++-*-
#ifndef INCLUDED_LoadFileThread_h_
#define INCLUDED_LoadFileThread_h_

#include "IPC/Thread.h"
#include "Shared/plist.h"
#include "IPC/MessageQueue.h"
#include <list>

//! Provides resources for loading time-based data from disk
/*! Runs in a separate thread, preloads data into shared memory buffers, and
 *  then sends the buffer when the time is right.  Thread doesn't start until
 *  you set the source or call loadFileList(). */
class LoadFileThread : public Thread, public plist::Dictionary, public plist::PrimitiveListener {
public:
	
	//! constructor
	/*! @param source the path to either a directory to load files from, or a single specific data file
	 *  @param filter a regular expression (POSIX.2 extended format) to select which files to load from @a source, if @a is a directory
	 *  @param fps frames per second, see #frameRate
	 *  @param messages the MessageQueue through which to send the data
	 *  @param doLoad if true, call loadFileList (subclasses probably want to pass false and call that themselves after their own construction)
	 *
	 *  The file list is not actually loaded until you call loadFileList -- this
	 *  gives you a chance to reset the default values you supplied in the
	 *  constructor with actual values from, for example, a preference file, and
	 *  then load the file list after the settings are correct.
	 */
	LoadFileThread(std::string source, std::string filter, float fps, MessageQueueBase& messages, bool doLoad=true)
	: Thread(), plist::Dictionary(),
		src(source),
		filenameFilter(filter),
		framerate(fps),
		verbose(0),
		loop(true),
		heartbeat(true), frozen(false),
		sent(), loaded(), timestamps(), files(), curfile(files.begin()), msgr(messages),
		startTimeOffset(0), timeOffset(0), lock(), loopRemainder(0), isIndexed(false), indexLoopTime(0), lastSent(0), frameSN(0), frozenTime(0)
	{
		addEntry("Source",src,"The directory from which data samples should be loaded, or a single specific file.\nA single file can be either a single data file (e.g. sensor or camera image), or an index file as output by VisionGUI, or in the format 'filename <tab> time', where 'filename' is an absolute path or relative to the directory containing the index file, and 'time' is in milliseconds, relative to the time at which the index file is loaded.\nIn the future, this could also be network addresses for teleoperation and remote processing.");
		addEntry("FileFilter",filenameFilter,"If Source is a directory or index file, only files matching the filter will be loaded from it.");
		addEntry("Framerate",framerate,"The rate at which images should be loaded -- only referenced when setting up files during loadFileList() -- later modifications won't reset the already loaded file list");
		addEntry("Verbose",verbose,"Controls how much feedback to give on the console regarding progress\n  0 - none\n  1 - report when message is sent\n  2 - also report when message is dropped\n  3 - also report when heartbeat is sent/dropped, and when loop occurs\n  4 - also report when each message is preloaded");
		addEntry("Loop",loop,"If true, restart file list at the beginning when the end is reached; otherwise just stop loading data");
		addEntry("Heartbeat",heartbeat,"If enabled, an empty \"heartbeat\" message is sent at the appropriate framerate, even if no data is being processed (i.e. frozen, no data loaded, or out of frames); this will cause an update event within the simulator, repeating processing on the previous data.");
		addEntry("Frozen",frozen,"If true, no frames will be sent, except via explicit 'advance' commands; if false, the thread will run and send messages at the requested times automatically");
		src.addPrimitiveListener(this);
		verbose.addPrimitiveListener(this);
		frozen.addPrimitiveListener(this);
		if(doLoad)
			loadFileList(false,false);
	}
	//! destructor
	~LoadFileThread();
	
	//! The directory from which data samples should be loaded, or a single specific file.
	/*! A single file can be either a single data file (e.g. sensor or camera image), or an index file as output by VisionGUI, or in the format 'filename <tab> time', where 'filename' is an absolute path or relative to the directory containing the index file, and 'time' is in milliseconds, relative to the time at which the index file is loaded.\nIn the future, this could also be network addresses for teleoperation and remote processing. */
	plist::Primitive<std::string> src;

	//! a regular expression (POSIX.2 extended format) to select which files to load from #src, if #src is a directory or index file
	plist::Primitive<std::string> filenameFilter;

	//! frames per second to send -- only referenced when setting up #files during loadFileList() -- later modifications won't reset the already loaded file list
	plist::Primitive<float> framerate;

	//! Controls how much feedback to give on the console regarding progress
	/*! 0 - none\n
	 *  1 - report when message is sent\n
	 *  2 - also report when message is dropped\n
	 *  3 - also report when heartbeat is sent/dropped, and when loop occurs\n
	 *  4 - also report when each message is preloaded */
	plist::Primitive<int> verbose; 

	//! controls whether to restart #curfile at the beginning of #files when it reaches the end
	plist::Primitive<bool> loop;
	
	//! if enabled, an empty "heartbeat" message is sent at the appropriate framerate, even if no data is being processed (i.e. no data loaded or out of frames); this will cause an update event within the simulator, repeating processing on the previous data.
	plist::Primitive<bool> heartbeat;
	
	//! if true, no frames will be sent, except via explicit (external) calls to advanceFrame(); if false, the thread will run and send messages at the requested times
	plist::Primitive<bool> frozen;
	

	//! call this to (re)load the list of available file names from disk
	/*! If @a clearCurrent is set, then the current file list will be cleared;
	 *  otherwise, the loaded files will be appended to the current queue */
	virtual void loadFileList(bool clearCurrent=true, bool reportMissing=true);
	//! sets the next frame to be sent (e.g. pass 0 to reset to the first frame)
	virtual void setFrame(unsigned int f);
	
	//! returns the timestamp of the next message in the queue ready to be sent
	virtual unsigned int nextTimestamp();
	
	//! returns the filename of the next frame scheduled to be sent
	virtual std::string getNextFrame();
	
	//! returns the serial number of the message
	static unsigned int recoverSN(RCRegion* msg);
	
	//! can reassign the serial number of the message
	static void resetSN(RCRegion* msg, unsigned int sn);
	
	//! recovers data from serialized IPC message, returns beginning of payload, or NULL if there's an error
	/*! each parameter can be a pointer where to store the field, or NULL if you don't care */
	static char* deserializeHeader(char* buf, unsigned int size, bool* verbose, unsigned int* sn, std::string* filename, bool* dataInQueue, unsigned int* payloadSize);
	
	//! sends the next frame in #loaded, and loads the next in filelist queue
	/*! returns true if there was a frame to send; will send heartbeat if configured, but returns false for heartbeats */
	virtual bool advanceFrame(bool forceQueue);

	virtual void plistValueChanged(const plist::PrimitiveBase& pl);
	
	//! reports number of files currently pre-loaded, available for sending
	virtual unsigned int numLoaded() { return loaded.size(); }
	
	virtual bool usingIndexFile() const { return isIndexed; }
	virtual float getLoopTime() const { return calcLoopTime(); }
	virtual void setLoopTime(float t);
	
protected:
	//! removes our reference to a region created by loadFile()
	virtual void freeRegion(RCRegion* rcr) { if(rcr!=NULL) rcr->RemoveReference(); }
	
	//! loads the next data element from file and provides its timestamp, skipping any files which have already expired
	/*! @param[in] data a pre-allocated region to use, or NULL to request creation of a new one
	 *  @param[out] data the address of the region with loaded data
	 *  @param[out] t the timestamp at which @a data should be consumed */
	virtual void getNextData(RCRegion*& data, unsigned int& t);
	
	//! does the actual work of loading and processing data from disk, subclasses should override this if they want to decompress/pre-parse the data
	/*! This implementation merely loads the data directly into memory with no processing or parsing.
	 *  @param[in] file full path of file to load
	 *  @param[in] data a pre-allocated region to use, or NULL to request creation of a new one
	 *  @param[out] data the address of the region which has been filled in
	 *  @return true if successful, false if error */
	virtual bool loadFile(const std::string& file, RCRegion*& data);
	using plist::Dictionary::loadFile; // there's also the LoadSave function by the same name for loading parameters from a file
	
	//! increments #curfile, looping if necessary; returns false if end of list reached and not #loop
	/*! @param[in] loopTime is the time of a single loop (the time of the last frame, plus inter-frame time)
	 *  @param[in] curTime is the current time, relative to beginning of current loop (#timeOffset)
	 *  @param[out] modified curTime if loop has occurred, subtracting loopTime 
	 *  @return true if successful, false if no files are left */
	virtual bool incrementCurfile(float loopTime, int& curTime);
	
	//! removes and returns first region in #sent with only one reference, or NULL if none exists
	virtual RCRegion* firstUnusedRegion();
	
	//! sets up some header info in the specified RCRegion, reallocating if the suggested region is not large enough (or is NULL), returns pointer to end of header info in the region
	virtual char* setupRegion(RCRegion*& region, const std::string& file, unsigned int payload);
	
	//! sends an empty heartbeat message indicating previous data should be reused
	virtual void sendHeartbeat();
	
	//! load a list of files from a directory specified by #src
	virtual void loadFileListFromDirectory();
	//! load a single file
	virtual void loadSingleFile(const std::string& file);
	//! load a list of files from an index file specified by #src
	/*! This supports either the format produced by VisionGUI, or a simplier '<code>filename [<tab> time]\n</code>' format,
	 *  where if @c time is unspecified, the frame's time is incremented by the #framerate from the previously listed file.
	 *  Filenames should either be either absolute paths or relative to the directory which contains the index file.*/
	virtual bool loadFileListFromIndex();

	//! returns time to sleep based front of timestamps -- limits to max sleep of 1 second or the current frame rate, if timestamp is already past, return 0
	unsigned int calcSleepTime() const;
	
	//! returns time of next heartbeat
	unsigned int calcNextHeartbeat(unsigned int curt) const;
	
	//! monitor #msgr, send new messages when their timestamp indicates they are due, then load upcoming messages
	virtual unsigned int runloop();
	
	//! if loaded has less than NUM_PRELOAD items, loads a new frame at the end
	virtual void updateLoaded(unsigned int curt);

	//! assign a new value @a t to #timeOffset -- clears loaded data queue (to be reloaded on next call to getNextData())
	virtual void resetTimeOffset(int t);
	
	//! computes the time of an entire loop through the files (including inter-frame time at the end of the loop)
	virtual float calcLoopTime() const;
	
	//! calls offsetTimestamps with the difference between @a t (usually the current time) and frozenTime , then resets frozenTime to @a t
	/*! takes a parameter instead of calling get_time itself because caller probably wants to do an "instantaneous" update, so don't want clock to tick while processing */
	void updateFrozenTime(unsigned int t);
	
	//! adds off to entries in #timestamps as well as #timeOffset and #startTimeOffset
	void offsetTimestamps(int off);

	static const unsigned int NUM_PRELOAD=2; //!< number of data elements to preload
	typedef std::list<RCRegion* > msgbuf_t; //!< type of collection of shared data regions
	typedef std::map<unsigned int,std::string> files_t; //!< type of #files, the list of files to load

	msgbuf_t sent; //!< for efficiency, reuse old buffers -- oldest at front, most recently used at back
	msgbuf_t loaded; //!< for efficiency, reuse old buffers -- oldest at front, most recently used at back
	std::list<unsigned int> timestamps; //!< the timestamps to send data queued in #loaded
	files_t files; //!< the list of files to load ('second'), and their timestamps ('first')
	files_t::const_iterator curfile; //!< an iterator referencing #files -- indicates next file to load
	MessageQueueBase& msgr; //!< the MessageQueue through which to send the data
	int startTimeOffset; //!< the requested time of start of run through #files -- if this is modified, it will be noticed during runloop() and #curfile will be updated appropriately
	int timeOffset; //!< the starting time of the run through #files, see #startTimeOffset
	Thread::Lock lock; //!< allows mutual exclusion over this object's member variables
	float loopRemainder; //!< a bit ugly -- this is the fractional leftover accumulated from looping so we won't "drift" over time
	bool isIndexed; //!< if we're using an index file, we have to handle the loopRemainder differently
	float indexLoopTime; //!< if we're using an index file, this can be set to the time per loop, which can be used to keep multiple streams in sync.  Must be set externally however, as this stream doesn't know about other streams.
	unsigned int lastSent; //!< timestamp of most recently sent message (or heartbeat)
	unsigned int frameSN; //!< serial number of next message to load (a lower number may be enqueued but not yet sent!)
	unsigned int frozenTime; //!< the time #frozen was set to true
};

/*! @file
 * @brief 
 * @author Ethan Tira-Thompson (ejt) (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.1 $
 * $State: Exp $
 * $Date: 2006/09/28 20:42:51 $
 */

#endif
