#include "LoadFileThread.h"
#include "Shared/get_time.h"
#include "Shared/debuget.h"
#include <sys/stat.h>
#include <regex.h>
#include <dirent.h>
#include <set>

using namespace std;

LoadFileThread::~LoadFileThread() {
	MarkScope l(lock);
	src.removePrimitiveListener(this);
	verbose.removePrimitiveListener(this);
	while(sent.size()>0) {
		freeRegion(sent.front());
		sent.pop_front();
	}
	while(loaded.size()>0) {
		freeRegion(loaded.front());
		loaded.pop_front();
	}
	timestamps.clear();
}

void LoadFileThread::loadFileList(bool clearCurrent/*=true*/, bool reportMissing/*=true*/) {
	MarkScope l(lock);
	struct stat sb;
	if(clearCurrent) {
		files.clear();
		curfile=files.begin(); // aka files.end()
		if(loaded.size()>0 && loaded.front()!=NULL)
			frameSN=recoverSN(loaded.front());
		while(loaded.size()>0) {
			if(loaded.back()!=NULL)
				sent.push_front(loaded.back());
			loaded.pop_back();
			timestamps.pop_back();
		}
	}
	isIndexed=false;
	if(stat(src.c_str(),&sb)) {
		if(reportMissing)
			std::cerr << "Could not access source " << src << std::endl;
		return;
	}
	if(sb.st_mode&S_IFDIR) {
		loadFileListFromDirectory();
	} else {
		//Test to see if the file matches the filter
		regex_t re;
		if(int err=regcomp(&re,filenameFilter.c_str(),REG_EXTENDED | REG_NOSUB)) {
			char msg[128];
			regerror(err,&re,msg,128);
			std::cerr << "Bad filter '" << filenameFilter << "': " << msg << std::endl;
			regfree(&re);
			return;
		}
		int match=regexec(&re,src.c_str(),0,NULL,0);
		regfree(&re);
		if(match==0) {
			loadSingleFile(src.c_str());
		} else if(match==REG_NOMATCH) {
			//if it doesn't match the image RE, assume it's an index file
			if(!loadFileListFromIndex())
				std::cerr << "Source '" << src << "' does not match the filename filter '" << filenameFilter << "' and is not an index list." << std::endl;
			else
				isIndexed=true;
		} else {
			char msg[128];
			regerror(match,&re,msg,128);
			std::cerr << "Regex error on '" << src << "': " << msg << std::endl;
		}
	}
	if(clearCurrent)
		setFrame(0);
	else
		curfile=files.begin(); //curfile may have been invalidated, getNextData will fast forward to current time
	for(unsigned int i=0; i<NUM_PRELOAD && curfile!=files.end(); ++i)
		updateLoaded(get_time());
}

void LoadFileThread::setFrame(unsigned int f) {
	MarkScope l(lock);
	loopRemainder=0;
	startTimeOffset=(int)(get_time()-f*1000/framerate);
	resetTimeOffset(startTimeOffset);
	curfile=files.begin();
	advance(curfile,f);
}

void LoadFileThread::loadSingleFile(const std::string& file) {
	MarkScope l(lock);
	files[(unsigned int)calcLoopTime()]=file;
}

void LoadFileThread::loadFileListFromDirectory() {
	MarkScope l(lock);
	regex_t re;
	if(int err=regcomp(&re,filenameFilter.c_str(),REG_EXTENDED | REG_NOSUB)) {
		char msg[128];
		regerror(err,&re,msg,128);
		std::cerr << "Bad filter '" << filenameFilter << "': " << msg << std::endl;
		regfree(&re);
		return;
	}
	DIR * d=opendir(src.c_str());
	if(d==NULL) {
		std::cerr << "Could not open directory " << src << std::endl;
		regfree(&re);
		return;
	}
	struct dirent* res;
	
#ifdef HAVE_READDIR_R
	struct dirent cur;
	if(readdir_r(d,&cur,&res)) {
		std::cerr << "Error reading files from " << src << std::endl;
		closedir(d);
		regfree(&re);
		return;
	}
#else
	res=readdir(d);
#endif

	std::set<std::string> dirfiles;
	while(res!=NULL) {
		int match=regexec(&re,res->d_name,0,NULL,0);
		if(match==0) {
			dirfiles.insert(res->d_name);
		} else if(match!=REG_NOMATCH) {
			char msg[128];
			regerror(match,&re,msg,128);
			std::cerr << "Regex error on '" << res->d_name << "': " << msg << std::endl;
		} // else std::cout << "Skipping " << res->d_name << std::endl;
#ifdef HAVE_READDIR_R
		if(readdir_r(d,&cur,&res)) {
			std::cerr << "Error reading files from " << src << std::endl;
			closedir(d);
			regfree(&re);
			return;
		}
#else
		res=readdir(d);
#endif
	}
	closedir(d);
	regfree(&re);
	
	float tinc=1000.f/framerate;
	float time=calcLoopTime();
	for(std::set<std::string>::const_iterator it=dirfiles.begin(); it!=dirfiles.end(); ++it) {
		//std::cout << "Enqueuing " << *it << std::endl;
		files[static_cast<unsigned int>(time)]=(src+"/")+(*it);
		time+=tinc;
	}
}

bool LoadFileThread::loadFileListFromIndex() {
	regex_t re;
	if(int err=regcomp(&re,filenameFilter.c_str(),REG_EXTENDED | REG_NOSUB)) {
		char msg[128];
		regerror(err,&re,msg,128);
		std::cerr << "Bad filter '" << filenameFilter << "': " << msg << std::endl;
		regfree(&re);
		return false;
	}
	
	indexLoopTime=0;
	ifstream in(src.c_str());
	string cur;
	getline(in,cur);
	if(cur.find("First frame ")==0) //skip the header line from the GUI, e.g. 'First frame 42898 timestamp: 1439018'
		getline(in,cur);
	
	float tinc=1000.f/framerate;
	float time=calcLoopTime();
	while(in) {
		string fn = cur.substr(0,cur.find('\t'));
		if(fn.size()!=cur.size()) {
			const char * timep=cur.c_str()+cur.rfind('\t');
			char * endp=NULL;
			time=strtof(timep,&endp);
			if(timep==endp) {
				std::cerr << "ERROR: '" << src << "' does not seem to be a valid index file." << std::endl;
				std::cerr << "       Use output from VisionGUI, or use format 'filename <tab> time'" << std::endl;
				std::cerr << "       Where 'time' is the time in milliseconds at which the file should be processed, relative" << std::endl;
				std::cerr << "       to the time at which the index file is loaded." << std::endl;
				regfree(&re);
				return false;
			}
		}
		int match=regexec(&re,fn.c_str(),0,NULL,0);
		if(match==0) {
			if(fn[0]!='/') {
				unsigned int srcdir=src.rfind('/');
				if(srcdir!=string::npos)
					fn=src.substr(0,srcdir+1)+fn;
			}
			//std::cout << "Enqueuing " << fn << " at " << time << std::endl;
			files[static_cast<unsigned int>(time)]=fn;
			time+=tinc;
		} else if(match!=REG_NOMATCH) {
			char msg[128];
			regerror(match,&re,msg,128);
			std::cerr << "Regex error on '" << fn << "': " << msg << std::endl;
		} // else std::cout << "Skipping " << res->d_name << std::endl;
		getline(in,cur);
	}
	regfree(&re);
	return true;
}

bool LoadFileThread::incrementCurfile(float loopTime, int& curTime) {
	if(++curfile == files.end()) {
		if(!loop) {
			if(verbose>0)
				std::cout << "Last data from '" << src << "' loaded -- set 'loop' to true, or use 'restart' command to manually loop" << std::endl;
			return false;
		} else {
			if(verbose>2)
				std::cout << "Preparing to looping data from '" << src << "'" << std::endl;
			unsigned int loopTimeI = (unsigned int)(loopTime);
			loopRemainder+=loopTime-loopTimeI;
			unsigned int loopRemainderI=(unsigned int)(loopRemainder);
			loopRemainder-=loopRemainderI;
			timeOffset=(startTimeOffset+=(loopTimeI+=loopRemainderI));
			curTime-=loopTimeI;
			curfile=files.begin();
		}
	}
	return true;
}

unsigned int LoadFileThread::nextTimestamp() {
	unsigned int curt=get_time();
	if(frozen)
		updateFrozenTime(curt);
	if(!frozen && timestamps.size()>0) {
		return timestamps.front();
	} else if(heartbeat) {
		return calcNextHeartbeat(curt);
	} else
		return -1U;
}


std::string LoadFileThread::getNextFrame() {
	if(loaded.size()==0 || loaded.front()==NULL)
		return heartbeat ? "heartbeat" : "(none)";
	return std::string(loaded.front()->Base()+LoadSave::getSerializedSize<bool>()+2*LoadSave::getSerializedSize<int>());
}

unsigned int LoadFileThread::recoverSN(RCRegion* msg) {
	unsigned int ans;
	LoadSave::decode(ans,msg->Base()+LoadSave::getSerializedSize<bool>(),LoadSave::getSerializedSize(ans));
	return ans;
}

void LoadFileThread::resetSN(RCRegion* msg, unsigned int sn) {
	unsigned int used=LoadSave::encode(sn,msg->Base()+LoadSave::getSerializedSize<bool>(),LoadSave::getSerializedSize(sn));
	ASSERT(used==sizeof(sn),"bad resetSN " << used)
}

bool LoadFileThread::advanceFrame(bool forceQueue) {
	MarkScope l(lock);
	unsigned int curt=get_time();
	if(frozen)
		updateFrozenTime(curt);
	updateLoaded(curt);
	if(loaded.size()==0 || (frozen && !forceQueue)) {
		if(heartbeat) {
			sendHeartbeat();
			lastSent=curt;
		}
		return false;
	}
	lastSent=curt;
	if(forceQueue)
		offsetTimestamps(curt-timestamps.front());
	timestamps.pop_front();
	if(loaded.front()==NULL) {
		if(heartbeat)
			sendHeartbeat();
		if(loaded.size()!=0)
			loaded.pop_front();
		return false;
	} else { //has data
		if(verbose>0)
			std::cout << "Sending frame: \"" << getNextFrame() << "\" (advanced at " << curt << ')' << std::endl;
		sent.push_back(loaded.front());
		msgr.sendMessage(loaded.front());
		loaded.pop_front();
		updateLoaded(curt);
		return true;
	}
}

void LoadFileThread::plistValueChanged(const plist::PrimitiveBase& pl) {
	MarkScope l(lock);
	if(&pl==&src)
		loadFileList();
	else if(&pl==&verbose) {
		msgr.setReportDroppings(verbose>1);
		// update verbose flag of currently loaded
		for(msgbuf_t::iterator it=loaded.begin(); it!=loaded.end(); ++it)
			LoadSave::encode(verbose>0,(*it)->Base(),LoadSave::getSerializedSize<bool>());
	} else if(&pl==&frozen) {
		if(!frozen) {
			updateFrozenTime(get_time());
			if(!isRunning())
				start();
		} else if(frozen) {
			if(!heartbeat && isRunning())
				stop();
			frozenTime=get_time();
		}
	} else if(&pl==&heartbeat) {
		if(frozen && heartbeat && !isRunning())
			start();
		else if(frozen && !heartbeat && isRunning())
			stop();
	}
}

void LoadFileThread::setLoopTime(float t) {
	files_t::const_iterator lastfile=files.end();
	--lastfile;
	unsigned int last=lastfile->first;
	if(t<last && t!=0) {
		stringstream ss;
		ss << "LoadFileThread::setLoopTime error: passed loop time (" << t << ") is less than time of last frame (" << last << ")";
		throw std::invalid_argument(ss.str());
	}
	indexLoopTime=t;
}


void LoadFileThread::getNextData(RCRegion*& data, unsigned int& t) {
	if(files.size()==0 || curfile==files.end())
		return;
	if(startTimeOffset!=timeOffset)
		resetTimeOffset(startTimeOffset);

	unsigned int realt=get_time();
	int curt=realt-timeOffset;
	float loopTime=calcLoopTime();
		
	if(!frozen) {
		//skip old data -- current time is already past these frames, no point in loading them
		if(curt>(int)(loopTime)) {
			if(!loop) {
				if(verbose>2)
					std::cout << "Way out of data from '" << src << "' -- set 'loop' to true, or use 'restart' command to manually loop" << std::endl;
				curfile=files.end();
				return;
			}
			if(verbose>2)
				std::cout << "Mass looping data from '" << src << "'" << std::endl;
			unsigned int loops=(unsigned int)(curt/loopTime);
			float massLoopTime=loops*loopTime;
			unsigned int massLoopTimeI=(unsigned int)(massLoopTime);
			loopRemainder+=massLoopTime-massLoopTimeI;
			unsigned int loopRemainderI=(unsigned int)(loopRemainder);
			loopRemainder-=loopRemainderI;
			resetTimeOffset(startTimeOffset+=(massLoopTimeI+=loopRemainderI));
			curt=realt-timeOffset; //timeOffset set to startTimeOffset by previous line
			curfile=files.begin();
		}
		while(static_cast<int>(curfile->first)<curt)
			if(!incrementCurfile(loopTime,curt))
				return;
	}
	if(timestamps.size()>0) // if there's already stuff loaded, make sure we're loading after what's already there
		while(curfile->first+timeOffset<=timestamps.back())
			if(!incrementCurfile(loopTime,curt))
				return;
	
	if(verbose>3)
		std::cout << "Loading frame from " << curfile->second << " dt="<<curfile->first << " scheduled:" << (curfile->first+timeOffset) << std::endl;
	RCRegion* unused=firstUnusedRegion();
	if(!loadFile(curfile->second, unused)) { //loadfile can accept NULL region if none was found
		std::cerr << "Bad load on " << curfile->second << std::endl;
		incrementCurfile(loopTime,curt);
		if(unused!=NULL)
			sent.push_front(unused);
		return;
	}
	data=unused;
	t=curfile->first+timeOffset;
	incrementCurfile(loopTime,curt);
}

bool LoadFileThread::loadFile(const std::string& file, RCRegion*& data) {
	struct stat statbuf;
	if(stat(file.c_str(),&statbuf)!=0) {
		perror("LoadFileThread::loadFile");
		return false;
	}
	FILE * f=fopen(file.c_str(),"rb");
	int nread;
	try {
		if(f==NULL) {
			std::cerr << "LoadFileThread::loadFile(): File open failed" << std::endl;
			return false;
		}
		char* buf=setupRegion(data,file,statbuf.st_size);
		nread=fread(buf,1,statbuf.st_size,f);
	} catch (...) {
		fclose(f);
		std::cerr << "Exception occurred during LoadFileThread::loadFile" << std::endl;
		throw;
	}
	fclose(f);
	f=NULL;
	if(nread!=statbuf.st_size) {
		std::cerr << "LoadFileThread::loadFile(): failed to load entire file, "<<nread<<" read, "<<statbuf.st_size<<" requested" << std::endl;
		return false;
	}
	return true;
}

RCRegion* LoadFileThread::firstUnusedRegion() {
	for(msgbuf_t::iterator it=sent.begin();it!=sent.end(); ++it) {
		if((*it)->NumberOfReference()==1) {
			RCRegion * ans=*it;
			sent.erase(it);
			return ans;
		}
	}
	return NULL;
}

char* LoadFileThread::setupRegion(RCRegion*& region, const std::string& filename, unsigned int payload) {
	// header fields: bool:verbose, uint:frameSN, str:filename, uint:payload
	const unsigned int FILE_HEADER_SIZE=LoadSave::getSerializedSize<bool>() + LoadSave::getSerializedSize(frameSN) + LoadSave::getSerializedSize(filename) + LoadSave::getSerializedSize<bool>() + LoadSave::getSerializedSize(payload);
	if(region==NULL)
		region=new RCRegion(FILE_HEADER_SIZE+payload+sizeof(char)*filename.size()*2); //triple the allocation for filename so we don't have to resize if we get a longer name later
	else if(region->Size()<FILE_HEADER_SIZE+payload) {
		//too small -- free it, we'll drop it and make a bigger one
		freeRegion(region);
		region=new RCRegion(FILE_HEADER_SIZE+payload+sizeof(char)*filename.size()*2); //triple the allocation for filename so we don't have to resize if we get a longer name later
	}
	char* buf=region->Base();
	unsigned int remain=FILE_HEADER_SIZE;
	if(!LoadSave::encodeInc(payload>0 && verbose>0 || verbose>2,buf,remain)) return NULL;
	if(!LoadSave::encodeInc(frameSN++,buf,remain)) return NULL;
	if(!LoadSave::encodeInc(filename,buf,remain)) return NULL;
	if(!LoadSave::encodeInc(loaded.size()>0,buf,remain)) return NULL;
	if(!LoadSave::encodeInc(payload,buf,remain)) return NULL;
	ASSERT(remain==0,"LoadFileThread::setupRegion(): Leftover bytes in header? FILE_HEADER_SIZE is wrong\n");
	return buf;
}

char* LoadFileThread::deserializeHeader(char* buf, unsigned int size, bool* verbose, unsigned int* sn, std::string* filename, bool* dataInQueue, unsigned int* payloadSize) {
	if(verbose!=NULL) { if(!LoadSave::decodeInc(*verbose,buf,size)) return NULL; } else buf+=LoadSave::getSerializedSize(*verbose);
	if(sn!=NULL) { if(!LoadSave::decodeInc(*sn,buf,size)) return NULL; } else buf+=LoadSave::getSerializedSize(*sn);
	if(filename!=NULL) { if(!LoadSave::decodeInc(*filename,buf,size)) return NULL; } else {
		unsigned int len=0;
		if(!LoadSave::decodeInc(len,buf,size)) return NULL;
		buf+=len+1;
	}
	if(dataInQueue!=NULL) { if(!LoadSave::decodeInc(*dataInQueue,buf,size)) return NULL; } else buf+=LoadSave::getSerializedSize(*dataInQueue);
#ifndef DEBUG
	if(payloadSize!=NULL) { if(!LoadSave::decodeInc(*payloadSize,buf,size)) return NULL; } else buf+=LoadSave::getSerializedSize(*payloadSize);
#else
	unsigned int lPayloadSize; // want to error check payloadSize regardless of whether caller wants the info
	if(payloadSize==NULL)
		payloadSize=&lPayloadSize;
	if(!LoadSave::decodeInc(*payloadSize,buf,size)) return NULL;
	ASSERT(size>=*payloadSize,"short payload (" << size << " vs expected " << *payloadSize << ")");
	// this is normal (regions are generated a little bigger than they need to be to allow recycling)
	//ASSERT(size<=*payloadSize,"unhandled bytes after payload (" << size << " vs expected " << *payloadSize << ")");
#endif
	return buf;
}

void LoadFileThread::sendHeartbeat() {
	msgbuf_t::const_iterator it=loaded.begin();
	while(it!=loaded.end() && *it==NULL)
		++it;
	if(it!=loaded.end())
		frameSN=recoverSN(*it);
	RCRegion *unused=firstUnusedRegion();
	char * buf=setupRegion(unused,"heartbeat",0);
	if(unused==NULL || buf==NULL)
		return;
	if(verbose>2)
		std::cout << "Sending heartbeat at " << get_time() << std::endl;
	for(; it!=loaded.end(); ++it)
		if(*it!=NULL)
			resetSN(*it,frameSN++);
	msgr.sendMessage(unused);
	sent.push_back(unused);
}

unsigned int LoadFileThread::calcSleepTime() const {
	if(getTimeScale()<0)
		return 1000; // poll quickly when in full speed mode (100Hz)
	const unsigned int MAX_POLL=250000U; // longest to go between polls -- 4Hz
	if(getTimeScale()==0)
		return MAX_POLL; // poll (somewhat) slowly when paused (4Hz)
	//otherwise poll at framerate
	if(timestamps.size()==0) {
		// just to keep checking back in case of reset
		return min(MAX_POLL,static_cast<unsigned int>(1000000/getTimeScale()/framerate));
	} else {
		unsigned int curt=get_time();
		// return min of 1 second (in case of reset/user input), frame rate, or time until next frame
		return timestamps.front()<=curt ? 0 : min(MAX_POLL,static_cast<unsigned int>(min(1000000/getTimeScale()/framerate,1000*(timestamps.front()-curt)/getTimeScale())));
	}
}

unsigned int LoadFileThread::calcNextHeartbeat(unsigned int curt) const {
	unsigned int skip=static_cast<unsigned int>((curt-lastSent)*framerate/1000);
	return static_cast<unsigned int>(lastSent+(skip+1)*1000.f/framerate);
}

unsigned int LoadFileThread::runloop() {
	MarkScope l(lock);
	unsigned int curt=get_time();
	if(frozen) {
		if(heartbeat && curt>=(lastSent+1000.f/framerate)) {
			updateFrozenTime(curt);
			sendHeartbeat();
			lastSent=curt;
		}
	} else if(timestamps.size()==0 && heartbeat && curt>=(lastSent+1000.f/framerate)) {
		sendHeartbeat();
		lastSent=curt;
	} else if(timestamps.size()>0 && curt>=timestamps.front()) {
		if(verbose>0)
			std::cout << "Sending frame: \"" << getNextFrame() << "\" (scheduled " << timestamps.front() << ", now " << curt << ')' << std::endl;
		lastSent=timestamps.front();
		timestamps.pop_front();
		if(loaded.size()==0 || loaded.front()==NULL) {
			if(heartbeat)
				sendHeartbeat();
			if(loaded.size()!=0)
				loaded.pop_front();
		} else {
			sent.push_back(loaded.front());
			msgr.sendMessage(loaded.front());
			loaded.pop_front();
		}
	}
	updateLoaded(get_time());
	if(loaded.size()<NUM_PRELOAD) //try to catch up if we're behind
		updateLoaded(get_time()); // but only do one extra -- don't want to pause sending because we're busy loading
	return calcSleepTime();
}

void LoadFileThread::updateLoaded(unsigned int curt) {
	if(loaded.size()<NUM_PRELOAD) {
		if(curfile==files.end()) //out of data -- nothing to load, return
			return;
		RCRegion* rcr=NULL;
		unsigned int t=-1U;
		getNextData(rcr,t);
		if(t==-1U || rcr==NULL) {
			//bad image, skip a frame and then check again in case we get reset
			std::cerr << "Could not allocate new shared memory region or bad initial load" << std::endl;
			if(rcr!=NULL) //if rcr was created, error during load, don't throw away the region
				sent.push_front(rcr);
			if(heartbeat) {
				//send a heartbeat instead of missing frame if requested
				rcr=firstUnusedRegion();
				char * buf=setupRegion(rcr,"heartbeat",0);
				if(buf!=NULL && rcr!=NULL) {
					loaded.push_back(rcr);
					if(timestamps.size()>0)
						timestamps.push_back(static_cast<unsigned int>(timestamps.back()+1000.f/framerate));
					else {
						//skip heartbeats which have already passed
						timestamps.push_back(calcNextHeartbeat(curt));
					}
				}
			}
		} else if(timestamps.size()==0 || t>timestamps.back()) {
			loaded.push_back(rcr);
			timestamps.push_back(t);
		} else {
			frameSN--;
			std::cerr << "ERROR: LoadFileThread received old data from getNextData" << std::endl;
			sent.push_front(rcr);
		}
	}
}

void LoadFileThread::resetTimeOffset(int t) {
	if(loaded.size()>0 && loaded.front()!=NULL)
		frameSN=recoverSN(loaded.front());
	while(loaded.size()>0) {
		if(loaded.front()!=NULL)
			sent.push_front(loaded.front());
		loaded.pop_front();
	}
	timestamps.clear();
	if(t<timeOffset) {
		//need to rewind curfile
		curfile=files.begin(); //getNextData will fast forward to current time
	}
	timeOffset=t;
}

float LoadFileThread::calcLoopTime() const {
	if(files.size()==0)
		return 0;
	float tinc=1000.f/framerate;
	if(isIndexed) {
		if(indexLoopTime>0)
			return indexLoopTime;
		files_t::const_iterator lastfile=files.end();
		--lastfile;
		unsigned int last=lastfile->first;
		return last+tinc;
	} else {
		return files.size()*tinc;
	}
}

void LoadFileThread::updateFrozenTime(unsigned int t) {
	offsetTimestamps(t-frozenTime);
	frozenTime=t;
}

void LoadFileThread::offsetTimestamps(int off) {
	timeOffset=(startTimeOffset+=off);
	for(std::list<unsigned int>::iterator it=timestamps.begin(); it!=timestamps.end(); ++it)
		(*it)+=off;
}	

/*! @file
 * @brief 
 * @author Ethan Tira-Thompson (ejt) (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.1 $
 * $State: Exp $
 * $Date: 2006/09/28 20:42:51 $
 */
