#include "PNGGenerator.h"
#include "InterleavedYUVGenerator.h"
#include "Events/DataEvent.h"
#include "Events/EventRouter.h"
#include "Events/FilterBankEvent.h"
#include "Shared/Profiler.h"
#include "Wireless/Socket.h"

//better to put this here instead of the header
using namespace std; 

extern "C" {
	//! stores progress of user_write_png_data() between calls
	struct png_write_mem_status {
		png_bytep buf;  //!< beginning of buffer being writen into (doesn't move with progress)
		size_t bufsize; //!< total size of buffer
		size_t offset;  //!< position within buffer, i.e. amount of buffer written so far
	};
	//! user callback function for writing a png at @a data into user parameters stored in @a png_ptr
	void user_write_png_data(png_structp png_ptr, png_bytep data, png_size_t length) {
		png_write_mem_status* status=(png_write_mem_status*)(png_get_io_ptr(png_ptr));
		size_t endoff=status->offset+length;
		if(endoff<=status->bufsize) {
			memcpy(status->buf+status->offset,data,length);
			status->offset=endoff;
		} else {
			png_error(png_ptr, "Write Error - ran out of file");
		}
	}
	//! user callback function for flushing results of user_write_png_data() (this is a no-op)
	void user_flush_png_data(png_structp /*png_ptr*/) {}
}


PNGGenerator::PNGGenerator(unsigned int mysid, FilterBankGenerator* fbg, EventBase::EventTypeID_t tid)
: FilterBankGenerator("PNGGenerator","PNGGenerator",EventBase::visPNGEGID,mysid,fbg,tid), srcMode(SRC_AUTO), curMode(SRC_AUTO), bytesUsed(NULL)
{
	if(dynamic_cast<const InterleavedYUVGenerator*>(src)!=NULL)
		curMode=SRC_COLOR;
	else
		curMode=SRC_GRAYSCALE;
	
	//this part is only necessary if you override setNumImages yourself
	if(fbg!=NULL) {
		numLayers=numChannels=0; //this is to force setNumImages to override settings provided by FilterBankGenerator
		setNumImages(fbg->getNumLayers(),fbg->getNumChannels());
	}
}

PNGGenerator::PNGGenerator(unsigned int mysid, src_mode_t mode, FilterBankGenerator* fbg, EventBase::EventTypeID_t tid)
: FilterBankGenerator("PNGGenerator","PNGGenerator",EventBase::visPNGEGID,mysid,fbg,tid), srcMode(mode), curMode(mode), bytesUsed(NULL)
{
	if(srcMode==SRC_AUTO) {
		if(dynamic_cast<const InterleavedYUVGenerator*>(src)!=NULL)
			curMode=SRC_COLOR;
		else
			curMode=SRC_GRAYSCALE;
	}
	
	//this part is only necessary if you override setNumImages yourself
	if(fbg!=NULL) {
		numLayers=numChannels=0; //this is to force setNumImages to override settings provided by FilterBankGenerator
		setNumImages(fbg->getNumLayers(),fbg->getNumChannels());
	}
}

PNGGenerator::~PNGGenerator() {
	freeCaches();
	destruct();
}


/*! The const casts in this function are regretable but necessary
*  since the corresponding OPEN-R functions require mutable
*  arguments, even though they shouldn't be modifying the data
*/
void
PNGGenerator::processEvent(const EventBase& event) {
	FilterBankGenerator::processEvent(event);
	if(event.getGeneratorID()==getListenGeneratorID() && event.getSourceID()==getListenSourceID()) {
		if(getSourceMode()==SRC_AUTO) { //if not auto, curMode was already set when srcMode was set
			if(dynamic_cast<const InterleavedYUVGenerator*>(src)!=NULL)
				curMode=SRC_COLOR;
			else
				curMode=SRC_GRAYSCALE;
		}
		FilterBankEvent fbkev(this,getGeneratorID(),getSourceID(),EventBase::activateETID);
		erouter->postEvent(fbkev);
		fbkev.setTypeID(EventBase::statusETID);
		erouter->postEvent(fbkev);
		fbkev.setTypeID(EventBase::deactivateETID);
		erouter->postEvent(fbkev);
	}
}

unsigned int
PNGGenerator::getBinSize() const {
	unsigned int used=FilterBankGenerator::getBinSize();
	char * type;
	if(getCurrentSourceFormat()==SRC_COLOR)
		type="PNGColor";
	else if(getCurrentSourceFormat()==SRC_GRAYSCALE)
		type="PNGGrayscale";
	else {
		serr->printf("getBinSize failed - unsuitable or unknown mode/generator pair");
		return 0;
	}
	used+=strlen(type)+LoadSave::stringpad;
	if(bytesUsed[selectedSaveLayer][selectedSaveChannel]!=0)
		used+=bytesUsed[selectedSaveLayer][selectedSaveChannel];
	else
		used+=widths[selectedSaveLayer]*heights[selectedSaveLayer]*3+PNG_HEADER_PAD;
	return used;
}

unsigned int
PNGGenerator::loadBuffer(const char buf[], unsigned int len) {
	unsigned int origlen=len;
	if(!FilterBankGenerator::loadBuffer(buf,len)) return 0;
	std::string tmp;
	if(!decodeInc(tmp,buf,len)) return 0;
	if(tmp!="PNGColor" && tmp!="PNGGrayscale") {
		serr->printf("Unhandled image type for PNGGenerator: %s",tmp.c_str());
		return 0;
	} else {
		if(tmp=="PNGColor" && getCurrentSourceFormat()==SRC_GRAYSCALE)
			serr->printf("Warning: loading grayscale into color image");
		if(tmp=="PNGGrayscale" && getCurrentSourceFormat()==SRC_COLOR)
			serr->printf("Warning: loading color into grayscale image");
		unsigned int tmpL;
		if(!decodeInc(tmpL,buf,len)) return 0;
		if(tmpL>len)
			return 0;
		if(images[selectedSaveLayer][selectedSaveChannel]!=NULL)
			delete [] images[selectedSaveLayer][selectedSaveChannel];
		images[selectedSaveLayer][selectedSaveChannel]=createImageCache(selectedSaveLayer,selectedSaveChannel);
		unsigned int used=bytesUsed[selectedSaveLayer][selectedSaveChannel]=tmpL;
		unsigned char* img=getImage(selectedSaveLayer,selectedSaveChannel);
		if(img==NULL)
			return 0;
		memcpy(img,buf,used);
		len-=used; buf+=used;
		imageValids[selectedSaveLayer][selectedSaveChannel]=true;
		return origlen-len;	
	}
}

unsigned int
PNGGenerator::saveBuffer(char buf[], unsigned int len) const {
	unsigned int origlen=len;
	if(!checkInc(FilterBankGenerator::saveBuffer(buf,len),buf,len)) return 0;
	
	char * type;
	if(getCurrentSourceFormat()==SRC_COLOR)
		type="PNGColor";
	else if(getCurrentSourceFormat()==SRC_GRAYSCALE)
		type="PNGGrayscale";
	else {
		serr->printf("saveBuffer failed - unsuitable or unknown mode/generator pair");
		return 0;
	}
	if(!encodeInc(type,buf,len)) return 0;
	
	if(images[selectedSaveLayer][selectedSaveChannel]==NULL) {
		serr->printf("PNGGenerator::saveBuffer() failed because selected image is NULL -- call selectSaveImage first to make sure it's up to date\n");
		return 0;
	}
	if(!imageValids[selectedSaveLayer][selectedSaveChannel]) {
		serr->printf("PNGGenerator::saveBuffer() failed because selected image is invalid -- call selectSaveImage first to make sure it's up to date\n");
		return 0;
	}
	unsigned char* img=images[selectedSaveLayer][selectedSaveChannel];
	if(img==NULL)
		return 0;
	if(!encodeInc(bytesUsed[selectedSaveLayer][selectedSaveChannel],buf,len)) return 0;
	unsigned int used=bytesUsed[selectedSaveLayer][selectedSaveChannel];
	if(used>len)
		return 0;
	memcpy(buf,img,used);
	len-=used;
	return origlen-len;
}

void
PNGGenerator::setNumImages(unsigned int nLayers, unsigned int nChannels) {
	if(nLayers==numLayers && nChannels==numChannels)
		return;
	FilterBankGenerator::setNumImages(nLayers,nChannels);
	for(unsigned int i=0; i<numLayers; i++)
		strides[i]=skips[i]=0;
	bytesUsed=new unsigned int*[numLayers];
	for(unsigned int res=0; res<numLayers; res++) {
		increments[res]=3;
		bytesUsed[res]=new unsigned int[numChannels];
		for(unsigned int i=0; i<numChannels; i++)
			bytesUsed[res][i]=0;
	}
}

unsigned char *
PNGGenerator::createImageCache(unsigned int layer, unsigned int /*chan*/) const {
	return new unsigned char[widths[layer]*heights[layer]*3+PNG_HEADER_PAD];
}

void
PNGGenerator::calcImage(unsigned int layer, unsigned int chan) {
	PROFSECTION("PNGGenerator::calcImage(...)",*mainProfiler);
	
	png_structp  png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	if (!png_ptr) {
		serr->printf("png_create_info_struct failed, %s unavailable.\n",getName().c_str());
		return;
	}
	png_infop  info_ptr = png_create_info_struct(png_ptr);
	if (!info_ptr) {
		png_destroy_write_struct(&png_ptr, NULL);
		serr->printf("png_create_info_struct failed, %s unavailable.\n",getName().c_str());
		return;
	}
	
	png_write_mem_status write_status;
	write_status.buf=images[layer][chan];
	write_status.bufsize=widths[layer]*heights[layer]*3+PNG_HEADER_PAD;
	write_status.offset=0;
	png_set_write_fn(png_ptr, &write_status, user_write_png_data, user_flush_png_data);
	
	if(setjmp(png_jmpbuf(png_ptr))) {
		serr->printf("An error occurred during PNG compression\n");
		png_destroy_write_struct(&png_ptr, &info_ptr);
		return;
	}
	int bit_depth=8;
	int color_type;
	if(getCurrentSourceFormat()==SRC_COLOR)
		color_type=PNG_COLOR_TYPE_RGB;
	else if(getCurrentSourceFormat()==SRC_GRAYSCALE)
		color_type=PNG_COLOR_TYPE_GRAY;
	else {
		serr->printf("calcImage failed - unsuitable or unknown mode/generator pair\n");
		return;
	}
	png_set_IHDR(png_ptr, info_ptr, widths[layer], heights[layer], bit_depth, color_type, PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);
	png_write_info(png_ptr, info_ptr);
	png_bytep row=reinterpret_cast<png_bytep>(src->getImage(layer,chan));
	const unsigned int inc=src->getIncrement(layer);
#ifdef DEBUG
	if(getCurrentSourceFormat()==SRC_COLOR && inc!=3 || getCurrentSourceFormat()==SRC_GRAYSCALE && inc!=1) {
		serr->printf("PNGGenerator only supports color mode from sources with interleaving of 3 samples (increment==3), or grayscale from \"pure\" sources (increment==1)\n");
		png_write_end(png_ptr, NULL);
		return;
	}
#endif
	unsigned int row_stride = src->getStride(layer);
	png_bytep endp=reinterpret_cast<png_bytep>(row+row_stride*heights[layer]);
	for(unsigned int h=0; h<heights[layer]; ++h) {
		if(row+row_stride>endp) {
			serr->printf("Ran out of src image -- bad height?\n");
			break;
		}
		png_write_row(png_ptr, row);
		row+=row_stride;
	}
	png_write_end(png_ptr, NULL);
	png_destroy_write_struct(&png_ptr, &info_ptr);
	bytesUsed[layer][chan]=write_status.offset;
	imageValids[layer][chan]=true;
}

void
PNGGenerator::destruct() {
	FilterBankGenerator::destruct();
	for(unsigned int res=0; res<numLayers; res++)
		delete [] bytesUsed[res];
	delete [] bytesUsed;
	bytesUsed=NULL;
}

/*! @file
 * @brief Implements PNGGenerator, which generates FilterBankEvents containing PNG compressed images
 * @author Ethan Tira-Thompson (ejt) (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.6 $
 * $State: Exp $
 * $Date: 2006/09/16 06:01:41 $
 */
