//-*-c++-*-
#ifndef INCLUDED_plistPrimitives_h_
#define INCLUDED_plistPrimitives_h_

#include "plistBase.h"

//!@name libxml2 forward declarations
//!forward declaration of the libxml2 struct of the same name
extern "C" {
	xmlNode* xmlAddPrevSibling(xmlNode* node, xmlNode* sibling);
	xmlNode* xmlNewText(const xmlChar* s);
	xmlNode* xmlNewComment(const xmlChar* s);
	xmlNode* xmlAddChild(xmlNode * parent, xmlNode* child);
	xmlNode* xmlNewChild(xmlNode* parent, xmlNs* ns, const xmlChar * name, const xmlChar * content);
	int xmlStrEqual(const xmlChar* a, const xmlChar* b);
	xmlChar* xmlNodeGetContent(xmlNode* node);
	void xmlNodeSetContent(xmlNode* node, const xmlChar* content);
	xmlAttr* xmlHasProperty(xmlNode* node, const xmlChar* name);
	xmlChar* xmlGetProperty(xmlNode* node, const xmlChar* name);
	long xmlGetLineNo(xmlNode* node);
	extern void (*xmlFree)(void* ptr);
	void xmlNodeSetName(xmlNode* node, const xmlChar* name);
	void xmlFreeNode(xmlNode* node);
	void xmlUnlinkNode(xmlNode* node);
}

namespace plist {

	//! Implements type-specific functionality through template specialization, mainly involving value conversion and stringification formatting
	/*! Provides smart-pointer style functionality for transparent
	 *  access to the value storage, as well as automatic casting
	 *  to and from the value type so you can almost always treat
	 *  the Primitive as if it was the value itself. */
	template<typename T>
	class Primitive : public PrimitiveBase {
	public:
		//! constructor
		Primitive() : ObjectBase(), val() {}
		//! constructor, provides automatic casting from the value type
		Primitive(const T& v) : ObjectBase(), val(v) {} 
		//! assignment from value type (template specializations add in-place modiciation (e.g. +=, *=))
		Primitive& operator=(const T& v) { val=v; fireValueChanged(); return *this; }

		Primitive& operator+=(const T& v) { val+=v; fireValueChanged(); return *this; } //!< add in-place
		Primitive& operator-=(const T& v) { val-=v; fireValueChanged(); return *this; } //!< subtract in-place
		Primitive& operator*=(const T& v) { val*=v; fireValueChanged(); return *this; } //!< multiply in-place
		Primitive& operator/=(const T& v) { val/=v; fireValueChanged(); return *this; } //!< divide in-place
		
		//! smart pointer access to value
		const T& operator*() const { return val; }
		//! smart pointer access to value
		const T* operator->() const { return &val; }
		
		//! automatic casting to the value type
		operator T() const { return val; }

		// **** Template specializations should provide their own implementations of loadXML and saveXML ****
		/*void loadXML(xmlNode* node) {
			if(node==NULL)
				return;
			//decode base64 from xml node
			//val->loadBuffer(buf,bufsize);
			fireValueChanged(); 
		}
		void saveXML(xmlNode* node) const {
			if(node==NULL)
				return;
			unsigned int bufsize=val->getBinSize();
			char * buf = new char[bufsize];
			val->saveBuffer(buf,bufsize);
			//base64 encode into xml node
			delete [] buf;
		}*/
		
		//! clone definition for Primitive<T>
		PLIST_CLONE_DEF(Primitive<T>,new Primitive<T>(val));

	protected:
		T val; //!< value storage
	};
	//! implements the clone function for Primitive<T>
	PLIST_CLONE_IMPT(class T,Primitive<T>,new Primitive<T>(val));

	
/*! @cond SHOW_PLIST_OBJECT_SPECIALIZATION */
	
	//! provides a @c bool specialization of Primitive<T>
	/*! A @c bool can be treated as either a string or an integer\n
	 *  When saving, "true" or "false" will be used, but users could
	 *  also specify "yes"/"no" or "on"/"off".  If an integer is used,
	 *  it will be interpreted as usual: 0==false, otherwise true. */
	template<>
	class Primitive<bool> : public PrimitiveBase {
	public:
		Primitive() : PrimitiveBase(), val() {} //!< constructor
		Primitive(const bool& v) : PrimitiveBase(), val(v) {} //!< casting constructor
		Primitive& operator=(const bool& v) { val=v; fireValueChanged(); return *this; } //!< assignment constructor
		//bool& operator*() { return val; }
		const bool& operator*() const { return val; } //!< dereference will return data storage
		//bool* operator->() { return &val; }
		const bool* operator->() const { return &val; } //!< can use -> to access members of data storage
		operator bool() const { return val; } //!< casting operator
		
		void loadXML(xmlNode* node); //!< interprets @a node as a bool
		void saveXML(xmlNode* node) const;  //!< saves #val into @a node
		void set(const std::string& str);
		std::string get() const {
			return val?"true":"false";
		}
		//! clone definition for Primitive<bool>
		PLIST_CLONE_DEF(Primitive<bool>,new Primitive<bool>(val));
		
	protected:
		bool val; //!< the actual data storage
	};
	
/*! @endcond */

	//! provides a @c char specialization of plist::Primitive<T>, adds a unique #numeric property to the usual template implementation
	/*! A @c char can be treated as either a string or an integer, you can use
	 *  the setNumeric(bool) function to indicate which style to use when saving */
	template<>
	class Primitive<char> : public PrimitiveBase {
	public:
		Primitive() : PrimitiveBase(), val(), numeric(false) {} //!< constructor
		Primitive(const char& v, bool isNum=false) : PrimitiveBase(), val(v), numeric(isNum) {} //!< casting constructor
		Primitive& operator=(const char& v) { val=v; fireValueChanged(); return *this; } //!< assignment
		Primitive& operator+=(const char& v) { val+=v; fireValueChanged(); return *this; } //!< add-assign
		Primitive& operator-=(const char& v) { val-=v; fireValueChanged(); return *this; } //!< subtract-assign
		Primitive& operator*=(const char& v) { val*=v; fireValueChanged(); return *this; } //!< multiply-assign
		Primitive& operator/=(const char& v) { val/=v; fireValueChanged(); return *this; } //!< divide-assign
		//char& operator*() { return val; }
		const char& operator*() const { return val; } //!< dereference will return data storage
		//char* operator->() { return &val; }
		const char* operator->() const { return &val; } //!< can use -> to access members of data storage
		operator char() const { return val; } //!< casting operator
		
		void setNumeric(bool isNum) { numeric=isNum; } //!< sets #numeric
		bool getNumeric() const { return numeric; } //!< returns #numeric
		
		void loadXML(xmlNode* node); //!< interprets @a node as a char
		void saveXML(xmlNode* node) const; //! saves #val into @a node
		void set(const std::string& str);
		std::string get() const {
			if(numeric) {
				std::stringstream sstr;
				sstr << (int)val;
				return sstr.str();
			} else
				return std::string(1,val);
		}
		//! clone definition for Primitive<char>
		PLIST_CLONE_DEF(Primitive<char>,new Primitive<char>(val));
		
	protected:
		char val; //!< data storage
		bool numeric; //!< if true, requests that saves store the numeric value instead of corresponding character
	};
	
	//! provides an @c unsigned @c char specialization of plist::Primitive<T>, adds a unique #numeric property to the usual template implementation
	/*! A @c char can be treated as either a string or an integer, you can use
	 *  the setNumeric(bool) function to indicate which style to use when saving */
	template<>
	class Primitive<unsigned char> : public PrimitiveBase {
	public:
		Primitive() : PrimitiveBase(), val(), numeric(false) {} //!< constructor
		Primitive(const unsigned char& v, bool isNum=false) : PrimitiveBase(), val(v), numeric(isNum) {} //!< casting constructor
		Primitive& operator=(const unsigned char& v) { val=v; fireValueChanged(); return *this; } //!< assignment
		Primitive& operator+=(const unsigned char& v) { val+=v; fireValueChanged(); return *this; } //!< add-assign
		Primitive& operator-=(const unsigned char& v) { val-=v; fireValueChanged(); return *this; } //!< subtract-assign
		Primitive& operator*=(const unsigned char& v) { val*=v; fireValueChanged(); return *this; } //!< multiple-assign
		Primitive& operator/=(const unsigned char& v) { val/=v; fireValueChanged(); return *this; } //!< divide-assign
		//unsigned char& operator*() { return val; }
		const unsigned char& operator*() const { return val; } //!< dereference will return data storage
		//unsigned char* operator->() { return &val; }
		const unsigned char* operator->() const { return &val; } //!< can use -> to access members of data storage
		operator unsigned char() const { return val; } //!< casting operator
		
		void setNumeric(bool isNum) { numeric=isNum; } //!< sets #numeric
		bool getNumeric() const { return numeric; } //!< returns #numeric
		
		void loadXML(xmlNode* node); //!< interprets @a node as a unsigned char
		void saveXML(xmlNode* node) const; //! saves #val into @a node
		void set(const std::string& str);
		std::string get() const {
			if(numeric) {
				std::stringstream sstr;
				sstr << (int)val;
				return sstr.str();
			} else
				return std::string(1,val);
		}
		//! clone definition for Primitive<unsigned char>
		PLIST_CLONE_DEF(Primitive<unsigned char>,new Primitive<unsigned char>(val));
		
	protected:
		unsigned char val; //!< data storage
		bool numeric; //!< if true, requests that saves store the numeric value instead of corresponding character
	};
	
/*! @cond SHOW_PLIST_OBJECT_SPECIALIZATION */

//! a macro to provide template specializations for the numeric primitives, which only need to vary their string name
#define PLIST_OBJECT_SPECIALIZATION(T,PRIM) \
	template<> \
	class Primitive<T> : public PrimitiveBase { \
	public: \
		/*! \brief constructor */ \
		Primitive() : PrimitiveBase(), val() {} \
		/*! \brief copy constructor, automatic conversion from value type */ \
		Primitive(const T& v) : PrimitiveBase(), val(v) {} \
		/*! \brief assignment from value type */ \
		Primitive& operator=(const T& v) { val=v; fireValueChanged(); return *this; } \
		Primitive& operator+=(const T& v) { val+=v; fireValueChanged(); return *this; } /*!< \brief in-place modification supported */ \
		Primitive& operator-=(const T& v) { val-=v; fireValueChanged(); return *this; } /*!< \brief in-place modification supported */ \
		Primitive& operator*=(const T& v) { val*=v; fireValueChanged(); return *this; } /*!< \brief in-place modification supported */ \
		Primitive& operator/=(const T& v) { val/=v; fireValueChanged(); return *this; } /*!< \brief in-place modification supported */ \
		/*! \brief dereference to access primitive storage */ \
		const T& operator*() const { return val; } \
		/*! \brief dereference to access primitive storage */ \
		const T* operator->() const { return &val; } \
		/*! \brief cast operator to automatically convert to value type */ \
		operator T() const { return val; } \
		/*! \brief interprets \a node as holding the specialization type */ \
		void loadXML(xmlNode* node); \
		/*! \brief saves #val into \a node */ \
		void saveXML(xmlNode* node) const; \
		void set(const std::string& str); \
		std::string get() const { \
			std::stringstream sstr; \
			sstr <<std::setprecision(64)<< val; \
			return sstr.str(); \
		} \
		PLIST_CLONE_DEF(Primitive<T>,new Primitive<T>(val)); \
	protected: \
		/*! \brief storage of primitve value */ \
		T val; \
	}
	
	PLIST_OBJECT_SPECIALIZATION(short,"integer");
	PLIST_OBJECT_SPECIALIZATION(unsigned short,"integer");
	PLIST_OBJECT_SPECIALIZATION(int,"integer");
	PLIST_OBJECT_SPECIALIZATION(unsigned int,"integer");
	PLIST_OBJECT_SPECIALIZATION(long,"integer");
	PLIST_OBJECT_SPECIALIZATION(unsigned long,"integer");
	PLIST_OBJECT_SPECIALIZATION(long long,"integer");
	PLIST_OBJECT_SPECIALIZATION(unsigned long long,"integer");
	PLIST_OBJECT_SPECIALIZATION(float,"real");
	PLIST_OBJECT_SPECIALIZATION(double,"real");
	
#undef PLIST_OBJECT_SPECIALIZATION
/*! @endcond */

	//! Provides a @c std::string specialization of Primitive<T>
	/*! Doesn't need to provide a operator cast because we subclass std::string itself! */
	template<>
	class Primitive<std::string> : public PrimitiveBase, public std::string {
	public:
		Primitive() : PrimitiveBase(), std::string() {} //!< constructor
		Primitive(const std::string& v) : PrimitiveBase(), std::string(v) {} //!< casting constructor
		Primitive(const std::string& v, size_type off, size_type count=npos) : PrimitiveBase(), std::string(v,off,count) {} //!< casting constructor
		Primitive(const char* v, size_type count) : PrimitiveBase(), std::string(v,count) {} //!< casting constructor
		Primitive(const char* v) : PrimitiveBase(), std::string(v) {} //!< casting constructor
		Primitive(size_type count, char v) : PrimitiveBase(), std::string(count,v) {} //!< casting constructor
		Primitive& operator=(const std::string& v) { std::string::operator=(v); fireValueChanged(); return *this; } //!< assignment
		Primitive& operator=(const char* v) { std::string::operator=(v); fireValueChanged(); return *this; } //!< assignment
		Primitive& operator=(char v) { std::string::operator=(v); fireValueChanged(); return *this; } //!< assignment
		//std::string& operator*() { return *this; }
		const std::string& operator*() const { return *this; } //!< dereference will return data storage as a string (for uniformity with the other Primitives, although unnecessary with this instantiation)
		//std::string* operator->() { return this; }
		const std::string* operator->() const { return this; } //!< returns a pointer to this (for uniformity with the other Primitives, although unnecessary with this instantiation)
		//no casting operator because we subclass string
		
		void loadXML(xmlNode* node); //!< interprets @a node as a string
		void saveXML(xmlNode* node) const; //!< saves the content of the string into @a node
		void set(const std::string& str) { operator=(str); } // operator= will fireValueChanged
		std::string get() const { return *this; }
		//! clone definition for Primitive<std::string>
		PLIST_CLONE_DEF(Primitive<std::string>,new Primitive<std::string>(get()));
	};
	
	//! Provides an interface for the use of enumerations in a plist -- you can specify values by either the string name or the corresponding integer value
	/*! Where an array of names is required, you must order the array such that
	 *  the enumeration value can be used as an index into the array.
	 *  The 'maxval' parameters should be one above the maximum enumeration -- 
	 *  if your enumeration runs sequentially from 0 to n, #max should be the
	 *  number of enumerations: n+1 */
	template<typename T> 
	class NamedEnumeration : public PrimitiveBase {
	public:
		NamedEnumeration() : PrimitiveBase(), val(), names(NULL), max(0) {} //!< constructor
		NamedEnumeration(const NamedEnumeration& ne) : PrimitiveBase(ne), val(ne.val), names(ne.names), max(ne.max) {} //!< copy constructor
		NamedEnumeration(const T& v, const char * const* enumnames, unsigned int maxval) : PrimitiveBase(), val(v), names(enumnames), max(maxval) {} //!< constructor, pass initial value, array of strings (the names), and the one-plus-maximum enum value (i.e. the number of enumeration values if they run sequentially from 0)
		NamedEnumeration(const T& v) : PrimitiveBase(), val(v), names(NULL), max(0) {} //!< automatic casting from the enumeration
		NamedEnumeration& operator=(const T& v) { val=v; fireValueChanged(); return *this; } //!< assignment from enumeration value (numeric)
		NamedEnumeration& operator=(const std::string& v) { set(v); return *this; } //!< assignment from string
		NamedEnumeration& operator=(const NamedEnumeration<T>& ne) { val=ne.val; names=ne.names; max=ne.max; return PrimitiveBase::operator=(ne); } //!< assignment
		//T& operator*() { return val; }
		const T& operator*() const { return val; } //!< value access
		operator T() const { return val; } //!< automatic casting to the enumeration value
		void setNames(const char *  const* enumnames, unsigned int maxval) { names=enumnames; max=maxval; } //!< (re)set the array of names and one-plus-maximum enum value (i.e. the number of enumeration values if they run sequentially from 0)
		const char*  const* getNames() const { return names; } //!< returns the array of names previously provided from constructor or setNames()
		const char* getName(unsigned int i) const { return names[i]; } //!< returns the name for a particular index
		unsigned int getMax() const { return max; } //!< returns the one-past-maximum of enumeration values previously provided to constructor or setNames()

		//! interprets @a node as either a string holding the name, or a number corresponding to its index (name is preferred)
		void loadXML(xmlNode* node) {
			if(node==NULL)
				return;
			if(xNodeHasName(node,"true") || xNodeHasName(node,"false")) {
				std::string name=(const char*)xNodeGetName(node);
				unsigned int i=findName(name.c_str());
				if(i==-1U)
					throw bad_format(node,("Error: plist NamedEnumeration cannot be '"+name+"'").c_str());
				val=static_cast<T>(i);
				std::cerr << "Warning: plist NamedEnumeration should use <string>" << name << "</string>, not <" << name << "/>" << std::endl;
			} else if(xNodeHasName(node,"integer") || xNodeHasName(node,"real") || xNodeHasName(node,"string")) {
				xmlChar * cont=xmlNodeGetContent(node);
				try {
					set((char*)cont);
				} catch(const bad_format& err) {
					xmlFree(cont);
					throw bad_format(node,err.what());
				} catch(...) {
					xmlFree(cont);
					throw;
				}
				xmlFree(cont);
			} else
				throw bad_format(node,"Error: plist NamedEnumeration must be numeric or valid string");
		}
		//! saves name of current value (if available, index used otherwise) into @a node
		void saveXML(xmlNode* node) const {
			if(node==NULL)
				return;
			if(names!=NULL && names[val]!=NULL && val>0 && (unsigned int)val<max) {
				xmlNodeSetName(node,(const xmlChar*)"string");
			} else {
				xmlNodeSetName(node,(const xmlChar*)"integer");
			}
			xmlNodeSetContent(node,(const xmlChar*)get().c_str());
		}
		void set(const std::string& str) {
			unsigned int i=findName(str.c_str());
			if(i==-1U) {
				if(sscanf(str.c_str(),"%d",&i)==0)
					throw bad_format(NULL,"Error: plist NamedEnumeration must be numeric or valid string");
			}
			val=static_cast<T>(i);
			fireValueChanged(); 
		}
		std::string get() const {
			if(names!=NULL && names[val]!=NULL && val>=0 && (unsigned int)val<max)
				return names[val];
			std::stringstream str;
			str << val;
			return str.str();
		}
		//! implements the clone function for NamedEnumeration<T>
		PLIST_CLONE_DEF(NamedEnumeration<T>,new NamedEnumeration<T>(*this));

	protected:
		//! returns the enum corresponding to @a name
		unsigned int findName(const char* name) {
			if(name==NULL || names==NULL)
				return -1U;
			//prefer case sensitive match
			for(unsigned int i=0; i<max; i++)
				if(names[i] && strcmp(name,names[i])==0)
					return i;
			//but allow case insensitive if exact match not found
			for(unsigned int i=0; i<max; i++)
				if(names[i] && strcasecmp(name,names[i])==0)
					return i;
			return -1U;
		}
		T val; //!< storage of enum value
		const char *  const* names; //!< pointer to array of names -- enum value must be able to serve as index for lookup
		unsigned int max; //!< one-plus-maximum enum value, i.e. the number of enum entries if they are ordered sequentially from 0
	};
	//! implements the clone function for NamedEnumeration<T>
	PLIST_CLONE_IMPT(class T,NamedEnumeration<T>,new NamedEnumeration<T>(*this));

} //namespace plist


/*! @file
 * @brief 
 * @author Ethan Tira-Thompson (ejt) (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.17 $
 * $State: Exp $
 * $Date: 2006/09/27 20:11:38 $
 */

#endif
