#include "plistPrimitives.h"

namespace plist {
	
	void Primitive<bool>::loadXML(xmlNode* node) {
		if(node==NULL)
			return;
		if(xNodeHasName(node,"true")) {
			val=true;
			fireValueChanged(); 
		} else if(xNodeHasName(node,"false")) {
			val=false;
			fireValueChanged(); 
		} else if(xNodeHasName(node,"integer") || xNodeHasName(node,"real")) {
			xmlChar * cont=xmlNodeGetContent(node);
			std::stringstream str((char*)cont);
			str >> val;
			xmlFree(cont);
			fireValueChanged(); 
		} else if(xNodeHasName(node,"string")) {
			xmlChar * cont=xmlNodeGetContent(node);
			try {
				set((char*)cont);
				std::cerr << "Warning: plist boolean expects value of '<true/>', '<false/>', or numeric.  String value of '" << (char*)cont << "' is not recommended." << std::endl;
			} catch(const bad_format& err) {
				xmlFree(cont);
				throw bad_format(node,err.what());
			} catch(...) {
				xmlFree(cont);
				throw;
			}
			xmlFree(cont);
		} else
			throw bad_format(node,"Error: plist boolean must be 'true', 'false', or numeric type");
	}
	void Primitive<bool>::saveXML(xmlNode* node) const {
		if(node==NULL)
			return;
		xmlNodeSetName(node,(const xmlChar*)(val?"true":"false"));
		xmlNodeSetContent(node,NULL);
	}
	void Primitive<bool>::set(const std::string& str) {
		if(matchTrue(str))
			val=true;
		else if(matchFalse(str))
			val=false;
		else {
			float t;
			if(sscanf(str.c_str(),"%g",&t)==0)
				throw bad_format(NULL,"Error: plist boolean must be 'true', 'false', or numeric type");
			val=t;
		}
		fireValueChanged(); 
	}
	//! implements the clone function for Primitive<bool>
	PLIST_CLONE_IMP(Primitive<bool>,new Primitive<bool>(val));

	
	void Primitive<char>::loadXML(xmlNode* node) {
		if(node==NULL)
			return;
		xmlChar* cont=xmlNodeGetContent(node);
		try {
			if(xNodeHasName(node,"string")) {
				set((char*)cont);
			} else if(xNodeHasName(node,"integer")) {
				val=strtol((const char*)cont,NULL,0);
				fireValueChanged(); 
			} else if(xNodeHasName(node,"real")) {
				val=(char)strtod((const char*)cont,NULL);
				fireValueChanged(); 
			} else if(xNodeHasName(node,"true")) {
				val=true;
				fireValueChanged(); 
			} else if(xNodeHasName(node,"false")) {
				val=false;
				fireValueChanged(); 
			} else {
				throw bad_format(node,"Error: plist char must be either a string or integer");
			} 
		} catch(const bad_format& err) {
			xmlFree(cont);
			throw bad_format(node,err.what());
		} catch(...) {
			xmlFree(cont);
			throw;
		}
		xmlFree(cont);
	}
	void Primitive<char>::saveXML(xmlNode* node) const {
		if(node==NULL)
			return;
		if(numeric) {
			xmlNodeSetName(node,(const xmlChar*)"integer");
			std::stringstream str;
			str << (int)val;
			xmlNodeSetContent(node,(const xmlChar*)str.str().c_str());
		} else {
			xmlNodeSetName(node,(const xmlChar*)"string");
			xmlChar str[2] = {val,'\0'};
			xmlNodeSetContent(node,(const xmlChar*)str);
		}
	}
	void Primitive<char>::set(const std::string& str) {
		if(str.size()==0)
			throw bad_format(NULL,"Error: plist char must have non-empty content");
		val=str[0];
		if(str.size()>1) {
			std::cerr << "Warning: plist expected single char, found multi-char string '" << str << "'";
			if(matchTrue(str))
				val=true;
			else if(matchFalse(str))
				val=false;
			else {
				std::cerr << " (using first character '" << val << "')" << std::endl;
				return;
			}
			std::cerr << " (interpreted as boolean " << (bool)val << ")" << std::endl;
		}
		fireValueChanged(); 
	}
	//! implements the clone function for Primitive<char>
	PLIST_CLONE_IMP(Primitive<char>,new Primitive<char>(val));
	
	void Primitive<unsigned char>::loadXML(xmlNode* node) {
		if(node==NULL)
			return;
		xmlChar* cont=xmlNodeGetContent(node);
		try {
			if(xNodeHasName(node,"string")) {
				set((char*)cont);
			} else if(xNodeHasName(node,"integer")) {
				val=strtol((const char*)cont,NULL,0);
				fireValueChanged(); 
			} else if(xNodeHasName(node,"real")) {
				val=(char)strtod((const char*)cont,NULL);
				fireValueChanged(); 
			} else if(xNodeHasName(node,"true")) {
				val=true;
				fireValueChanged(); 
			} else if(xNodeHasName(node,"false")) {
				val=false;
				fireValueChanged(); 
			} else {
				throw bad_format(node,"Error: plist unsigned char must be either a string or integer");
			} 
		} catch(const bad_format& err) {
			xmlFree(cont);
			throw bad_format(node,err.what());
		} catch(...) {
			xmlFree(cont);
			throw;
		}
		xmlFree(cont);
	}
	void Primitive<unsigned char>::saveXML(xmlNode* node) const {
		if(node==NULL)
			return;
		if(numeric) {
			xmlNodeSetName(node,(const xmlChar*)"integer");
			std::stringstream str;
			str << (int)val;
			xmlNodeSetContent(node,(const xmlChar*)str.str().c_str());
		} else {
			xmlNodeSetName(node,(const xmlChar*)"string");
			xmlChar str[2] = {val,'\0'};
			xmlNodeSetContent(node,(const xmlChar*)str);
		}
	}
	void Primitive<unsigned char>::set(const std::string& str) {
		if(str.size()==0)
			throw bad_format(NULL,"Error: plist char must have non-empty content");
		val=str[0];
		if(str.size()>1) {
			std::cerr << "Warning: plist expected single char, found multi-char string '" << str << "'";
			if(matchTrue(str))
				val=true;
			else if(matchFalse(str))
				val=false;
			else {
				std::cerr << " (using first character '" << val << "')" << std::endl;
				return;
			}
			std::cerr << " (interpreted as boolean " << (bool)val << ")" << std::endl;
		}
		fireValueChanged(); 
	}
	//! implements the clone function for Primitive<unsigned char>
	PLIST_CLONE_IMP(Primitive<unsigned char>,new Primitive<unsigned char>(val));
	
//! a macro to provide template specializations for the numeric primitives, which only need to vary their string name
#define PLIST_OBJECT_SPECIALIZATION(T,PRIM) \
	void Primitive<T>::loadXML(xmlNode* node) { \
		if(node==NULL) \
			return; \
		bool bt=xNodeHasName(node,"true"); \
		bool bf=xNodeHasName(node,"false"); \
		if(!bt && !bf && !xNodeHasName(node,"integer") && !xNodeHasName(node,"real") && !xNodeHasName(node,"string")) \
			throw bad_format(node,"Error: plist "#T" expects "PRIM" type"); \
		if(!xNodeHasName(node,PRIM)) \
			std::cerr << "Warning: plist expected "PRIM" got " << (const char*)xNodeGetName(node) << ", trying to convert. (line " << xmlGetLineNo(node) << ")" << std::endl; \
		if(bt) \
			val = true; \
		else if(bf) \
			val = false; \
		else { \
			xmlChar * cont=xmlNodeGetContent(node); \
			std::stringstream str((const char*)cont); \
			str >> val; \
			xmlFree(cont); \
		} \
		fireValueChanged(); \
	} \
	void Primitive<T>::saveXML(xmlNode* node) const { \
		if(node==NULL) \
			return; \
		xmlNodeSetName(node,(const xmlChar*)PRIM); \
		std::stringstream str; \
		str <<std::setprecision(128)<< val; \
		xmlNodeSetContent(node,(const xmlChar*)str.str().c_str()); \
	} \
	void Primitive<T>::set(const std::string& str) { \
		std::stringstream sstr(str); \
		sstr >> val; \
		while(sstr.good() && isspace(sstr.peek())) \
			sstr.get(); \
		if(sstr.fail()) { \
			if(matchTrue(str)) \
				val=true; \
			else if(matchFalse(str)) \
				val=false; \
			else { \
				std::string err="Expected "; err+=PRIM; err+=" value, got '"+str+"'"; \
				throw bad_format(NULL,err); \
			} \
			std::cerr << "Warning: expected " << PRIM << " value, interpreting '" << str << "' as boolean (value of " << val << ")" << std::endl; \
		} \
		if(sstr.good()) { \
			std::cerr << "Warning: expected " << PRIM << " value, truncating remainder '"; \
			char c=sstr.get(); \
			while(sstr) { std::cerr << c; c=sstr.get(); } \
			std::cerr << "'" << std::endl; \
		} \
		fireValueChanged(); \
	} \
	PLIST_CLONE_IMP(Primitive<T>,new Primitive<T>(val));

	PLIST_OBJECT_SPECIALIZATION(short,"integer");
	PLIST_OBJECT_SPECIALIZATION(unsigned short,"integer");
	PLIST_OBJECT_SPECIALIZATION(int,"integer");
	PLIST_OBJECT_SPECIALIZATION(unsigned int,"integer");
	PLIST_OBJECT_SPECIALIZATION(long,"integer");
	PLIST_OBJECT_SPECIALIZATION(unsigned long,"integer");
	PLIST_OBJECT_SPECIALIZATION(long long,"integer");
	PLIST_OBJECT_SPECIALIZATION(unsigned long long,"integer");
	PLIST_OBJECT_SPECIALIZATION(float,"real");
	PLIST_OBJECT_SPECIALIZATION(double,"real");
	
#undef PLIST_OBJECT_SPECIALIZATION
	
	
	void Primitive<std::string>::loadXML(xmlNode* node) {
		if(node==NULL)
			return;
		if(xNodeHasName(node,"string")) {
			xmlChar * cont=xmlNodeGetContent(node);
			*this=(char*)cont;
			xmlFree(cont);
		} else {
			if(xNodeHasName(node,"integer") || xNodeHasName(node,"real")) {
				xmlChar * cont=xmlNodeGetContent(node);
				*this=(char*)cont;
				xmlFree(cont);
			} else if(xNodeHasName(node,"true"))
				*this="true";
			else if(xNodeHasName(node,"false"))
				*this="false";
			else
				throw bad_format(node,"Error: plist string must be 'true', 'false', or numeric type");
			std::cerr << "Warning: plist string expected, found " << (const char*)xNodeGetName(node) << " on line " << xmlGetLineNo(node) << std::endl;
			fireValueChanged(); 
		}
	}
	void Primitive<std::string>::saveXML(xmlNode* node) const {
		if(node==NULL)
			return;
		xmlNodeSetName(node,(const xmlChar*)"string");
		xmlNodeSetContent(node,(const xmlChar*)c_str());
	}
	//! implements the clone function for Primitive<std::string>
	PLIST_CLONE_IMP(Primitive<std::string>,new Primitive<std::string>(get()));

} //namespace plist

/*! @file
* @brief 
* @author Ethan Tira-Thompson (ejt) (Creator)
*
* $Author: ejt $
* $Name: tekkotsu-3_0 $
* $Revision: 1.10 $
* $State: Exp $
* $Date: 2006/09/16 17:32:40 $
*/

