//-*-c++-*-
#ifndef INCLUDED_plistCollections_h_
#define INCLUDED_plistCollections_h_

#include "plistPrimitives.h"
#include <map>
#include <vector>
#include <set>

namespace plist {
	
	//! Provides a common base class for the collection-oriented primitives, Dictionary, Map, Array, and Vector
	class Collection : public ObjectBase {
	public:
		//! get notified of changes; be sure to call removeCollectionListener() before destructing @a l!
		virtual void addCollectionListener(CollectionListener* l);
		//! no longer take notification of changes to this object's value
		virtual void removeCollectionListener(CollectionListener* l);
		//! test if @a l is currently registered as a listener
		virtual bool isCollectionListener(CollectionListener* l);

		//! insert a new entry to the dictionary, with key @a name and value @a val (replaces any previous entry by same name, but gives a warning)
		virtual void setEntry(const std::string& name, ObjectBase& val, bool warnExists=false)=0;
		//! insert a new entry to the dictionary, with key @a name and value @a val (replaces any previous entry by same name, but gives a warning)
		virtual void addEntry(const std::string& name, ObjectBase& val, const std::string& comment="")=0;
		//! insert a new entry to the dictionary, with key @a name and value @a val, control of deallocation given to collection
		virtual void setEntry(const std::string& name, ObjectBase* val, bool warnExists=false)=0;
		//! insert a new entry to the dictionary, with key @a name and value @a val, control of deallocation given to collection
		virtual void addEntry(const std::string& name, ObjectBase* val, const std::string& comment="")=0;
		
		//! remove the entry with key @a name
		virtual void removeEntry(const std::string& name)=0;
		//! return the value of the key @a name, or NULL if it doesn't exist
		virtual ObjectBase* getEntry(const std::string& name) const=0;
		//! return the value of the key @a name, or NULL if it doesn't exist (equivalent to getEntry(name))
		virtual ObjectBase* operator[](const std::string& name) const { return getEntry(name); }
		//! remove all entries in one fell swoop
		virtual void clear()=0;
		
		//! replaces previous comment for @a name, or adds it if it doesn't already exist (can preceed actual entry!)
		virtual void setComment(const std::string& name, const std::string& comment)=0;
		//! returns comment retrieved from loaded file, or any subsequent call to setComment
		virtual const std::string& getComment(const std::string& name) const=0;

		void setUnusedWarning(bool b) { warnUnused=b; } //!< set #warnUnused
		bool getUnusedWarning() { return warnUnused; } //!< returns #warnUnused

		virtual bool getTrimExtraLoad() const { return trimExtraLoad; } //!< returns #trimExtraLoad
		virtual bool getTrimExtraSave() const { return trimExtraSave; } //!< returns #trimExtraSave
		virtual void setTrimExtra(bool trim) { trimExtraLoad=trimExtraSave=trim; } //!< sets #trimExtraLoad and #trimExtraSave to the save value
		virtual void setTrimExtra(bool trimLoad, bool trimSave) { trimExtraLoad=trimLoad; trimExtraSave=trimSave; } //!< sets #trimExtraLoad and #trimExtraSave
		
	protected:
		//! constructor
		explicit Collection(bool bl, bool bs) : ObjectBase(), collectionListeners(), warnUnused(true), trimExtraLoad(bl), trimExtraSave(bs) {autoFormat=false;}
		//! copy constructor (don't assign listeners)
		Collection(const Collection& d) : ObjectBase(d), collectionListeners(), warnUnused(d.warnUnused), trimExtraLoad(d.trimExtraLoad), trimExtraSave(d.trimExtraSave) {autoFormat=false;}
		//! assignment (don't assign listeners); subclass should call fireEntriesChanged after calling this (and updating it storage)
		Collection& operator=(const Collection& d) { if(&d==this) return *this; ObjectBase::operator=(d); return *this; }
		//! destructor
		~Collection();
		
		//! run through #collectionListeners, calling CollectionListener::plistCollectionEntryAdded(*this,val)
		virtual void fireEntryAdded(ObjectBase& val);
		//! run through #collectionListeners, calling CollectionListener::plistCollectionEntryRemoved(*this,val)
		virtual void fireEntryRemoved(ObjectBase& val);
		//! run through #collectionListeners, calling CollectionListener::plistCollectionEntriesChanged(*this)
		virtual void fireEntriesChanged();
		//! stores a list of the current listeners
		std::list<CollectionListener*>* collectionListeners;
		
		//! return the length of the longest key for formatting purposes
		virtual unsigned int getLongestKeyLen() const=0;
		//! a forwarding function to allow recursive flow of control (gets around not being able to call protected functions on non-this objects)
		static unsigned int getLongestKeyLenOther(const Collection& c) { return c.getLongestKeyLen(); }
		
		//! if true (the default) loadXML will give warnings if there are unused entries in the node it is passed (can only happen if trimExtraLoad is false), or extra values in a file being saved into (can only happen if trimExtraSave is false)
		bool warnUnused;		
		//! if true, unloaded items in the collection will be removed after a load, and new entries will be created for extras in the source (brings storage into complete sync with input)
		bool trimExtraLoad;
		//! if true, unsaved items in the destination will be removed after a save (brings output XML tree into complete sync with storage)
		bool trimExtraSave;
		
		//! when an empty string is needed for not found items
		/*!  (defined as a function instead of just a constant member so there's no issues with initialization order) */
		static const std::string& Collection::emptyStr() {
			static std::string mt;
			return mt;
		}
		//! defines separator between sub-collections
		/*!  (defined as a function instead of just a constant member so there's no issues with initialization order) */
		static const std::string& Collection::subCollectionSep() {
			static std::string sep=".";
			return sep;
		}
	};
	
	//! Maintains a set of (key,value) pairs, where a value can be any subclass of ObjectBase
	/*! This class supports callbacks upon modification through the use of the
	 *  CollectionListener interface.  Note that we only store a pointer to the
	 *  listener list, which is typically unallocated when no listeners are
	 *  active.  This should ensure minimal memory usage per object, as well as
	 *  support safe storage of plist objects in inter-process shared memory
	 *  regions.
	 *
	 *  If you are using these in a shared memory region, just be sure that only
	 *  the process with listeners does any and all modifications, and that it
	 *  unsubscribes before detaching from the region (or else destroys the region
	 *  itself)
	 *
	 *  There isn't a callback if entries themselves are modified, only when new
	 *  entries are added, or old ones removed.  If you want to know any time any
	 *  aspect of any entry is modified, listen for the add and remove callbacks,
	 *  and then add yourself as a listener to each entry individually.  */
	class Dictionary : public Collection {
		friend std::ostream& operator<<(std::ostream& os, const Dictionary& d);
	public:
		//! shorthand for the type of the storage
		typedef std::map<std::string,ObjectBase*> storage_t;
		//! shorthand for iterators to be returned
		typedef storage_t::iterator iterator;
		//! shorthand for iterators to be returned
		typedef storage_t::const_iterator const_iterator;
		
		//! constructor
		Dictionary() : Collection(false,false), dict(), myRef(), comments() {}
		//! constructor
		explicit Dictionary(bool growable) : Collection(growable,growable), dict(), myRef(), comments() {}
		//! copy constructor (don't assign listeners)
		Dictionary(const Dictionary& d) : Collection(d), dict(d.dict), myRef(d.myRef), comments(d.comments) { cloneMyRef(); }
		//! assignment (don't assign listeners); subclass should call fireEntriesChanged after calling this (and updating its own storage)
		Dictionary& operator=(const Dictionary& d) { if(&d==this) return *this; clear(); Collection::operator=(d); dict=d.dict; myRef=d.myRef; comments=d.comments; cloneMyRef(); fireEntriesChanged(); return *this; }
		//! destructor
		~Dictionary() { clear(); }
				
		//! insert a new entry to the map; expects @a val to be either a primitive type, like int, float, etc., or one of the variable-sized Collection's, like Vector
		template<typename T>
			void setValue(const std::string& name, const T& val, bool warnExists=false) { setEntry(name,new Primitive<T>(val),warnExists); }
		//! insert a new entry to the map, and corresponding comment; expects @a val to be either a primitive type, like int, float, etc., or one of the variable-sized Collection's, like Vector
		template<typename T>
			void addValue(const std::string& name, const T& val, const std::string& comment="", bool warnExists=true) { addEntry(name,new Primitive<T>(val),comment,warnExists); }
		//! "specialization" (actually just another override) for handling character arrays as strings
		virtual void setValue(const std::string& name, const char val[], bool warnExists=false) { setEntry(name,new Primitive<std::string>(val),warnExists); }
		//! "specialization" (actually just another override) for handling character arrays as strings
		virtual void addValue(const std::string& name, const char val[], const std::string& comment="") { addEntry(name,new Primitive<std::string>(val),comment,true); }
		//! "specialization" (actually just another override) for handling character arrays as strings
		virtual void addValue(const std::string& name, const char val[], const std::string& comment, bool warnExists) { addEntry(name,new Primitive<std::string>(val),comment,warnExists); }
		
		//! insert a new entry to the dictionary, with key @a name and value @a val (replaces any previous entry by same name, but can give a warning)
		virtual void setEntry(const std::string& name, ObjectBase& val, bool warnExists=false);
		//! insert a new entry to the dictionary, with key @a name and value @a val (replaces any previous entry by same name, but can give a warning)
		virtual void addEntry(const std::string& name, ObjectBase& val, const std::string& comment="") { addEntry(name,val,comment,true); }
		//! insert a new entry to the dictionary, with key @a name and value @a val (replaces any previous entry by same name, but can give a warning)
		virtual void addEntry(const std::string& name, ObjectBase& val, const std::string& comment, bool warnExists);
		virtual void setEntry(const std::string& name, ObjectBase* val, bool warnExists=false);
		virtual void addEntry(const std::string& name, ObjectBase* val, const std::string& comment="") { addEntry(name,val,comment,true); }
		//! insert a new entry to the dictionary, with key @a name and value @a val (replaces any previous entry by same name, but can give a warning)
		virtual void addEntry(const std::string& name, ObjectBase* val, const std::string& comment, bool warnExists);
				
		//! remove the entry with key @a name
		virtual void removeEntry(const std::string& name);
		//! return the value of the key @a name, or NULL if it doesn't exist
		virtual ObjectBase* getEntry(const std::string& name) const;
				
		virtual void clear();
		
		//! return an STL const_iterator to the first entry
		const_iterator begin() const { return dict.begin(); }
		//! return the one-past-end const_iterator
		const_iterator end() const { return dict.end(); }
		//! return the size of the dictionary
		size_t size() const { return dict.size(); }
		
		//! replaces previous comment for @a name, or adds it if it doesn't already exist (can preceed actual entry!)
		void setComment(const std::string& name, const std::string& comment);
		//! returns comment retrieved from loaded file, or any subsequent call to setComment
		const std::string& getComment(const std::string& name) const;
		
		virtual void loadXML(xmlNode* node);
		virtual void saveXML(xmlNode* node) const;
		
		virtual std::string toString() const;
		
		//! clone implementation for Dictionary
		PLIST_CLONE_DEF(Dictionary,new Dictionary(*this));
		
	protected:
		virtual void fireEntryRemoved(ObjectBase& val);
		
		//! indicates that the storage implementation should mark this as an externally supplied heap reference, which needs to be deleted on removal/destruction
		virtual void takeObject(const std::string& name, ObjectBase* obj);
		
		//! called with each node being loaded so subclass can handle appropriately
		virtual bool loadXMLNode(const std::string& key, xmlNode* val, const std::string& comment);
		
		virtual unsigned int getLongestKeyLen() const;
		
		//! returns an entry matching just the prefix
		/*! @param[in] name the name being looked up
		 *  @param[out] seppos the position of the separator (sub-collections are separated by '.')
		 *  @return iterator of the sub entry*/
		iterator getSubEntry(const std::string& name, std::string::size_type& seppos);
		//! returns an entry matching just the prefix
		/*! @param[in] name the name being looked up
		 *  @param[out] seppos the position of the separator (sub-collections are separated by '.')
		 *  @return iterator of the sub entry*/
		const_iterator getSubEntry(const std::string& name, std::string::size_type& seppos) const;
		
		//! called after an assignment or copy to clone the objects in #myRef to perform a deep copy
		virtual void cloneMyRef();
		
		//! storage of entries -- mapping from keys to value pointers
		storage_t dict;
		
		//! objects which have been handed over to the collection for eventual de-allocation
		std::set<ObjectBase*> myRef;

		//! shorthand for the type of #comments
		typedef std::map<std::string,std::string> comments_t;
		//! storage of entry comments -- mapping from keys to help text comments for manual editing or user feedback
		/*! not every key necessarily has a comment! */
		comments_t comments;
	};
	//! provides textual output
	std::ostream& operator<<(std::ostream& os, const Dictionary& d);
	
	//! A collection of plist objects, similar to a Dictionary, but no keys -- order matters!
	/*! There's two versions of the plist array -- one is this Array class, the other is
	 *  the Vector.  This class is designed to be a fixed size -- you should add the entries
	 *  which you expect to find in the array, and extra entries will be ignored when loading
	 *  (with an optional warning).  You can mix different primitive types in any order
	 *  you choose.
	 *
	 *  The other class, Vector, takes the opposite tack -- it handles loading a variable
	 *  number of entries, but handles the allocation of those objects internally.
	 */
	class Array : public Collection {
		friend std::ostream& operator<<(std::ostream& os, const Array& d);
	public:
		//! shorthand for the type of the storage
		typedef std::vector<ObjectBase*> storage_t;
		//! shorthand for iterators to be returned
		typedef storage_t::iterator iterator;
		//! shorthand for iterators to be returned
		typedef storage_t::const_iterator const_iterator;
		
		//! constructor
		Array() : Collection(false,false), arr(), myRef(), comments() {}
		//! constructor
		explicit Array(bool growable) : Collection(growable,growable), arr(), myRef(), comments() {}
		//! copy constructor (don't assign listeners)
		Array(const Array& d) : Collection(d), arr(d.arr), myRef(d.myRef), comments(d.comments) { cloneMyRef(); }
		//! assignment (don't assign listeners); subclass should call fireEntriesChanged after calling this (and updating it storage)
		Array& operator=(const Array& d) { if(&d==this) return *this; clear(); Collection::operator=(d); arr=d.arr; myRef=d.myRef; comments=d.comments; cloneMyRef(); return *this; }
		//! destructor
		~Array() { clear(); }
		
		//! insert a new entry to the end of the vector, and corresponding comment; expects @a val to be either a primitive type, like int, float, etc., or one of the variable-sized Collection's, like Vector, control of (de)allocation will be assumed by the Vector
		template<typename T>
			void addValue(const T& val, const std::string& comment="") { addEntry(new Primitive<T>(val),comment); }
		//! "specialization" (actually just another override) for handling character arrays as strings
		virtual void addValue(const char val[], const std::string& comment="") { if(comment.size()>0) setComment(size(),comment); arr.push_back(new Primitive<std::string>(val)); takeObject(size()-1,arr.back()); fireEntryAdded(*arr.back()); }
		
		//! replaces previous entry at the specified @a index, which must represent an integer value less than or equal to the current array size; clone of @a val will be added
		virtual void addEntry(ObjectBase& val, const std::string& comment="") { if(comment.size()>0) setComment(size(),comment); arr.push_back(&val); fireEntryAdded(*arr.back()); }
		//! replaces previous entry at the specified @a index, which must represent an integer value less than or equal to the current array size; clone of @a val will be added
		virtual void addEntry(ObjectBase* val, const std::string& comment="") { if(comment.size()>0) setComment(size(),comment); arr.push_back(val); takeObject(size()-1,val); fireEntryAdded(*arr.back()); }
		
		//! replaces previous entry at the specified @a index, which must be less than or equal to the current array size, control of (de)allocation will be assumed by the Vector
		template<typename T>
			void setValue(size_t index, const T& val, bool warnExists=false) { setEntry(index,new Primitive<T>(val),warnExists); }
		//! replaces previous entry at the specified @a index, which must be less than or equal to the current array size, control of (de)allocation will be assumed by the Vector
		template<typename T>
			void addValue(size_t index, const T& val, const std::string& comment="") { addEntry(index,new Primitive<T>(val),comment); }
		//! "specialization" (actually just another override) for handling character arrays as strings
		virtual void setValue(size_t index, const char val[], bool warnExists=false) { setEntry(index,new Primitive<std::string>(val),warnExists); }
		//! "specialization" (actually just another override) for handling character arrays as strings
		virtual void addValue(size_t index, const char val[], const std::string& comment="") { addEntry(index,new Primitive<std::string>(val),comment); }
		
		//! replaces previous entry at the specified @a index, which must be less than or equal to the current array size; clone will be added
		virtual void setEntry(size_t index, ObjectBase& val, bool warnExists=false);
		//! replaces previous entry at the specified @a index, which must be less than or equal to the current array size; clone will be added
		virtual void addEntry(size_t index, ObjectBase& val, const std::string& comment="");
		//! replaces previous entry at the specified @a index, which must be less than or equal to the current array size; clone will be added
		virtual void setEntry(size_t index, ObjectBase* val, bool warnExists=false);
		//! replaces previous entry at the specified @a index, which must be less than or equal to the current array size; clone will be added
		virtual void addEntry(size_t index, ObjectBase* val, const std::string& comment="");
		
		//! replaces previous entry of the same name, which must represent an integer value less than or equal to the current array size, control of (de)allocation will be assumed by the Vector
		template<typename T>
			void setValue(const std::string& name, const T& val,bool warnExists=false) { setEntry(name,new Primitive<T>(val),warnExists); }
		//! replaces previous entry of the same name, which must represent an integer value less than or equal to the current array size, control of (de)allocation will be assumed by the Vector
		template<typename T>
			void addValue(const std::string& name, const T& val, const std::string& comment="") { addEntry(name,new Primitive<T>(val),comment); }
		//! "specialization" (actually just another override) for handling character arrays as strings
		virtual void setValue(const std::string& name, const char val[], bool warnExists=false) { setEntry(name,new Primitive<std::string>(val),warnExists); }
		//! "specialization" (actually just another override) for handling character arrays as strings
		virtual void addValue(const std::string& name, const char val[], const std::string& comment="") { addEntry(name,new Primitive<std::string>(val),comment); }
		
		//! replaces previous entry of the same name, which must represent an integer value less than or equal to the current array size; clone will be added
		virtual void setEntry(const std::string& name, ObjectBase& val, bool warnExists=false);
		//! replaces previous entry of the same name, which must represent an integer value less than or equal to the current array size; clone will be added
		virtual void addEntry(const std::string& name, ObjectBase& val, const std::string& comment="");
		//! replaces previous entry of the same name, which must represent an integer value less than or equal to the current array size; clone will be added
		virtual void setEntry(const std::string& name, ObjectBase* val, bool warnExists=false);
		//! replaces previous entry of the same name, which must represent an integer value less than or equal to the current array size; clone will be added
		virtual void addEntry(const std::string& name, ObjectBase* val, const std::string& comment="");
		
		//! remove the entry at position @a index
		virtual void removeEntry(size_t index);
		//! return the value at position @a index
		ObjectBase& getEntry(size_t index) const { return *arr[index]; }
		//! return the value of the key @a name, or NULL if it doesn't exist (equivalent to getEntry(index))
		ObjectBase& operator[](size_t index) const { return *arr[index]; }
		
		virtual void removeEntry(const std::string& name);
		virtual ObjectBase* getEntry(const std::string& name) const;
		virtual ObjectBase* operator[](const std::string& name) const { return getEntry(name); }
		
		virtual void clear();
		
		//! return an STL const_iterator to the first entry
		const_iterator begin() const { return arr.begin(); }
		//! return the one-past-end const_iterator
		const_iterator end() const { return arr.end(); }
		//! return the size of the dictionary
		size_t size() const { return arr.size(); }
		
		//! replaces previous comment for @a name, or adds it if it doesn't already exist (can preceed actual entry!)
		virtual void setComment(size_t index, const std::string& comment);
		//! returns comment retrieved from loaded file, or any subsequent call to setComment
		virtual const std::string& getComment(size_t index) const;
		
		//! replaces previous comment for @a name, or adds it if it doesn't already exist (can preceed actual entry!)
		virtual void setComment(const std::string& name, const std::string& comment);
		//! returns comment retrieved from loaded file, or any subsequent call to setComment
		virtual const std::string& getComment(const std::string& name) const;
		
		virtual void loadXML(xmlNode* node);
		virtual void saveXML(xmlNode* node) const;
		
		//! returns index corresponding to @a name, which should encode an integer value less than or equal to the current size
		size_t getIndex(const std::string& name) const;
		
		virtual std::string toString() const;
		
		//! clone implementation for Array
		PLIST_CLONE_DEF(Array,new Array(*this));
			
	protected:
		virtual void fireEntryRemoved(ObjectBase& val);
		
		//! indicates that the storage implementation should mark this as an externally supplied heap reference, which needs to be deleted on removal/destruction
		virtual void takeObject(size_t index, ObjectBase* obj);
		
		//! called with each node being loaded so subclass can handle appropriately
		virtual bool loadXMLNode(size_t index, xmlNode* val, const std::string& comment);
		
		virtual unsigned int getLongestKeyLen() const;
		
		//! returns an entry matching just the prefix
		/*! @param[in] name the name being looked up
		 *  @param[out] seppos the position of the separator (sub-collections are separated by '.')
		 *  @return iterator of the sub entry*/
		iterator getSubEntry(const std::string& name, std::string::size_type& seppos);
		//! returns an entry matching just the prefix
		/*! @param[in] name the name being looked up
		 *  @param[out] seppos the position of the separator (sub-collections are separated by '.')
		 *  @return iterator of the sub entry*/
		const_iterator getSubEntry(const std::string& name, std::string::size_type& seppos) const;

		//! called after an assignment or copy to clone the objects in #myRef to perform a deep copy
		virtual void cloneMyRef();
		
		//! storage of entries -- mapping from keys to value pointers
		storage_t arr;
		
		//! objects which have been handed over to the collection for eventual de-allocation
		std::set<ObjectBase*> myRef;
		
		//! shorthand for the type of #comments
		typedef std::map<size_t,std::string> comments_t;
		//! storage of entry comments -- mapping from keys to help text comments for manual editing or user feedback
		/*! not every key necessarily has a comment! */
		comments_t comments;
	};
	//! provides textual output
	std::ostream& operator<<(std::ostream& os, const Array& d);
	
	//! specialization of Dictionary::setValue() for ObjectBase subclasses
	template<>
	inline void Dictionary::setValue(const std::string& name, const ObjectBase& val, bool warnExists) { setEntry(name,dynamic_cast<ObjectBase*>(val.clone()),warnExists); }
	//! specialization of Dictionary::addValue() for ObjectBase subclasses
	template<>
	inline void Dictionary::addValue(const std::string& name, const ObjectBase& val, const std::string& comment, bool warnExists/*=true*/) { addEntry(name,dynamic_cast<ObjectBase*>(val.clone()),comment,warnExists); }
	//! specialization of Dictionary::setValue() for char* strings
	template<>
	inline void Dictionary::setValue(const std::string& name, const char* const& val, bool warnExists) { setEntry(name,new Primitive<std::string>(val),warnExists); }
	//! specialization of Dictionary::addValue() for char* strings
	template<>
	inline void Dictionary::addValue(const std::string& name, const char* const& val, const std::string& comment, bool warnExists/*=true*/) { addEntry(name,new Primitive<std::string>(val),comment,warnExists); }
	//! specialization of Dictionary::setValue() for char* strings
	template<>
	inline void Dictionary::setValue(const std::string& name, char* const& val, bool warnExists) { setEntry(name,new Primitive<std::string>(val),warnExists); }
	//! specialization of Dictionary::addValue() for char* strings
	template<>
	inline void Dictionary::addValue(const std::string& name, char* const& val, const std::string& comment, bool warnExists/*=true*/) { addEntry(name,new Primitive<std::string>(val),comment,warnExists); }
		
	//! specialization of Array::addValue() for ObjectBase subclasses
	template<>
	inline void Array::addValue(const ObjectBase& val, const std::string& comment/*=""*/) { addEntry(dynamic_cast<ObjectBase*>(val.clone()),comment); }
	//! specialization of Array::addValue() for char* strings
	template<>
	inline void Array::addValue(const char* const& val, const std::string& comment/*=""*/) { addEntry(new Primitive<std::string>(val),comment); }
	//! specialization of Array::addValue() for char* strings
	template<>
	inline void Array::addValue(char* const& val, const std::string& comment/*=""*/) { addEntry(new Primitive<std::string>(val),comment); }
	
	//! specialization of Array::setValue() for ObjectBase subclasses
	template<>
	inline void Array::setValue(size_t index, const ObjectBase& val, bool warnExists) { setEntry(index,dynamic_cast<ObjectBase*>(val.clone()),warnExists); }
	//! specialization of Array::setValue() for char* strings
	template<>
	inline void Array::setValue(size_t index, const char* const& val, bool warnExists) { setEntry(index,new Primitive<std::string>(val),warnExists); }
	//! specialization of Array::setValue() for char* strings
	template<>
	inline void Array::setValue(size_t index, char* const& val, bool warnExists) { setEntry(index,new Primitive<std::string>(val),warnExists); }
	//! specialization of Array::addValue() for ObjectBase subclasses
	template<>
	inline void Array::addValue(size_t index, const ObjectBase& val, const std::string& comment) { addEntry(index,dynamic_cast<ObjectBase*>(val.clone()),comment); }
	//! specialization of Array::addValue() for char* strings
	template<>
	inline void Array::addValue(size_t index, const char* const& val, const std::string& comment) { addEntry(index,new Primitive<std::string>(val),comment); }
	//! specialization of Array::addValue() for char* strings
	template<>
	inline void Array::addValue(size_t index, char* const& val, const std::string& comment) { addEntry(index,new Primitive<std::string>(val),comment); }
	
	//! specialization of Array::setValue() for ObjectBase subclasses
	template<>
	inline void Array::setValue(const std::string& name, const ObjectBase& val, bool warnExists) { setEntry(name,dynamic_cast<ObjectBase*>(val.clone()),warnExists); }
	//! specialization of Array::setValue() for char* strings
	template<>
	inline void Array::setValue(const std::string& name, const char* const& val, bool warnExists) { setEntry(name,new Primitive<std::string>(val),warnExists); }
	//! specialization of Array::setValue() for char* strings
	template<>
	inline void Array::setValue(const std::string& name, char* const& val, bool warnExists) { setEntry(name,new Primitive<std::string>(val),warnExists); }
	//! specialization of Array::addValue() for ObjectBase subclasses
	template<>
	inline void Array::addValue(const std::string& name, const ObjectBase& val, const std::string& comment) { addEntry(name,dynamic_cast<ObjectBase*>(val.clone()),comment); }
	//! specialization of Array::addValue() for char* strings
	template<>
	inline void Array::addValue(const std::string& name, const char* const& val, const std::string& comment) { addEntry(name,new Primitive<std::string>(val),comment); }
	//! specialization of Array::addValue() for char* strings
	template<>
	inline void Array::addValue(const std::string& name, char* const& val, const std::string& comment) { addEntry(name,new Primitive<std::string>(val),comment); }
	
} //namespace plist


/*! @file
 * @brief 
 * @author Ethan Tira-Thompson (ejt) (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.8 $
 * $State: Exp $
 * $Date: 2006/09/19 05:39:33 $
 */

#endif
