#include "plistCollections.h"
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <iomanip>

//better to put this here instead of the header
using namespace std; 

namespace plist {

	Collection::~Collection() {
		delete collectionListeners;
		collectionListeners=NULL;
	}
	
	void Collection::addCollectionListener(CollectionListener* l) {
		if(l!=NULL) {
			if(collectionListeners==NULL)
				collectionListeners=new std::list<CollectionListener*>;
			collectionListeners->push_back(l);
		}
	}
		
	void Collection::removeCollectionListener(CollectionListener* l) {
		if(collectionListeners==NULL)
			return;
		std::list<CollectionListener*>::iterator it=find(collectionListeners->begin(),collectionListeners->end(),l);
		if(it!=collectionListeners->end()) {
			collectionListeners->erase(it);
			if(collectionListeners->empty()) {
				delete collectionListeners;
				collectionListeners=NULL;
			}
		}
	}
	
	bool Collection::isCollectionListener(CollectionListener* l) {
		if(l==NULL)
			return false;
		if(collectionListeners==NULL)
			return false;
		std::list<CollectionListener*>::iterator it=find(collectionListeners->begin(),collectionListeners->end(),l);
		return it!=collectionListeners->end();
	}
		
	void Collection::fireEntryAdded(ObjectBase& val) {
		if(collectionListeners==NULL)
			return;
		std::list<CollectionListener*>::iterator it=collectionListeners->begin();
		while(it!=collectionListeners->end()) {
			std::list<CollectionListener*>::iterator cur=it++; //increment early in case the listener changes subscription
			(*cur)->plistCollectionEntryAdded(*this,val);
		}
	}
	
	void Collection::fireEntryRemoved(ObjectBase& val) {
		if(collectionListeners==NULL)
			return;
		std::list<CollectionListener*>::iterator it=collectionListeners->begin();
		while(it!=collectionListeners->end()) {
			std::list<CollectionListener*>::iterator cur=it++; //increment early in case the listener changes subscription
			(*cur)->plistCollectionEntryRemoved(*this,val);
		}
	}
	
	void Collection::fireEntriesChanged() {
		if(collectionListeners==NULL)
			return;
		std::list<CollectionListener*>::iterator it=collectionListeners->begin();
		while(it!=collectionListeners->end()) {
			std::list<CollectionListener*>::iterator cur=it++; //increment early in case the listener changes subscription
			(*cur)->plistCollectionEntriesChanged(*this);
		}
	}
	


	
	
	void Dictionary::setEntry(const std::string& name, ObjectBase& val, bool warnExists/*=false*/) {
		storage_t::iterator it=dict.find(name);
		if(it!=dict.end()) {
			//found exact name match
			if(&val==it->second)
				return; // same val reference already registered
			if(warnExists) {
				cerr << "Warning: new entry ("<<name<<","<<val<<") conflicted with previous entry ("<<name<<","<<(it->second)<<")"<<endl;
				cerr << "         (use setEntry(...,false) if you expect you might need to overwrite)" << endl;
			}
			removeEntry(name);
			//fall through to add new val
		} else {
			//perhaps there's a sub-dictionary
			string::size_type p;
			it=getSubEntry(name,p);
			if(it!=dict.end()) {
				//found a matching sub-collection, have it find the rest recursively
				Collection* d=dynamic_cast<Collection*>(it->second);
				d->setEntry(name.substr(p+1),val,warnExists);
				return;
			}
			//if still here, no sub-collection, fall through to add new entry
		}
		dict[name]=&val;
		fireEntryAdded(val);
	}
	void Dictionary::addEntry(const std::string& name, ObjectBase& val, const std::string& comment, bool warnExists) {
		storage_t::iterator it=dict.find(name);
		if(it!=dict.end()) {
			//found exact name match
			if(&val==it->second)
				return; // same val reference already registered
			if(warnExists) {
				cerr << "Warning: new entry ("<<name<<","<<val<<") conflicted with previous entry ("<<name<<","<<(it->second)<<")"<<endl;
				cerr << "         (use setEntry() if you expect you might need to overwrite)" << endl;
			}
			removeEntry(name);
			//fall through to add new val
		} else {
			//perhaps there's a sub-dictionary
			string::size_type p;
			it=getSubEntry(name,p);
			if(it!=dict.end()) {
				//found a matching sub-collection, have it find the rest recursively
				Collection* d=dynamic_cast<Collection*>(it->second);
				d->addEntry(name.substr(p+1),val,comment);
				return;
			}
			//if still here, no sub-collection, fall through to add new entry
		}
		if(comment.size()>0)
			comments[name]=comment;
		dict[name]=&val;
		fireEntryAdded(val);
	}
	void Dictionary::setEntry(const std::string& name, ObjectBase* val, bool warnExists/*=false*/) {
		storage_t::iterator it=dict.find(name);
		if(it!=dict.end()) {
			//found exact name match
			if(val==it->second)
				return; // same val reference already registered
			if(warnExists) {
				cerr << "Warning: new entry ("<<name<<","<<val<<") conflicted with previous entry ("<<name<<","<<(it->second)<<")"<<endl;
				cerr << "         (use setEntry(...,false) if you expect you might need to overwrite)" << endl;
			}
			removeEntry(name);
			//fall through to add new val
		} else {
			//perhaps there's a sub-dictionary
			string::size_type p;
			it=getSubEntry(name,p);
			if(it!=dict.end()) {
				//found a matching sub-collection, have it find the rest recursively
				Collection* d=dynamic_cast<Collection*>(it->second);
				d->setEntry(name.substr(p+1),val,warnExists);
				return;
			}
			//if still here, no sub-collection, fall through to add new entry
		}
		dict[name]=val;
		takeObject(name,val);
		fireEntryAdded(*val);
	}
	void Dictionary::addEntry(const std::string& name, ObjectBase* val, const std::string& comment, bool warnExists) {
		storage_t::iterator it=dict.find(name);
		if(it!=dict.end()) {
			//found exact name match
			if(val==it->second)
				return; // same val reference already registered
			if(warnExists) {
				cerr << "Warning: new entry ("<<name<<","<<val<<") conflicted with previous entry ("<<name<<","<<(it->second)<<")"<<endl;
				cerr << "         (use setEntry() if you expect you might need to overwrite)" << endl;
			}
			removeEntry(name);
			//fall through to add new val
		} else {
			//perhaps there's a sub-dictionary
			string::size_type p;
			it=getSubEntry(name,p);
			if(it!=dict.end()) {
				//found a matching sub-collection, have it find the rest recursively
				if(Dictionary* d=dynamic_cast<Dictionary*>(it->second)) {
					d->addEntry(name.substr(p+1),val,comment,warnExists);
				} else {
					Collection* c=dynamic_cast<Collection*>(it->second);
					c->addEntry(name.substr(p+1),val,comment);
				}
				return;
			}
			//if still here, no sub-collection, fall through to add new entry
		}
		dict[name]=val;
		if(comment.size()>0)
			comments[name]=comment;
		takeObject(name,val);
		fireEntryAdded(*val);
	}
	
	
	void Dictionary::removeEntry(const std::string& name) {
		storage_t::iterator it=dict.find(name);
		if(it!=dict.end()) {
			//found exact name match
			ObjectBase* obj=it->second;
			dict.erase(it);
			fireEntryRemoved(*obj);
		} else {
			//perhaps there's a sub-dictionary
			string::size_type p;
			it=getSubEntry(name,p);
			if(it!=dict.end()) {
				//found a matching sub-collection, have it find the rest recursively
				Collection* d=dynamic_cast<Collection*>(it->second);
				d->removeEntry(name.substr(p+1));
			}
		}
	}

	ObjectBase* Dictionary::getEntry(const std::string& name) const {
		//do we have a key with this name?
		const_iterator it=dict.find(name);
		if(it!=dict.end())
			return it->second; //yes, return it
		
		//perhaps there's a sub-dictionary
		string::size_type p;
		it=getSubEntry(name,p);
		if(it!=dict.end()) {
			//found a matching sub-collection, have it find the rest recursively
			const Collection* d=dynamic_cast<const Collection*>(it->second);
			return d->getEntry(name.substr(p+1));
		}
		
		//got nothin'
		return NULL;
	}

	void Dictionary::setComment(const std::string& name, const std::string& comment) {
		storage_t::iterator it=dict.find(name);
		if(it==dict.end()) {
			//perhaps there's a sub-dictionary
			string::size_type p;
			it=getSubEntry(name,p);
			if(it!=dict.end()) {
				//found a matching sub-collection, have it find the rest recursively
				Collection* d=dynamic_cast<Collection*>(it->second);
				d->setComment(name.substr(p+1),comment);
				return;
			}
		}
		if(comment.size()>0)
			comments[name]=comment;
		else
			comments.erase(name);
	}

	const std::string& Dictionary::getComment(const std::string& name) const {
		storage_t::const_iterator it=dict.find(name);
		if(it!=dict.end()) {
			//found exact name match
			comments_t::const_iterator cit=comments.find(name);
			return (cit!=comments.end()) ? cit->second : emptyStr();
		} else {
			//perhaps there's a sub-dictionary
			string::size_type p;
			it=getSubEntry(name,p);
			if(it!=dict.end()) {
				//found a matching sub-collection, have it find the rest recursively
				const Collection* d=dynamic_cast<const Collection*>(it->second);
				return d->getComment(name.substr(p+1));
			}
			return emptyStr();
		}
	}

	void Dictionary::loadXML(xmlNode* node) {
		//check if our node has been set to NULL (invalid or not found)
		if(node==NULL)
			return;
		
		std::string comment;
		set<std::string> seen;
		//process children nodes
		for(xmlNode* cur = skipToElement(node->children,comment); cur!=NULL; cur = skipToElement(cur->next,comment)) {
						
			//find the next key node
			xmlNode * k=cur;
			if(xmlStrcmp(k->name, (const xmlChar *)"key"))
				throw bad_format(k,"Dictionary format error: expect data in pairs of key and value (two values found in a row)");
			cur=skipToElement(cur->next);
			
			//find the following value (non-key) node
			xmlNode * v=cur;
			if(v==NULL)
				throw bad_format(cur,"Dictionary format error: expect data in pairs of key and value (dictionary ended with hanging key)");
			if(!xmlStrcmp(v->name, (const xmlChar *)"key"))
				throw bad_format(v,"Dictionary format error: expect data in pairs of key and value (two keys found in a row)");
			
			//find corresponding entry
			xmlChar* cont=xmlNodeGetContent(k);
			string key=(const char*)cont;
			xmlFree(cont);
			seen.insert(key);
			loadXMLNode(key,v,comment);
		}
		if(trimExtraLoad && seen.size()!=size()) {
			set<std::string> rem;
			for(const_iterator it=begin(); it!=end(); ++it) {
				if(seen.find(it->first)==seen.end())
					rem.insert(it->first);
			}
			for(set<std::string>::const_iterator it=rem.begin(); it!=rem.end(); ++it)
				removeEntry(*it);
		}
	}
	
	void Dictionary::saveXML(xmlNode* node) const {
		//check if our node has been set to NULL (invalid or not found)
		if(node==NULL)
			return;
		
		//set the type of the current node
		xmlNodeSetName(node,(const xmlChar*)"dict");
		
		// we'll use this to keep track of which nodes were already present, so we'll
		// know which ones were missing for which we need to make new nodes
		storage_t seen;

		//find the depth of the target node in the xml tree to maintain proper indentation
		std::string perIndent("    ");
		std::string indentStr;
		for(xmlNode* cur=node->parent; cur!=NULL; cur=cur->parent) {
			if((void*)cur==(void*)node->doc) { //if we hit the document node, discount it and we're done
				if(indentStr.size()>0)
					indentStr=indentStr.substr(0,indentStr.size()-perIndent.size());
				break;
			}
			indentStr+=perIndent;
		}
		
		//This will hold any comments found between elements -- if no comment is found, a new one may be added
		std::string comment;

		//process children nodes
		xmlNode* prev=node->children;
		for(xmlNode* cur = skipToElement(node->children,comment); cur!=NULL; cur = skipToElement(cur,comment)) {
			
			//find the next key node
			xmlNode * k=cur;
			if(xmlStrcmp(k->name, (const xmlChar *)"key"))
				throw bad_format(k,"Dictionary format error: expect data in pairs of key and value (two values found in a row)");
			cur=skipToElement(cur->next);
			
			//find the following value (non-key) node
			xmlNode * v=cur;
			if(v==NULL)
				throw bad_format(cur,"Dictionary format error: expect data in pairs of key and value (dictionary ended with hanging key)");
			if(!xmlStrcmp(v->name, (const xmlChar *)"key"))
				throw bad_format(v,"Dictionary format error: expect data in pairs of key and value (two keys found in a row)");
			
			//find corresponding entry
			xmlChar* cont=xmlNodeGetContent(k);
			string key=(const char*)cont;
			xmlFree(cont);
			storage_t::const_iterator it=dict.find(key);
			if(it==dict.end()) {
				cur=xNodeGetNextNode(cur);
				if(trimExtraSave) {
					while(prev!=cur) {
						xmlNode* n=prev;
						prev=xNodeGetNextNode(prev);
						xmlUnlinkNode(n);
						xmlFreeNode(n);
					}
				} else {
					if(warnUnused)
						cerr << "Warning: saving over existing plist dictionary, key '" << key << "' does not match a registered variable.  Ignoring..." << endl;
				}
				prev=cur;
				continue;
			}
			if(comment.size()==0) {
				bool isSub=dynamic_cast<Collection*>(it->second);
				if(isSub) {
					xmlAddPrevSibling(k,xmlNewText((const xmlChar*)"\n"));
					xmlAddPrevSibling(k,xmlNewComment((const xmlChar*)("======== "+it->first+" ========").c_str()));
				}
				comments_t::const_iterator cit=comments.find(key);
				if(cit!=comments.end()) {
					if(isSub || cit->second.substr(0,key.size())==key)
						comment=cit->second;
					else //if not a sub-dictionary, and comment doesn't already start with entry name, prepend entry name
						comment=key+": "+cit->second;
					xmlAddPrevSibling(k,xmlNewText((const xmlChar*)"\n"));
					xmlAddPrevSibling(k,xmlNewComment((const xmlChar*)comment.c_str()));
					xmlAddPrevSibling(k,xmlNewText((const xmlChar*)("\n"+indentStr).c_str()));
				}
			}
			it->second->saveXML(v);
			seen.insert(*it);
			prev=cur=xNodeGetNextNode(cur);
		}

		if(seen.size()!=dict.size()) {
			// the main dictionary has entries that weren't seen... find which ones
			// if needed, this could be made faster (O(n) vs. current O(n lg n)) by assuming the maps
			// are sorted and moving two iterators through together instead of repeated find()'s
			for(storage_t::const_iterator it=dict.begin(); it!=dict.end(); ++it) {
				if(seen.find(it->first)==seen.end()) {
					//we didn't see this node in the existing xml tree, have to add a new node pair for it
					bool isSub=dynamic_cast<Collection*>(it->second);
					if(isSub) {
						xmlAddChild(node,xmlNewText((const xmlChar*)"\n"));
						xmlAddChild(node,xmlNewComment((const xmlChar*)("======== "+it->first+" ========").c_str()));
					}
					comments_t::const_iterator cit=comments.find(it->first);
					if(cit!=comments.end()) {
						if(isSub || cit->second.substr(0,it->first.size())==it->first)
							comment=cit->second;
						else
							comment=it->first+": "+cit->second;
						xmlAddChild(node,xmlNewText((const xmlChar*)"\n"));
						xmlAddChild(node,xmlNewComment((const xmlChar*)comment.c_str()));
					}
					xmlAddChild(node,xmlNewText((const xmlChar*)("\n"+indentStr).c_str()));
					xmlNode* k=xmlNewChild(node,NULL,(const xmlChar*)"key",(const xmlChar*)it->first.c_str());
					if(k==NULL)
						throw bad_format(node,"Error: plist Dictionary xml error on saving key");
					xmlAddChild(node,xmlNewText((const xmlChar*)" "));
					xmlNode* v=xmlNewChild(node,NULL,(const xmlChar*)"",NULL);
					if(v==NULL)
						throw bad_format(node,"Error: plist Dictionary xml error on saving value");
					if(indentStr.size()>=perIndent.size())
						xmlAddChild(node,xmlNewText((const xmlChar*)("\n"+indentStr.substr(perIndent.size())).c_str()));
					else
						xmlAddChild(node,xmlNewText((const xmlChar*)("\n")));
					it->second->saveXML(v);
				}
			}
		}
	}

	std::string Dictionary::toString() const {
		stringstream s;
		s << *this;
		return s.str();
	}
	
	//! implements the clone function for Dictionaries
	PLIST_CLONE_IMP(Dictionary,new Dictionary(*this));

	unsigned int Dictionary::getLongestKeyLen() const {
		size_t longest=0;
		size_t seplen=subCollectionSep().size();
		for(Dictionary::const_iterator it=begin(); it!=end(); ++it) {
			size_t cur=it->first.size();
			if(Collection* dp=dynamic_cast<Collection*>(it->second))
				cur+=getLongestKeyLenOther(*dp)+seplen;
			longest=std::max(longest,cur);
		}
		return longest;
	}
	
	Dictionary::iterator Dictionary::getSubEntry(const std::string& name, std::string::size_type& seppos) {
		seppos=name.find(subCollectionSep());
		if(seppos==string::npos)
			return dict.end(); //no '.'s found -- go away
		iterator it=dict.find(name.substr(0,seppos));
		if(it==dict.end())
			return dict.end(); //no entry matching prefix -- go away
		const Collection* d=dynamic_cast<const Collection*>(it->second);
		if(d==NULL)
			return dict.end(); //matching prefix is not a collection -- go away
		return it;
	}
	Dictionary::const_iterator Dictionary::getSubEntry(const std::string& name, std::string::size_type& seppos) const {
		seppos=name.find(subCollectionSep());
		if(seppos==string::npos)
			return dict.end(); //no '.'s found -- go away
		const_iterator it=dict.find(name.substr(0,seppos));
		if(it==dict.end())
			return dict.end(); //no entry matching prefix -- go away
		const Collection* d=dynamic_cast<const Collection*>(it->second);
		if(d==NULL)
			return dict.end(); //matching prefix is not a collection -- go away
		return it;
	}
		
	std::ostream& operator<<(std::ostream& os, const Dictionary& d) {
		unsigned int longest=std::max(Collection::getLongestKeyLenOther(d),static_cast<unsigned int>(os.width()));
		unsigned int seplen=Collection::subCollectionSep().size();
		for(Dictionary::storage_t::const_iterator it=d.dict.begin(); it!=d.dict.end(); ++it) {
			if(Collection* dp=dynamic_cast<Collection*>(it->second)) {
				stringstream ss;
				ss << setw(longest-it->first.size()-seplen) << *dp;
				string line;
				for(getline(ss,line); ss; getline(ss,line))
					os << it->first << Collection::subCollectionSep() << line << endl;
			} else {
				os << left << setw(longest) << it->first << " = " << *it->second << endl;
			}
		}
		return os;
	}
	


	
	void Array::setEntry(size_t index, ObjectBase& val, bool warnExists/*=false*/) {
		if(index==size()) {
			arr.push_back(&val);
			fireEntryAdded(val);
		} else {
			if(arr[index]==&val)
				return;
			if(warnExists) {
				cerr << "Warning: new entry "<<index<<" ("<<val<<") conflicted with previous entry "<<index<<" ("<<(*arr[index])<<")"<<endl;
				cerr << "         (use setEntry(...,false) if you expect you might need to overwrite)" << endl;
			}
			arr[index]=&val;
			fireEntriesChanged();
		}
	}
	void Array::addEntry(size_t index, ObjectBase& val, const std::string& comment/*=""*/) {
		if(index==size()) {
			arr.push_back(&val);
		} else {
			storage_t::iterator it=arr.begin();
			advance(it,index);
			arr.insert(it,&val);
		}
		if(comment.size()>0)
			setComment(index,comment);
		fireEntryAdded(val);
	}
	void Array::setEntry(size_t index, ObjectBase* val, bool warnExists/*=false*/) {
		if(index>size())
			throw bad_format(NULL,"Error: attempted to setEntry() past end of Array");
		else if(index==size()) {
			arr.push_back(val);
			fireEntryAdded(*val);
		} else {
			if(arr[index]==val)
				return;
			if(warnExists) {
				cerr << "Warning: new entry "<<index<<" ("<<val<<") conflicted with previous entry "<<index<<" ("<<(*arr[index])<<")"<<endl;
				cerr << "         (use setEntry(...,false) if you expect you might need to overwrite)" << endl;
			}
			std::set<ObjectBase*>::iterator it=myRef.find(arr[index]);
			if(it!=myRef.end()) {
				myRef.erase(*it);
				delete arr[index];
			}
			arr[index]=val;
			takeObject(index,val);
			fireEntriesChanged();
		}
	}
	void Array::addEntry(size_t index, ObjectBase* val, const std::string& comment/*=""*/) {
		if(index>size())
			throw bad_format(NULL,"Error: attempted to setEntry() past end of Array");
		else if(index==size()) {
			arr.push_back(val);
		} else {
			storage_t::iterator it=arr.begin();
			advance(it,index);
			arr.insert(it,val);
		}
		takeObject(index,val);
		if(comment.size()>0)
			setComment(index,comment);
		fireEntryAdded(*val);
	}
	
	void Array::removeEntry(size_t index) {
		storage_t::iterator it=arr.begin();
		advance(it,index);
		ObjectBase* obj=*it;
		arr.erase(it);
		fireEntryRemoved(*obj);
	}
	

	void Array::setEntry(const std::string& name, ObjectBase& val, bool warnExists/*=false*/) {
		size_t index=getIndex(name);
		if(index>size()) {
			string::size_type p;
			const_iterator it=getSubEntry(name,p);
			if(it!=arr.end()) {
				Collection * d=dynamic_cast<Collection*>(*it);
				d->setEntry(name.substr(p+1),val,warnExists);
			} else
				throw bad_format(NULL,("Array::setEntry(name,val,warn) was called with an invalid numeric name:" + name).c_str());
		} else {
			setEntry(index,val,warnExists);
		}
	}
	void Array::addEntry(const std::string& name, ObjectBase& val, const std::string& comment/*=""*/) {
		size_t index=getIndex(name);
		if(index>size()) {
			string::size_type p;
			const_iterator it=getSubEntry(name,p);
			if(it!=arr.end()) {
				Collection * d=dynamic_cast<Collection*>(*it);
				d->addEntry(name.substr(p+1),val,comment);
			} else
				throw bad_format(NULL,("Array::addEntry(name,val,comment) was called with an invalid numeric name:" + name).c_str());
		} else {
			addEntry(index,val,comment);
		}
	}
	void Array::setEntry(const std::string& name, ObjectBase* val, bool warnExists/*=false*/) {
		size_t index=getIndex(name);
		if(index>size()) {
			string::size_type p;
			const_iterator it=getSubEntry(name,p);
			if(it!=arr.end()) {
				Collection * d=dynamic_cast<Collection*>(*it);
				d->setEntry(name.substr(p+1),val,warnExists);
			} else
				throw bad_format(NULL,("Array::setEntry(name,*val,warn) was called with an invalid numeric name:" + name).c_str());
		} else {
			setEntry(index,val,warnExists);
		}
	}
	void Array::addEntry(const std::string& name, ObjectBase* val, const std::string& comment/*=""*/) {
		size_t index=getIndex(name);
		if(index>size()) {
			string::size_type p;
			const_iterator it=getSubEntry(name,p);
			if(it!=arr.end()) {
				Collection * d=dynamic_cast<Collection*>(*it);
				d->addEntry(name.substr(p+1),val,comment);
			} else
				throw bad_format(NULL,("Array::addEntry(name,*val,comment) was called with an invalid numeric name:" + name).c_str());
		} else {
			addEntry(index,val,comment);
		}
	}
	
	void Array::removeEntry(const std::string& name) {
		size_t index=getIndex(name);
		if(index>size()) {
			string::size_type p;
			iterator it=getSubEntry(name,p);
			if(it!=arr.end()) {
				Collection * d=dynamic_cast<Collection*>(*it);
				d->removeEntry(name.substr(p+1));
			} else
				throw bad_format(NULL,("Array::removeEntry(name) was called with an invalid numeric name:" + name).c_str());
		} else {
			removeEntry(index);
		}
	}
	ObjectBase* Array::getEntry(const std::string& name) const {
		size_t index=getIndex(name);
		if(index>size()) {
			string::size_type p;
			const_iterator it=getSubEntry(name,p);
			if(it!=arr.end()) {
				const Collection * d=dynamic_cast<const Collection*>(*it);
				return d->getEntry(name.substr(p+1));
			} else
				throw bad_format(NULL,("Array::getEntry(name) was called with an invalid numeric name:" + name).c_str());
		} else {
			return &getEntry(index);
		}
	}
		
	void Array::setComment(size_t index, const std::string& comment) {
		if(comment.size()>0)
			comments[index]=comment;
		else
			comments.erase(index);
	}
	
	const std::string& Array::getComment(size_t index) const {
		comments_t::const_iterator it=comments.find(index);
		if(it==comments.end())
			return emptyStr();
		else
			return it->second;
	}
	
	void Array::setComment(const std::string& name, const std::string& comment) {
		size_t index=getIndex(name);
		if(index>size()) {
			string::size_type p;
			const_iterator it=getSubEntry(name,p);
			if(it!=arr.end()) {
				Collection * d=dynamic_cast<Collection*>(*it);
				d->setComment(name.substr(p+1),comment);
			} else
				throw bad_format(NULL,("Array::setComment(name,comment) was called with an invalid numeric name:" + name).c_str());
		} else {
			setComment(index,comment);
		}
	}
	
	const std::string& Array::getComment(const std::string& name) const {
		size_t index=getIndex(name);
		if(index>size()) {
			string::size_type p;
			const_iterator it=getSubEntry(name,p);
			if(it!=arr.end()) {
				const Collection * d=dynamic_cast<const Collection*>(*it);
				return d->getComment(name.substr(p+1));
			} else
				throw bad_format(NULL,("Array::getComment(name) was called with an invalid numeric name:" + name).c_str());
		} else {
			return getComment(index);
		}
	}
	
	void Array::loadXML(xmlNode* node) {
		//check if our node has been set to NULL (invalid or not found)
		if(node==NULL)
			return;
		
		std::string comment;
		unsigned int i=0;
		for(xmlNode* cur = skipToElement(node->children,comment); cur!=NULL; cur = skipToElement(cur->next,comment)) {
			if(!loadXMLNode(i++, cur, comment))
				 break;
		}
		if(trimExtraLoad) {
			while(i<size())
				removeEntry(size()-1);
		} 
	}
	
	void Array::saveXML(xmlNode* node) const {
		//check if our node has been set to NULL (invalid or not found)
		if(node==NULL)
			return;
		
		//set the type of the current node
		xmlNodeSetName(node,(const xmlChar*)"array");
		
		//find the depth of the target node in the xml tree to maintain proper indentation
		std::string perIndent("    ");
		std::string indentStr;
		for(xmlNode* cur=node->parent; cur!=NULL; cur=cur->parent) {
			if((void*)cur==(void*)node->doc) { //if we hit the document node, discount it and we're done
				if(indentStr.size()>0)
					indentStr=indentStr.substr(0,indentStr.size()-perIndent.size());
				break;
			}
			indentStr+=perIndent;
		}
		std::string parentIndent;
		if(indentStr.size()>=perIndent.size())
			parentIndent=indentStr.substr(perIndent.size());
		
		//This will hold any comments found between elements -- if no comment is found, a new one may be added
		std::string comment;
		
		//This will be the index of the item we're loading next
		unsigned int i=0;
		
		//process children nodes
		xmlNode * prev=node->children;
		for(xmlNode* cur = skipToElement(node->children,comment); cur!=NULL; cur = skipToElement(cur,comment)) {
			
			if(i==arr.size()) {
				if(trimExtraSave) {
					while(prev!=NULL) {
						xmlNode* n=prev;
						prev=xNodeGetNextNode(prev);
						xmlUnlinkNode(n);
						xmlFreeNode(n);
					}
				} else {
					if(warnUnused)
						cerr << "Warning: plist::Array ignoring extraneous items in destination during save..." << endl;
				}
				break;
			}
			if(comment.size()==0) {
				comments_t::const_iterator cit=comments.find(i);
				if(cit!=comments.end()) {
					char buf[20];
					unsigned int len;
					snprintf(buf,20,"%u%n",i,&len);
					if(/*isSub ||*/ strncmp(cit->second.c_str(),buf,len)==0)
						comment=cit->second;
					else { //if not a sub-dictionary, and comment doesn't already start with entry name, prepend entry name
						comment=buf;
						comment+=": "+cit->second;
					}
					xmlAddPrevSibling(cur,xmlNewText((const xmlChar*)"\n"));
					xmlAddPrevSibling(cur,xmlNewComment((const xmlChar*)comment.c_str()));
					xmlAddPrevSibling(cur,xmlNewText((const xmlChar*)("\n"+indentStr).c_str()));
				}
			}
			arr[i]->saveXML(cur);
			prev=cur=xNodeGetNextNode(cur);
		}
		
		bool hadUnsaved = (i<arr.size());
		for(; i<arr.size(); ++i) {
			comments_t::const_iterator cit=comments.find(i);
			if(cit!=comments.end()) {
				char buf[20];
				unsigned int len;
				snprintf(buf,20,"%u%n",i,&len);
				if(/*isSub ||*/ strncmp(cit->second.c_str(),buf,len)==0)
					comment=cit->second;
				else { //if not a sub-dictionary, and comment doesn't already start with entry name, prepend entry name
					comment=buf;
					comment+=": "+cit->second;
				}
				xmlAddChild(node,xmlNewText((const xmlChar*)("\n"+parentIndent).c_str()));
				xmlAddChild(node,xmlNewComment((const xmlChar*)comment.c_str()));
			}
			xmlAddChild(node,xmlNewText((const xmlChar*)("\n"+indentStr).c_str()));
			xmlNode* v=xmlNewChild(node,NULL,(const xmlChar*)"",NULL);
			if(v==NULL)
				throw bad_format(node,"Error: plist Array xml error on saving value");
			arr[i]->saveXML(v);
		}
		if(hadUnsaved)
			xmlAddChild(node,xmlNewText((const xmlChar*)("\n"+parentIndent).c_str()));
	}
	
	std::string Array::toString() const {
		stringstream s;
		s << *this;
		return s.str();
	}
	
	//! implements the clone function for Array
	PLIST_CLONE_IMP(Array,new Array(*this));

	size_t Array::getIndex(const std::string& name) const {
		char * endp=0;
		long index=strtol(name.c_str(),&endp,0);
		if(index<0)
			return (size_t)-1;
			//throw bad_format(NULL,"Array passed negative index encoded in string: "+name);
		/*
		if(*endp=='.') {
			const Collection* d=dynamic_cast<const Collection*>(arr[index]);
			if(d==NULL)
				return (size_t)-1; //matching prefix is not a dictionary -- go away
			
			//found a matching sub-dictionary, have it find the rest recursively
			return index;
		}*/
		if(*endp!='\0')
			return (size_t)-1;
		//throw bad_format(NULL,"Array::setEntry(name,val) was called with a non-numeric name");
		return index;
	}
		
	unsigned int Array::getLongestKeyLen() const {
		size_t longest=0;
		size_t seplen=subCollectionSep().size();
		for(size_t i=0; i<size(); ++i) {
			size_t cur=snprintf(NULL,0,"%lu",static_cast<unsigned long>(i));
			if(Collection* dp=dynamic_cast<Collection*>(arr[i]))
				cur+=getLongestKeyLenOther(*dp)+seplen;
			longest=std::max(longest,cur);
		}
		return longest;
	}
	
	Array::iterator Array::getSubEntry(const std::string& name, std::string::size_type& seppos) {
		seppos=name.find(subCollectionSep());
		if(seppos==string::npos)
			return arr.end(); //no '.'s found -- go away
		size_t index=getIndex(name.substr(0,seppos));
		if(index>=size())
			return arr.end(); //no entry matching prefix -- go away
		iterator it=arr.begin();
		advance(it,index);
		const Collection* d=dynamic_cast<const Collection*>(*it);
		if(d==NULL)
			return arr.end(); //matching prefix is not a collection -- go away
		return it;
	}
	Array::const_iterator Array::getSubEntry(const std::string& name, std::string::size_type& seppos) const {
		seppos=name.find(subCollectionSep());
		if(seppos==string::npos)
			return arr.end(); //no '.'s found -- go away
		size_t index=getIndex(name.substr(0,seppos));
		if(index>=size())
			return arr.end(); //no entry matching prefix -- go away
		const_iterator it=arr.begin();
		advance(it,index);
		const Collection* d=dynamic_cast<const Collection*>(*it);
		if(d==NULL)
			return arr.end(); //matching prefix is not a collection -- go away
		return it;
	}
	std::ostream& operator<<(std::ostream& os, const Array& d) {
		unsigned int longest=std::max(Collection::getLongestKeyLenOther(d),static_cast<unsigned int>(os.width()));
		unsigned int seplen=Collection::subCollectionSep().size();
		for(unsigned long i=0; i<d.arr.size(); ++i) {
			if(Collection* dp=dynamic_cast<Collection*>(d.arr[i])) {
				stringstream ss;
				ss << setw(longest-snprintf(NULL,0,"%lu",i)-seplen) << *dp;
				string line;
				for(getline(ss,line); ss; getline(ss,line))
					os << i << Collection::subCollectionSep() << line << endl;
			} else {
				os << left << setw(longest) << i << " = " << *d.arr[i] << endl;
			}
		}//Array::storage_t::const_iterator it=d.arr.begin(); it!=d.arr.end(); ++it) {
		return os;
	}
	
	void Dictionary::clear() {
		for(std::set<ObjectBase*>::iterator it=myRef.begin(); it!=myRef.end(); ++it)
			delete *it;
		myRef.clear();
		dict.clear();
		fireEntriesChanged();
	}
	
	void Dictionary::takeObject(const std::string& /*name*/, ObjectBase* obj) {
		myRef.insert(obj);
	}

	void Dictionary::fireEntryRemoved(ObjectBase& val) {
		Dictionary::fireEntryRemoved(val);
		std::set<ObjectBase*>::iterator it=myRef.find(&val);
		if(it!=myRef.end()) {
			delete &val;
			myRef.erase(it);
		}
	}
	
	void Dictionary::cloneMyRef() {
		for(iterator dit=dict.begin(); dit!=dict.end(); ++dit) {
			std::set<ObjectBase*>::iterator rit=myRef.find(dit->second);
			if(rit!=myRef.end()) {
				myRef.erase(rit);
				myRef.insert(dit->second=dynamic_cast<ObjectBase*>((dit->second)->clone()));
			}
		}
		
		//slower implementation, but can handle multiple pointers to the same instance (which we don't support elsewhere, so no point in doing it)
		/*
		 std::set<ObjectBase*> ns;
		for(std::set<ObjectBase*>::iterator it=myRef.begin(); it!=myRef.end(); ++it) {
			ObjectBase* n=dynamic_cast<ObjectBase*>((*it)->clone());
			bool used=false;
			for(iterator dit=dict.begin(); dit!=dict.end(); ++dit) {
				if(*it==dit->second) {
					dit->second=n;
					used=true;
				}
			}
			if(!used) {
				cerr << "Warning: dictionary claims control over pointer not found in dictionary" << endl;
				delete n;
			} else
				ns.insert(n);
		}
		myRef=ns;
		*/
	}
	
	void Array::clear() {
		for(std::set<ObjectBase*>::iterator it=myRef.begin(); it!=myRef.end(); ++it)
			delete *it;
		myRef.clear();
		arr.clear();
		fireEntriesChanged();
	}
	
	void Array::takeObject(size_t /*index*/, ObjectBase* obj) {
		myRef.insert(obj);
	}
	
	void Array::fireEntryRemoved(ObjectBase& val) {
		Array::fireEntryRemoved(val);
		std::set<ObjectBase*>::iterator it=myRef.find(&val);
		if(it!=myRef.end()) {
			delete &val;
			myRef.erase(it);
		}
	}
	
	void Array::cloneMyRef() {
		for(iterator dit=arr.begin(); dit!=arr.end(); ++dit) {
			std::set<ObjectBase*>::iterator rit=myRef.find(*dit);
			if(rit!=myRef.end()) {
				myRef.erase(rit);
				myRef.insert(*dit=dynamic_cast<ObjectBase*>((*dit)->clone()));
			}
		}
		
		//slower implementation, but can handle multiple pointers to the same instance (which we don't support elsewhere, so no point in doing it)
		/*
		std::set<ObjectBase*> ns;
		for(std::set<ObjectBase*>::iterator it=myRef.begin(); it!=myRef.end(); ++it) {
			ObjectBase* n=dynamic_cast<ObjectBase*>((*it)->clone());
			bool used=false;
			for(iterator dit=arr.begin(); dit!=arr.end(); ++dit) {
				if(*it==*dit) {
					*dit=n;
					used=true;
				}
			}
			if(!used) {
				cerr << "Warning: dictionary claims control over pointer not found in dictionary" << endl;
				delete n;
			} else
				ns.insert(n);
		}
		myRef=ns;
		*/
	}
	
	bool Dictionary::loadXMLNode(const std::string& key, xmlNode* val, const std::string& comment) {
		storage_t::const_iterator it=dict.find(key);
		if(it!=dict.end()) {
			//found pre-existing entry
			try {
				//it's reasonable to assume that in common usage, the same type will be used each time
				//so let's try to load into the current entry as is
				it->second->loadXML(val);
				if(comment.size()>0)
					setComment(key,comment);
				return true;
			} catch(const bad_format& ex) {
				//apparently that didn't work, let's try a fresh load using the polymorphic loader (fall through below)
				removeEntry(key);
			}
		} else if(!trimExtraLoad) {
			if(warnUnused)
				std::cerr << "Warning: reading plist dictionary, key '" << key << "' does not match a registered variable.  Ignoring..." << std::endl;
			return false;
		}
		ObjectBase * obj=plist::loadXML(val);
		if(obj==NULL)
			throw bad_format(val,"Dictionary encountered an unknown value type");
		addEntry(key,obj,comment);
		return true;
	}
	
	bool Array::loadXMLNode(size_t index, xmlNode* val, const std::string& comment) {
		if(index<size()) {
			//have pre-existing entry
			try {
				//it's reasonable to assume that in common usage, the same type will be used each time
				//so let's try to load into the current entry as is
				arr[index]->loadXML(val);
				if(comment.size()>0)
					setComment(index,comment);
				return true;
			} catch(const bad_format& ex) {
				//apparently that didn't work, let's try a fresh load using the polymorphic loader (fall through below)
				removeEntry(index);
			}
		} else if(!trimExtraLoad) {
			if(warnUnused)
				cerr << "Warning: plist::Array ran out of registered items (" << size() << ") during load.  Ignoring extraneous items from source..." << endl;
			return false;
		}
		ObjectBase * obj=plist::loadXML(val);
		if(obj==NULL)
			throw bad_format(val,"Array encountered an unknown value type");
		addEntry(index,obj,comment);
		return true;
	}
	
} //namespace plist


/*! @file
 * @brief 
 * @author Ethan Tira-Thompson (ejt) (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.5 $
 * $State: Exp $
 * $Date: 2006/09/16 06:01:41 $
 */
