#include "ProjectInterface.h"
#include "Wireless/Socket.h"
#include "Vision/SegmentedColorGenerator.h"
#include "debuget.h"
#include <exception>

namespace ProjectInterface {

	bool displayException(const char * file, int line, const char * message, const std::exception* ex) {
		if(file!=NULL) {
			serr->printf("Exception caught at %s:%d => ",debuget::extractFilename(file),line);
		} else {
			serr->printf("Exception => ");
		}
		if(ex!=NULL) {
			serr->printf("'%s'",ex->what());
		} else {
			serr->printf("'%s'","Unknown type");
		}
		if(message!=NULL) {
			serr->printf(" (%s)\n",message);
		} else {
			serr->printf("\n");
		}
#ifndef PLATFORM_APERIOS
		serr->printf("\tWhen running in gdb, try 'catch throw' to break where exceptions are first thrown.\n");
#endif
		return true;
	}
	bool (*uncaughtException)(const char * file, int line, const char * message, const std::exception* ex)=&displayException;

	//! default implementation assigned to lookupColorIndexByName(); checks that #defSegmentedColorGenerator is non-NULL and returns getColorIndex on it
	unsigned int defLookupColorIndexByName(const std::string& name) {
		if(defSegmentedColorGenerator==NULL)
			return -1U;
		return defSegmentedColorGenerator->getColorIndex(name);
	}
	unsigned int (*lookupColorIndexByName)(const std::string& name)=&defLookupColorIndexByName;
	
	//! default value initially assigned to lookupColorIndexByRgb(); checks that #defSegmentedColorGenerator is non-NULL and returns getColorIndex on it
	unsigned int defLookupColorIndexByRgb(const rgb rgbval) {
		if(defSegmentedColorGenerator==NULL)
			return -1U;
		return defSegmentedColorGenerator->getColorIndex(rgbval);
	}
	//! returns color index for color with specified "representitive" RGB color
	unsigned int (*lookupColorIndexByRgb)(const rgb rgbval)=&defLookupColorIndexByRgb;
	
	//! default implementation assigned to lookupColorRGB(); checks that #defSegmentedColorGenerator is non-NULL and returns getColorRGB on it
	rgb defLookupColorRGB(unsigned int index) {
		if(defSegmentedColorGenerator==NULL)
			return rgb();
		return defSegmentedColorGenerator->getColorRGB(index);
	}
	rgb (*lookupColorRGB)(unsigned int index)=&defLookupColorRGB;

	//! default value initially assigned to lookupNumColors(); checks that #defSegmentedColorGenerator is non-NULL and returns getNumColors on it
	unsigned int defLookupNumColors() {
		if ( defSegmentedColorGenerator == NULL ) 
			return -1U; 
		return defSegmentedColorGenerator->getNumColors();
	}
	//! returns the number of indexed colors which are currently defined
	unsigned int (*lookupNumColors)() = &defLookupNumColors;

	//! displays an rgb value in the form '[r,g,b]'
	std::ostream& operator<<(std::ostream &os, const rgb &rgbval) {
		os << "[" << (unsigned int)rgbval.red
		   << "," << (unsigned int)rgbval.green
		   << "," << (unsigned int)rgbval.blue
		   << "]";
		return os;
	}

	//! returns @a rgbval in the form 'r g b'
	std::string toString(const rgb &rgbval) {
		char buff[15];
		snprintf(buff,15,"%d %d %d",rgbval.red,rgbval.green,rgbval.blue);
		return buff;
	}



	//! A collection of the various stages of vision processing.  None of these are absolutely required, but are needed to run included demo behaviors and TekkotsuMon modules
	/*! @name Vision Setup */
	//! pointer to generator
	FilterBankGenerator * defRawCameraGenerator=NULL;
	FilterBankGenerator * defInterleavedYUVGenerator=NULL;
	JPEGGenerator * defColorJPEGGenerator=NULL;
	JPEGGenerator * defGrayscaleJPEGGenerator=NULL;
	PNGGenerator * defColorPNGGenerator=NULL;
	PNGGenerator * defGrayscalePNGGenerator=NULL;
	SegmentedColorGenerator * defSegmentedColorGenerator=NULL;
	RLEGenerator * defRLEGenerator=NULL;
	RegionGenerator * defRegionGenerator=NULL;
	//@}

	//! Default source IDs for the various generators; These are given default values, but you can reassign them if you like.
	/*! @name Vision SIDs */
	//! source id for event
	unsigned int visRawCameraSID=0;

	unsigned int visInterleaveSID=0;

	unsigned int visColorJPEGSID=0;
	unsigned int visGrayscaleJPEGSID=1;

	unsigned int visColorPNGSID=0;
	unsigned int visGrayscalePNGSID=1;

	unsigned int visSegmentSID=0;

	unsigned int visRLESID=0;

	unsigned int visRegionSID=0;

	unsigned int visPinkBallSID=0;
	unsigned int visBlueBallSID=1;
	unsigned int visGreenBallSID=2;
	unsigned int visYellowBallSID=3;
	unsigned int visHandSID=4;
	//@}

	//! Allows you to request a particular layer abstractly - this isn't used by the framework, just a suggestion for clarity
	/*! @name Layer Resolutions */
	unsigned int doubleLayer=5;
	unsigned int fullLayer=4;
	unsigned int halfLayer=3;
	unsigned int quarterLayer=2;
	unsigned int eighthLayer=1;
	unsigned int sixteenthLayer=0;
	//@}

}

/*! @file
 * @brief Provides instantiation of the non-required members of ProjectInterface
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.15 $
 * $State: Exp $
 * $Date: 2006/09/25 23:31:06 $
 */

