#include "Shared/Config.h"
#include "Shared/ProjectInterface.h"
#include "Behaviors/Mon/RawCamBehavior.h"
#include "Vision/RawCameraGenerator.h"
#include "Vision/SegmentedColorGenerator.h"
#include <stdio.h>
#include <string>
#include <cstring>
#include <ctype.h>
#include "Wireless/Socket.h"
#ifdef PLATFORM_APERIOS
#  include <OPENR/OPrimitiveControl.h>
#  include <OPENR/OPENRAPI.h>
#else
#include <sys/param.h>
#include <unistd.h>
typedef unsigned int OSpeakerVolume;
const OSpeakerVolume ospkvolinfdB = 0x8000;
const OSpeakerVolume ospkvol25dB  = 0xe700;
const OSpeakerVolume ospkvol18dB  = 0xee00;
const OSpeakerVolume ospkvol10dB  = 0xf600;
#endif

using namespace std;

Config* config=NULL;

void Config::setFileSystemRoot(const std::string& fsr) {
#ifdef PLATFORM_APERIOS
	if(fsr[0]=='/')
		fsRoot="/ms"+fsr;
	else
		fsRoot="/ms/"+fsr;
#else
	char buf[MAXPATHLEN+2];
	if(getcwd(buf,MAXPATHLEN+2)==NULL)
		perror("Config::setFileSystemRoot(): getcwd");
	buf[MAXPATHLEN+1]='\0';
	std::string sbuf(buf);
	if(sbuf[sbuf.size()-1]!='/' && fsr[0]!='/')
		fsRoot=sbuf+'/'+fsr;
	else if(sbuf[sbuf.size()-1]=='/' && fsr[0]=='/')
		fsRoot=sbuf+fsr.substr(1);
	else
		fsRoot=sbuf+fsr;
#endif
}


std::string Config::portPath(const std::string& path) const {
	if(fsRoot.size()==0)
		return path;
	if(path.size()==0)
		return fsRoot;
	if(path.substr(0,fsRoot.size())==fsRoot)
		return path;
	else if(fsRoot[fsRoot.size()-1]=='/') {
	  if(path[0]!='/')
	    return fsRoot+path;
	  else
	    return fsRoot+path.substr(1);
	} else {
	  if(path[0]!='/')
	    return fsRoot+'/'+path;
	  else
	    return fsRoot+path;
	}
}

void* Config::setValue(section_t section, const char *key, const char *value, bool /*updated*/) {
	switch (section) {
		case sec_wireless:
			if (strncasecmp(key,"id",29)==0) {
				wireless.id=atoi(value);
				return &wireless.id;
			}
			break;
		case sec_vision:
			if (strncasecmp(key,"white_balance",29)==0) {
				if (strncasecmp(value,"indoor",49)==0) {
					vision.white_balance=1;
				} else if (strncasecmp(value,"flourescent",49)==0) {
					std::cout << "Your tekkotsu.cfg file uses deprecated 'flourescent' (misspelled) -- use 'fluorescent' instead" << std::endl;
					vision.white_balance=3;
				} else if (strncasecmp(value,"fluorescent",49)==0) {
					vision.white_balance=3;
				} else if (strncasecmp(value,"outdoor",49)==0) {
					vision.white_balance=2;
				}
#ifdef PLATFORM_APERIOS
				//this will actually send the new setting to the system
				OPrimitiveID fbkID = 0;
				if(OPENR::OpenPrimitive(CameraLocator, &fbkID) != oSUCCESS){
					std::cout << "Open FbkImageSensor failure." << std::endl;
				} else {
					OPrimitiveControl_CameraParam owb(vision.white_balance);
					if(OPENR::ControlPrimitive(fbkID, oprmreqCAM_SET_WHITE_BALANCE, &owb, sizeof(owb), 0, 0) != oSUCCESS){
						std::cout << "CAM_SET_WHITE_BALANCE : Failed!" << std::endl;
					}
					OPENR::ClosePrimitive(fbkID);
				}
#endif
				return &vision.white_balance;
			} else if (strncasecmp(key,"gain",29)==0) {
				if (strncasecmp(value,"low",49)==0) {
					vision.gain=1;
				} else if (strncasecmp(value,"mid",49)==0) {
					vision.gain=2;
				} else if (strncasecmp(value,"high",49)==0) {
					vision.gain=3;
				}
#ifdef PLATFORM_APERIOS
				//this will actually send the new setting to the system
				OPrimitiveID fbkID = 0;
				if(OPENR::OpenPrimitive(CameraLocator, &fbkID) != oSUCCESS){
					std::cout << "Open FbkImageSensor failure." << std::endl;
				} else {
					OPrimitiveControl_CameraParam ogain(vision.gain);
					if(OPENR::ControlPrimitive(fbkID, oprmreqCAM_SET_GAIN, &ogain, sizeof(ogain), 0, 0) != oSUCCESS)
						std::cout << "CAM_SET_GAIN : Failed!" << std::endl;
					OPENR::ClosePrimitive(fbkID);
				}
#endif
				return &vision.gain;
			} else if (strncasecmp(key,"shutter_speed",29)==0) {
				if (strncasecmp(value,"slow",49)==0) {
					vision.shutter_speed=1;
				} else if (strncasecmp(value,"mid",49)==0) {
					vision.shutter_speed=2;
				} else if (strncasecmp(value,"fast",49)==0) {
					vision.shutter_speed=3;
				}
#ifdef PLATFORM_APERIOS
				//this will actually send the new setting to the system
				OPrimitiveID fbkID = 0;
				if(OPENR::OpenPrimitive(CameraLocator, &fbkID) != oSUCCESS){
					std::cout << "Open FbkImageSensor failure." << std::endl;
				} else {
					OPrimitiveControl_CameraParam oshutter(vision.shutter_speed);
					if(OPENR::ControlPrimitive(fbkID,oprmreqCAM_SET_SHUTTER_SPEED, &oshutter, sizeof(oshutter), 0, 0) != oSUCCESS)
						std::cout << "CAM_SET_SHUTTER_SPEED : Failed!" << std::endl;
					OPENR::ClosePrimitive(fbkID);
				}
#endif
				return &vision.shutter_speed;
			} else if (strncasecmp(key,"resolution",29)==0) {
				if (strncasecmp(value,"full",49)==0) {
					vision.resolution=1;
				} else if (strncasecmp(value,"half",49)==0) {
					vision.resolution=2;
				} else if (strncasecmp(value,"quarter",49)==0) {
					vision.resolution=3;
				}
				return &vision.resolution;
			} else if (strncasecmp(key,"thresh",29)==0) {
				vision.thresh.push_back(value);
				return &vision.thresh;
			} else if (strncasecmp(key,"colors",29)==0) {
				strncpy(vision.colors,value,49);
				return &vision.colors;
			} else if (strncasecmp(key,"rawcam_port",29)==0 || strncasecmp(key,"raw_port",29)==0) {
				if(strncasecmp(key,"raw_port",29)==0)
					std::cout << "Your tekkotsu.cfg file uses deprecated raw_port -- use rawcam_port instead" << std::endl;
				vision.rawcam_port=atoi(value);
				return &vision.rawcam_port;
			} else if (strncasecmp(key,"rawcam_transport",29)==0 || strncasecmp(key,"raw_transport",29)==0) {
				if(strncasecmp(key,"raw_transport",29)==0)
					std::cout << "Your tekkotsu.cfg file uses deprecated raw_transport -- use rawcam_transport instead" << std::endl;
				if (strncasecmp(value,"udp",49)==0)
					vision.rawcam_transport=0;
				else if (strncasecmp(value,"tcp",49)==0)
					vision.rawcam_transport=1;
				return &vision.rawcam_transport;
			} else if (strncasecmp(key,"rawcam_interval",29)==0) {
				vision.rawcam_interval=(unsigned int)atoi(value);
				return &vision.rawcam_interval;
			} else if (strncasecmp(key,"segcam_port",29)==0 || strncasecmp(key,"rle_port",29)==0) {
				if(strncasecmp(key,"rle_port",29)==0)
					std::cout << "Your tekkotsu.cfg file uses deprecated rle_port -- use segcam_port instead" << std::endl;
				vision.segcam_port=atoi(value);
				return &vision.segcam_port;
			} else if (strncasecmp(key,"segcam_transport",29)==0 || strncasecmp(key,"rle_transport",29)==0) {
				if(strncasecmp(key,"rle_transport",29)==0)
					std::cout << "Your tekkotsu.cfg file uses deprecated rle_transport -- use segcam_transport instead" << std::endl;
				if (strncasecmp(value,"udp",49)==0)
					vision.segcam_transport=0;
				else if (strncasecmp(value,"tcp",49)==0)
					vision.segcam_transport=1;
				return &vision.segcam_transport;
			} else if (strncasecmp(key,"segcam_interval",29)==0 || strncasecmp(key,"rle_interval",29)==0) {
				if(strncasecmp(key,"rle_interval",29)==0)
					std::cout << "Your tekkotsu.cfg file uses deprecated rle_interval -- use segcam_interval instead" << std::endl;
				vision.segcam_interval=(unsigned int)atoi(value);
				return &vision.segcam_interval;
			} else if (strncasecmp(key,"region_port",29)==0) {
				vision.region_port=atoi(value);
				return &vision.region_port;
			} else if (strncasecmp(key,"region_transport",29)==0) {
				if (strncasecmp(value,"udp",49)==0)
					vision.region_transport=0;
				else if (strncasecmp(value,"tcp",49)==0)
					vision.region_transport=1;
				return &vision.region_transport;
			} else if (strncasecmp(key,"obj_port",29)==0) {
				vision.obj_port=atoi(value);
				return &vision.obj_port;
			} else if (strncasecmp(key,"restore_image",29)==0) {
				vision.restore_image=atoi(value);
				return &vision.restore_image;
			} else if (strncasecmp(key,"region_calc_total",29)==0) {
				vision.region_calc_total=atoi(value);
				return &vision.region_calc_total;
			} else if (strncasecmp(key,"jpeg_dct_method",29)==0) {
				if (strncasecmp(value,"islow",49)==0) {
					vision.jpeg_dct_method=JDCT_ISLOW;
				} else if (strncasecmp(value,"ifast",49)==0) {
					vision.jpeg_dct_method=JDCT_IFAST;
				} else if (strncasecmp(value,"float",49)==0) {
					vision.jpeg_dct_method=JDCT_FLOAT;
				}
				return &vision.jpeg_dct_method;
			} else if (strncasecmp(key,"rawcam_encoding",29)==0) {
				if (strncasecmp(value,"color",49)==0) {
					vision.rawcam_encoding=vision_config::ENCODE_COLOR;
					vision.rawcam_channel=RawCameraGenerator::CHAN_Y;
				} else if (strncasecmp(value,"y_only",49)==0) {
					vision.rawcam_encoding=vision_config::ENCODE_SINGLE_CHANNEL;
					vision.rawcam_channel=RawCameraGenerator::CHAN_Y;
				} else if (strncasecmp(value,"uv_only",49)==0) {
					vision.rawcam_encoding=vision_config::ENCODE_COLOR;
					vision.rawcam_channel=-1;
				} else if (strncasecmp(value,"u_only",49)==0) {
					vision.rawcam_encoding=vision_config::ENCODE_SINGLE_CHANNEL;
					vision.rawcam_channel=RawCameraGenerator::CHAN_U;
				} else if (strncasecmp(value,"v_only",49)==0) {
					vision.rawcam_encoding=vision_config::ENCODE_SINGLE_CHANNEL;
					vision.rawcam_channel=RawCameraGenerator::CHAN_V;
				} else if (strncasecmp(value,"y_dx_only",49)==0) {
					vision.rawcam_encoding=vision_config::ENCODE_SINGLE_CHANNEL;
					vision.rawcam_channel=RawCameraGenerator::CHAN_Y_DX;
				} else if (strncasecmp(value,"y_dy_only",49)==0) {
					vision.rawcam_encoding=vision_config::ENCODE_SINGLE_CHANNEL;
					vision.rawcam_channel=RawCameraGenerator::CHAN_Y_DY;
				} else if (strncasecmp(value,"y_dxdy_only",49)==0) {
					vision.rawcam_encoding=vision_config::ENCODE_SINGLE_CHANNEL;
					vision.rawcam_channel=RawCameraGenerator::CHAN_Y_DXDY;
				}
				return &vision.rawcam_encoding;
			} else if (strncasecmp(key,"rawcam_compression",29)==0) {
				if (strncasecmp(value,"none",49)==0) {
					vision.rawcam_compression=vision_config::COMPRESS_NONE;
				} else if (strncasecmp(value,"jpeg",49)==0) {
					vision.rawcam_compression=vision_config::COMPRESS_JPEG;
				} else if (strncasecmp(value,"png",49)==0) {
					vision.rawcam_compression=vision_config::COMPRESS_PNG;
				}
				return &vision.rawcam_compression;
			} else if (strncasecmp(key,"rawcam_compress_quality",29)==0) {
				vision.rawcam_compress_quality=atoi(value);
				return &vision.rawcam_compress_quality;
			} else if (strncasecmp(key,"rawcam_y_skip",29)==0) {
				vision.rawcam_y_skip=atoi(value);
				if(ProjectInterface::defRawCameraGenerator!=NULL && vision.rawcam_y_skip>(int)ProjectInterface::defRawCameraGenerator->getNumLayers()-1)
					vision.rawcam_y_skip=ProjectInterface::defRawCameraGenerator->getNumLayers()-1;
				return &vision.rawcam_y_skip;
			} else if (strncasecmp(key,"rawcam_uv_skip",29)==0) {
				vision.rawcam_uv_skip=atoi(value);
				if(ProjectInterface::defRawCameraGenerator!=NULL && vision.rawcam_uv_skip>(int)ProjectInterface::defRawCameraGenerator->getNumLayers()-1)
					vision.rawcam_uv_skip=ProjectInterface::defRawCameraGenerator->getNumLayers()-1;
				return &vision.rawcam_uv_skip;
			} else if (strncasecmp(key,"segcam_skip",29)==0 || strncasecmp(key,"rlecam_skip",29)==0) {
				if(strncasecmp(key,"rlecam_skip",29)==0)
					std::cout << "Your tekkotsu.cfg file uses deprecated rlecam_skip -- use segcam_skip instead" << std::endl;
				vision.segcam_skip=atoi(value);
				if(ProjectInterface::defSegmentedColorGenerator!=NULL && vision.segcam_skip>(int)ProjectInterface::defSegmentedColorGenerator->getNumLayers()-1)
					vision.segcam_skip=ProjectInterface::defSegmentedColorGenerator->getNumLayers()-1;
				return &vision.segcam_skip;
			} else if (strncasecmp(key,"segcam_channel",29)==0 || strncasecmp(key,"rlecam_channel",29)==0) {
				if(strncasecmp(key,"rlecam_channel",29)==0)
					std::cout << "Your tekkotsu.cfg file uses deprecated rlecam_channel -- use segcam_channel instead" << std::endl;
				vision.segcam_channel=atoi(value);
				return &vision.segcam_channel;
			} else if (strncasecmp(key,"segcam_compression",29)==0 || strncasecmp(key,"rlecam_compression",29)==0) {
				if(strncasecmp(key,"rlecam_compression",29)==0)
					std::cout << "Your tekkotsu.cfg file uses deprecated rlecam_compression -- use segcam_compression instead" << std::endl;
				if (strncasecmp(value,"none",49)==0) {
					vision.segcam_compression=vision_config::COMPRESS_NONE;
				} else if (strncasecmp(value,"rle",49)==0) {
					vision.segcam_compression=vision_config::COMPRESS_RLE;
				} else if (strncasecmp(value,"png",49)==0) {
					vision.segcam_compression=vision_config::COMPRESS_PNG;
				}
				return &vision.segcam_compression;
			} else if (strncasecmp(key,"regioncam_skip",29)==0) {
				vision.regioncam_skip=atoi(value);
				return &vision.regioncam_skip;
			} else if (strncasecmp(key,"focal_len_x",29)==0) {
				vision.focal_len_x=atof(value);
				return &vision.focal_len_x;
			} else if (strncasecmp(key,"focal_len_y",29)==0) {
				vision.focal_len_y=atof(value);
				return &vision.focal_len_y;
			} else if (strncasecmp(key,"principle_point_x",29)==0) {
				vision.principle_point_x=atof(value);
				return &vision.principle_point_x;
			} else if (strncasecmp(key,"principle_point_y",29)==0) {
				vision.principle_point_y=atof(value);
				return &vision.principle_point_y;
			} else if (strncasecmp(key,"skew",29)==0) {
				vision.skew=atof(value);
				return &vision.skew;
			} else if (strncasecmp(key,"kc1_r2",29)==0) {
				vision.kc1_r2=atof(value);
				return &vision.kc1_r2;
			} else if (strncasecmp(key,"kc2_r4",29)==0) {
				vision.kc2_r4=atof(value);
				return &vision.kc2_r4;
			} else if (strncasecmp(key,"kc5_r6",29)==0) {
				vision.kc5_r6=atof(value);
				return &vision.kc5_r6;
			} else if (strncasecmp(key,"kc3_tan1",29)==0) {
				vision.kc3_tan1=atof(value);
				return &vision.kc3_tan1;
			} else if (strncasecmp(key,"kc4_tan2",29)==0) {
				vision.kc4_tan2=atof(value);
				return &vision.kc4_tan2;
			} else if (strncasecmp(key,"calibration_res_x",29)==0) {
				vision.calibration_res_x=atoi(value);
				return &vision.kc4_tan2;
			} else if (strncasecmp(key,"calibration_res_y",29)==0) {
				vision.calibration_res_y=atoi(value);
				return &vision.calibration_res_y;
			}
			break;
		case sec_main:
			if (strncasecmp(key,"seed_rng",29)==0) {
				main.seed_rng=atoi(value);
				return &main.console_port;
			} else if (strncasecmp(key,"console_port",29)==0) {
				main.console_port=atoi(value);
				return &main.console_port;
			} else if (strncasecmp(key,"consoleMode",29)==0) {
				if (strncasecmp(value,"controller",49)==0) {
					main.consoleMode=main_config::CONTROLLER;
				} else if (strncasecmp(value,"textmsg",49)==0) {
					main.consoleMode=main_config::TEXTMSG;
				} else if (strncasecmp(value,"auto",49)==0) {
					main.consoleMode=main_config::AUTO;
				}
				return &main.consoleMode;
			} else if (strncasecmp(key,"stderr_port",29)==0) {
				main.stderr_port=atoi(value);
				return &main.stderr_port;
			} else if (strncasecmp(key,"error_level",29)==0) {
				main.error_level=atoi(value);
				return &main.error_level;
			} else if (strncasecmp(key,"debug_level",29)==0) {
				main.debug_level=atoi(value);
				return &main.debug_level;
			} else if (strncasecmp(key,"verbose_level",29)==0) {
				main.verbose_level=atoi(value);
				return &main.verbose_level;
			} else if (strncasecmp(key,"wsjoints_port",29)==0) {
				main.wsjoints_port=atoi(value);
				return &main.wsjoints_port;
			} else if (strncasecmp(key,"wspids_port",29)==0) {
				main.wspids_port=atoi(value);
				return &main.wspids_port;
			} else if (strncasecmp(key,"headControl_port",29)==0) {
				main.headControl_port=atoi(value);
				return &main.headControl_port;
			} else if (strncasecmp(key,"walkControl_port",29)==0) {
				main.walkControl_port=atoi(value);
				return &main.walkControl_port;
			} else if (strncasecmp(key,"estopControl_port",29)==0) {
				main.estopControl_port=atoi(value);
				return &main.estopControl_port;
			} else if (strncasecmp(key,"aibo3d_port",29)==0) {
				main.aibo3d_port=atoi(value);
				return &main.aibo3d_port;
			} else if (strncasecmp(key,"wmmonitor_port",29)==0) {
				main.wmmonitor_port=atoi(value);
				return &main.wmmonitor_port;
			} else if (strncasecmp(key,"use_VT100",29)==0) {
				main.use_VT100=extractBool(value);
				return &main.use_VT100;
			}
			break;
		case sec_behaviors:
			if (strncasecmp(key,"flash_bytes",29)==0) {
				behaviors.flash_bytes = atoi(value);
				return &behaviors.flash_bytes;
			} else if (strncasecmp(key,"flash_on_start",29)==0) {
				behaviors.flash_on_start = atoi(value);
				return &behaviors.flash_on_start;
			}
			break;
		case sec_controller:
			if (strncasecmp(key,"gui_port",29)==0) {
				controller.gui_port = atoi(value);
				return &controller.gui_port ;
			} else if (strncasecmp(key,"select_snd",29)==0) {
				strncpy(controller.select_snd,value,49);
				return &controller.select_snd;
			} else if (strncasecmp(key,"next_snd",29)==0) {
				strncpy(controller.next_snd,value,49);
				return &controller.next_snd;
			} else if (strncasecmp(key,"prev_snd",29)==0) {
				strncpy(controller.prev_snd,value,49);
				return &controller.prev_snd;
			} else if (strncasecmp(key,"read_snd",29)==0) {
				strncpy(controller.read_snd,value,49);
				return &controller.read_snd;
			} else if (strncasecmp(key,"cancel_snd",29)==0) {
				strncpy(controller.cancel_snd,value,49);
				return &controller.cancel_snd;
			} else if (strncasecmp(key,"error_snd",29)==0) {
				strncpy(controller.error_snd,value,49);
				return &controller.error_snd;
			}
			break;
		case sec_motion:
			if (strncasecmp(key,"root",29)==0) {
				motion.root=value;
				return &motion.root;
			} else if (strncasecmp(key,"walk",29)==0) {
				motion.walk=value;
				return &motion.walk;
			} else if (strncasecmp(key,"kinematics",29)==0) {
				motion.kinematics=value;
				return &motion.walk;
			} else if (strncasecmp(key,"kinematic_chains",29)==0) {
				motion.kinematic_chains.push_back(value);
				return &motion.kinematic_chains;
			} else if (strncasecmp(key,"calibrate:",10)==0) {
				for(unsigned int i=PIDJointOffset; i<PIDJointOffset+NumPIDJoints; i++)
					if(strncasecmp(&key[10],outputNames[i],outputNameLen+1)==0) { //+1 to ensure full match
						motion.calibration[i-PIDJointOffset] = atof(value);
						return &motion.calibration[i];
					}
						std::cout << "WARNING: Could not match '" << (strlen(key)>10?&key[10]:key) << "' as calibration parameter" << std::endl;
				return NULL;
			} else if (strncasecmp(key,"estop_on_snd",29)==0) {
				strncpy(motion.estop_on_snd,value,49);
				return &motion.estop_on_snd;
			} else if (strncasecmp(key,"estop_off_snd",29)==0) {
				strncpy(motion.estop_off_snd,value,49);
				return &motion.estop_off_snd;
			} else if (strncasecmp(key,"max_head_tilt_speed",29)==0) {
				motion.max_head_tilt_speed=atof(value);
				return &motion.max_head_tilt_speed;
			} else if (strncasecmp(key,"max_head_pan_speed",29)==0) {
				motion.max_head_pan_speed=atof(value);
				return &motion.max_head_pan_speed;
			} else if (strncasecmp(key,"max_head_roll_speed",29)==0) {
				motion.max_head_roll_speed=atof(value);
				return &motion.max_head_roll_speed;
			} else if (strncasecmp(key,"inf_walk_accel",29)==0) {
				motion.inf_walk_accel = atoi(value);
				return &motion.inf_walk_accel;
			} else if (strncasecmp(key,"console_port",29)==0) {
				motion.console_port = atoi(value);
				return &motion.console_port;
			} else if (strncasecmp(key,"stderr_port",29)==0) {
				motion.stderr_port = atoi(value);
				return &motion.stderr_port ;
			}
			break;
		case sec_sound:
			if (strncasecmp(key,"root",29)==0) {
				sound.root=value;
				return &sound.root;
			} else if (strncasecmp(key,"volume",29)==0) {
				if(strncasecmp(value,"mute",49)==0)
					sound.volume=ospkvolinfdB;
				else if(strncasecmp(value,"level_1",49)==0)
					sound.volume=ospkvol25dB;
				else if(strncasecmp(value,"level_2",49)==0)
					sound.volume=ospkvol18dB;
				else if(strncasecmp(value,"level_3",49)==0)
					sound.volume=ospkvol10dB;
				else
					return NULL;
				return &sound.volume;
			} else if (strncasecmp(key,"sample_rate",29)==0) {
				sound.sample_rate = atoi(value);
				return &sound.sample_rate ;
			} else if (strncasecmp(key,"sample_bits",29)==0) {
				sound.sample_bits = atoi(value);
				return &sound.sample_bits ;
			} else if (strncasecmp(key,"preload",29)==0) {
				sound.preload.push_back(value);
				return &sound.preload ;
			} else if (strncasecmp(key,"pitchConfidenceThreshold",29)==0) {
				sound.pitchConfidenceThreshold=atof(value);
				return &sound.pitchConfidenceThreshold ;
			} else if (strncasecmp(key,"streaming.mic_port",29)==0) {
				sound.streaming.mic_port = atoi(value);
				return &sound.streaming.mic_port;
			} else if (strncasecmp(key,"streaming.mic_sample_rate",29)==0) {
				sound.streaming.mic_sample_rate = atoi(value);
				return &sound.streaming.mic_sample_rate;
			} else if (strncasecmp(key,"streaming.mic_bits",29)==0) {
				sound.streaming.mic_sample_bits = atoi(value);
				return &sound.streaming.mic_sample_bits;
			} else if (strncasecmp(key,"streaming.mic_stereo",29)==0) {
				sound.streaming.mic_stereo = extractBool(value);
				return &sound.streaming.mic_stereo;
			} else if (strncasecmp(key,"streaming.speaker_port",29)==0) {
				sound.streaming.speaker_port = atoi(value);
				return &sound.streaming.speaker_port;
			} else if (strncasecmp(key,"streaming.speaker_frame_len",29)==0) {
				sound.streaming.speaker_frame_length = atoi(value);
				return &sound.streaming.speaker_frame_length;
			} else if (strncasecmp(key,"streaming.speaker_max_delay",29)==0) {
				sound.streaming.speaker_max_delay = atoi(value);
				return &sound.streaming.speaker_max_delay;
			}
			break;
		case sec_invalid:
			break;
	}
	return NULL;
}

Config::section_t Config::parseSection(const char* key) {
	if (strncasecmp(key,"wireless",29)==0) {
		return sec_wireless;
	} else if (strncasecmp(key,"vision",29)==0) {
		return sec_vision;
	} else if (strncasecmp(key,"main",29)==0) {
		return sec_main;
	} else if (strncasecmp(key,"behaviors",29)==0) {
		return sec_behaviors;
	} else if (strncasecmp(key,"controller",29)==0) {
		return sec_controller;
	} else if (strncasecmp(key,"motion",29)==0) {
		return sec_motion;
	} else if (strncasecmp(key,"sound",29)==0) {
		return sec_sound;
	} else {
		return sec_invalid;
	}
}

void Config::readConfig(const std::string& filename) {
	FILE* fp = fopen(portPath(filename).c_str(), "r");
	if (fp==NULL)
		return;
	
	char buf[256], key[30], value[50];
	section_t section=sec_invalid;
	bool ignoring=false;
	
	std::vector<std::string> curmodel;
#ifdef PLATFORM_APERIOS
	char rdStr[orobotdesignNAME_MAX + 1];
	memset(rdStr, 0, sizeof(rdStr));
	if (OPENR::GetRobotDesign(rdStr) != oSUCCESS) {
		printf("OPENR::GetRobotDesign() failed.\n");
		rdStr[0]='\0';
	}
#else
#  if TGT_ERS7
	char rdStr[]="ERS-7";
#  elif TGT_ERS210
	char rdStr[]="ERS-210";
#  elif TGT_ERS220
	char rdStr[]="ERS-220";
#  elif TGT_ERS2xx
#    warning "TGT_2xx is not specific for simulation purposes - defaulting to ERS210"
	char rdStr[]="ERS-210";
#  else
#    warning "TGT_<model> undefined - defaulting to ERS7"
	char rdStr[]="ERS-7";
#  endif
#endif
	
	
	unsigned int lineno=0;
	while (fgets(buf,256,fp)!=NULL) {
		lineno++;
		char * p=buf;
		while(*p!='\0' && isspace(*p))
			p++;
		if(*p=='\0' || *p=='#')
			continue; //empty line or comment
		
		if (sscanf(buf,"<%29[^>]>",key)>0) {
			if(key[0]=='/') {
				if(curmodel.size()==0) {
					printf("WARNING: not in a model specific section, line %d\n",lineno);
					continue;
				}
				bool subset=matchNoCase(&key[1],curmodel.back());
				bool superset=matchNoCase(curmodel.back(),&key[1]);
				if(subset && superset) {
					//printf("leaving modelsection %s\n",curmodel.back().c_str());
					curmodel.pop_back();
				} else if(superset) {
					while(curmodel.size()>0) {
						//printf("leaving modelsection %s (==%s)\n",curmodel.back().c_str(),&key[1]);
						curmodel.pop_back();
						if(!matchNoCase(curmodel.back(),&key[1]))
							break;
					}
				} else
					printf("WARNING: config model mismatch, line %d\n",lineno);
				
				ignoring=false; //scan through current model selection stack, see if we're still ignoring
				for(unsigned int i=0; i<curmodel.size(); i++)
					if(!matchNoCase(rdStr,curmodel[i])) {
						ignoring=true;
						break;
					}
						//printf("ignoring==%d\n",ignoring);
						
			} else {
				curmodel.push_back(key);
				//printf("entering section %s\n",curmodel.back().c_str());
				ignoring=ignoring || !matchNoCase(rdStr,key);
				//printf("ignoring==%d\n",ignoring);
			}
		} else if(!ignoring) {
			if (sscanf(buf,"[%29[^]]]",key)>0) {
				section=parseSection(key);
				if(section==sec_invalid)
					std::cerr << "WARNING: Reading " << filename << ", unknown section " << key << " at line number " << lineno << endl;
				//std::cout << "now parsing section " << section << std::endl;
			} else if (sscanf(buf,"%29[^ =] =%49s",key,value)>1) {
				//printf("setValue(%d,'%s','%s');\n",section,key,value);
				if(section!=sec_invalid)
					if(setValue(section, key, value)==NULL)
						std::cerr << "WARNING: Reading " << filename << ", illegal key/value " << key << '/' << value << " at line number " << lineno << endl;
			} else {
				std::cerr << "WARNING: Reading " << filename << ", parse error on key/value " << key << '/' << value << " at line number " << lineno << endl;
			}
		}
	}
		fclose(fp);
}

bool Config::matchNoCase(const std::string& model, const std::string& pattern) {
	unsigned int i=0;
	if(i==pattern.size() && i==model.size())
		return true;
	if(i==pattern.size() || i==model.size())
		return false;
	while(pattern[i]!='*') {
		if(toupper(pattern[i])!=toupper(model[i]))
			return false;
		i++;
		if(i==pattern.size() && i==model.size())
			return true;
		if(i==pattern.size() || i==model.size())
			return false;
	}
	i=pattern.size()-1;
	unsigned int j=model.size()-1;
	while(pattern[i]!='*') {
		if(toupper(pattern[i])!=toupper(model[j]))
			return false;
		i--; j--;
	}
	return true;
}

bool Config::extractBool(const char * value) {
	int i=0;
	while(isspace(value[i])) i++;
	if(strncasecmp(&value[i],"t",29)==0)
		return true;
	else if(strncasecmp(&value[i],"f",29)==0)
		return false;
	else if(strncasecmp(&value[i],"true",29)==0)
		return true;
	else if(strncasecmp(&value[i],"false",29)==0)
		return false;
	else if(strncasecmp(&value[i],"y",29)==0)
		return true;
	else if(strncasecmp(&value[i],"n",29)==0)
		return false;
	else if(strncasecmp(&value[i],"yes",29)==0)
		return true;
	else if(strncasecmp(&value[i],"no",29)==0)
		return false;
	else
		return atoi(value);
}

/*! @file
 * @brief Implements Config, which provides global access to system configuration information
 * @author alokl (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.59 $
 * $State: Exp $
 * $Date: 2006/09/27 20:36:06 $
 */

