#include "EmergencyStopMC.h"
#include "Shared/WorldState.h"
#include "Shared/get_time.h"
#include "Motion/MotionManager.h"
#include "Sound/SoundManager.h"
#include "Shared/Config.h"
#include "Events/EventRouter.h"
#include "Shared/ERS210Info.h"
#include "Shared/ERS220Info.h"
#include "Shared/ERS7Info.h"
#include "Wireless/Wireless.h"

EmergencyStopMC::EmergencyStopMC()
	: PostureMC(), paused(false), stilldown(false), active(true), period(2000),
		timeoflastbtn(0), timeofthisbtn(0), timeoflastfreeze(0), timeoflastrelease(0), duration(600),
		pidcutoff(0.2), ledengine()
{
	for(unsigned int i=0; i<NumPIDJoints; i++)
		piddutyavgs[i]=0;
	if(state->robotDesign&WorldState::ERS210Mask) {
		ledengine.cycle(ERS210Info::TlRedLEDMask,period,1,0,period/2);
		ledengine.cycle(ERS210Info::TlBluLEDMask,period,1);
	} else if(state->robotDesign&WorldState::ERS220Mask) {
		ledengine.cycle(ERS220Info::TailCenterLEDMask, period, 2.0f, -.5f, (int)(period * 0/5.5));
		ledengine.cycle(ERS220Info::TailLeftLEDMask|ERS220Info::TailRightLEDMask,   period, 2.0f, -.5f, (int)(period * 1/5.5));
		ledengine.cycle(ERS220Info::BackLeft3LEDMask|ERS220Info::BackRight1LEDMask, period, 2.0f, -.5f, (int)(period * 2/5.5));
		ledengine.cycle(ERS220Info::BackLeft2LEDMask|ERS220Info::BackRight2LEDMask, period, 2.0f, -.5f, (int)(period * 3/5.5));
		ledengine.cycle(ERS220Info::BackLeft1LEDMask|ERS220Info::BackRight3LEDMask, period, 2.0f, -.5f, (int)(period * 4/5.5));
	} else if(state->robotDesign&WorldState::ERS7Mask) {
		ledengine.cycle(1<<(ERS7Info::MdBackColorLEDOffset-LEDOffset),2*period/3,.15,.15/2-.5,0);
	} else {
		serr->printf("Emergency Stop: unknown model\n");
		ledengine.cycle(1<<(NumLEDs-1),period,1,0,period/2);
		ledengine.cycle(1<<(NumLEDs-2),period,1);
	}
	defaultMaxSpeed(.15);
	takeSnapshot();
}


int EmergencyStopMC::updateOutputs() {
	if(trigger()) {
		if(!stilldown) {
			stilldown=true;
			timeoflastbtn=timeofthisbtn;
			timeofthisbtn=get_time();
			//			cout << "Press " << timeofthisbtn << ' ' << timeoflastbtn << endl;
		}
		//		cout << "Down" << endl;
	} else if(stilldown) {
		//		cout << "Release " << get_time() << endl;
		stilldown=false;
		if((get_time()-timeoflastbtn)<duration)
			setStopped(!paused);
	}
	unsigned int curt=get_time();
	dirty=dirty || (curt<timeoflastrelease);
	if(!paused) {
		if(!dirty)
			return 0;
		if(curt>=timeoflastrelease) {
			for(unsigned int i=LEDOffset; i<LEDOffset+NumLEDs; i++)
				motman->setOutput(this,i,0.f); //blank out LEDs to avoid residual background display
			dirty=false;
			return 0;
		}
		float w = (curt>=timeoflastrelease) ? 0 : (static_cast<float>(timeoflastrelease-curt)/FADE_OUT_TIME);
		for(unsigned int i=0; i<NumOutputs; i++)
			cmds[i].weight=w;
	} else {
		//immediately following a pause, just hold current position at first to prevent twitching if we were in motion
		if(curt-timeoflastfreeze>FrameTime*NumFrames*5) { 
			//once joints have come to rest, respond to outside forces
			for(unsigned int i=0; i<NumPIDJoints; i++) {
				//exponential average of duty cycles to filter out noise
				piddutyavgs[i]=piddutyavgs[i]*.9f+state->pidduties[i]*.1f;
				//reset if there's something significantly out of place (perhaps we're being overridden)
				if(fabsf(state->outputs[PIDJointOffset+i]-cmds[PIDJointOffset+i].value)>.15f) {
					//if(PIDJointOffset+i==LFrLegOffset+RotatorOffset)
					//cout << "resetting from " << cmds[PIDJointOffset+i].value << " to " << state->outputs[PIDJointOffset+i] << endl;
					curPositions[PIDJointOffset+i]=cmds[PIDJointOffset+i].value=state->outputs[PIDJointOffset+i];
					dirty=true;
					targetReached=false;
				}
				//give if there's a force...
				if(fabsf(piddutyavgs[i])>pidcutoff) {
					cmds[PIDJointOffset+i].value-=piddutyavgs[PIDJointOffset+i]; //move in the direction of the force
					dirty=true;
					targetReached=false;
				}
			}
		}
	}
	ledengine.updateLEDs(&cmds[LEDOffset]);
	if(state->robotDesign&WorldState::ERS7Mask) {
		//a special Battlestar Galactica inspired effect for the ERS-7
		static float acts[5];
		static bool wasPaused=false;
		if(!wasPaused && paused) {
			for(int i=0; i<5; i++)
				acts[i]=0;
			wasPaused=paused;
		}
		float t=curt;
		t/=period;
		t=(((int)t)&1)?(int)t+1-t:(t-(int)t);
		t*=8;
		const float invsigma=-6;
		const float gamma=.83;
		const float amp=.5;
		float imp[5];
		// w is used to fade out LEDs when releasing estop
		float w = (paused || curt>=timeoflastrelease) ? 1 : (static_cast<float>(timeoflastrelease-curt)/FADE_OUT_TIME);
		for(int i=0; i<5; i++) {
			imp[i]=exp(invsigma*(t-i-2)*(t-i-2))*w;
			acts[i]+=amp*imp[i];
			acts[i]*=gamma*w;
		}
		cmds[ERS7Info::FaceLEDPanelOffset+ 6].value=acts[0]/2+imp[0];
		cmds[ERS7Info::FaceLEDPanelOffset+ 8].value=acts[1]/2+imp[1];
		cmds[ERS7Info::FaceLEDPanelOffset+10].value=acts[2]/2+imp[2];
		cmds[ERS7Info::FaceLEDPanelOffset+ 9].value=acts[3]/2+imp[3];
		cmds[ERS7Info::FaceLEDPanelOffset+ 7].value=acts[4]/2+imp[4];
	}
	int changed=PostureMC::updateOutputs();
	dirty=(curt<timeoflastrelease);
	return changed;
}

void EmergencyStopMC::setActive(bool a) {
	if(paused) {
		if(!a && active)
			releaseJoints();
		else if(a && !active)
			freezeJoints();
	}
	active=a;
}


void EmergencyStopMC::setStopped(bool p, bool sound) {
	if(p!=paused) {
		paused=p;
		if(active) {
			if(paused) {
				freezeJoints();
				if(sound)
					sndman->playFile(config->motion.estop_on_snd);
				cout << "*** PAUSED ***" << endl;
			} else {
				releaseJoints();
				if(sound)
					sndman->playFile(config->motion.estop_off_snd);
				cout << "*** UNPAUSED ***" << endl;
			}
		}
	}
}

void EmergencyStopMC::freezeJoints() {
	dirty=true;
	targetReached=false;
	for(unsigned int i=0; i<NumOutputs; i++) {
		OutputCmd c=motman->getOutputCmd(i);
		curPositions[i]=cmds[i].value = (c.weight==0) ? state->outputs[i] : c.value;
	}
	for(unsigned int i=0; i<NumPIDJoints; i++)
		piddutyavgs[i]=0; //or: state->pidduties[i];
	for(unsigned int i=0; i<LEDOffset; i++)
		cmds[i].weight=1;
	for(unsigned int i=LEDOffset; i<LEDOffset+NumLEDs; i++)
		cmds[i].unset(); // let other commands' LEDs "show through"
	for(unsigned int i=LEDOffset+NumLEDs; i<NumOutputs; i++)
		cmds[i].weight=1;
	if(state->robotDesign&WorldState::ERS210Mask) {
		cmds[ERS210Info::TlRedLEDOffset].set(0,.5);
		cmds[ERS210Info::TlBluLEDOffset].set(0,.5);
	} else if(state->robotDesign&WorldState::ERS220Mask) {
		for(unsigned int i = 0; i < NumLEDs; i++)
			if((ERS220Info::TailLEDMask|ERS220Info::BackLEDMask) & (1 << i))
				cmds[LEDOffset + i].set(0, .5);
	} else if(state->robotDesign&WorldState::ERS7Mask) {
		cmds[ERS7Info::MdBackColorLEDOffset].set(0,.5);
		for(int i=6; i<6+5; i++)
			cmds[ERS7Info::FaceLEDPanelOffset+i].set(0,0.5);
	} else {
		cmds[LEDOffset+NumLEDs-1].set(0,.5);
		cmds[LEDOffset+NumLEDs-2].set(0,.5);
	}
	postEvent(EventBase(EventBase::estopEGID,getID(),EventBase::activateETID,0));
	timeoflastfreeze=get_time();
}

void EmergencyStopMC::releaseJoints() {
	dirty=true;
	targetReached=false;
	unsigned int curt=get_time();
	timeoflastrelease=curt+FADE_OUT_TIME;
	postEvent(EventBase(EventBase::estopEGID,getID(),EventBase::deactivateETID,curt-timeoflastfreeze));
}

bool EmergencyStopMC::trigger() {
	WorldState * st=WorldState::getCurrent(); // this is need because trigger is a static, so it doesn't access the Motion
	if(st->robotDesign&WorldState::ERS210Mask)
		return st->button_times[ERS210Info::BackButOffset];
	if(st->robotDesign&WorldState::ERS220Mask)
		return st->button_times[ERS220Info::BackButOffset];
	if(st->robotDesign&WorldState::ERS7Mask)
		return st->button_times[ERS7Info::FrontBackButOffset]+st->button_times[ERS7Info::MiddleBackButOffset]+st->button_times[ERS7Info::RearBackButOffset];
	serr->printf("EmergencyStopMC: unsupported model!\n");
	return false;
}

/*! @file
 * @brief Implements EmergencyStopMC, overrides all joints, allows modelling, blinks tail
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.30 $
 * $State: Exp $
 * $Date: 2006/09/18 18:08:01 $
 */

