//-*-c++-*-
#ifndef INCLUDED_Thread_h_
#define INCLUDED_Thread_h_

#ifdef PLATFORM_APERIOS
#  warning Thread class is not Aperios compatable
#else

#include "Shared/Resource.h"
#include <stddef.h>

//! provides Thread related data structures
namespace ThreadNS {
	//! an inter-thread lock -- doesn't work across processes, only threads within a process.  (see MutexLock for inter-process locks)
	class Lock : public Resource {
	public:
		Lock(); //!< constructor
		//explicit Lock(const Lock& l); //!< copy constructor -- shallow copy, share a lock, is handy for locking over a scope!!! (lock is automatically obtained on copy -- to avoid autolock, pass false to the two-argument constructor: Lock(const Lock& l, bool autolock) )
		//Lock(const Lock& l, bool autolock); //!< copy constructor -- shallow copy, share a lock, is handy for locking over a scope!!!
		//Lock& operator=(const Lock& l); //!< assignment -- dereference (and release) any previous lock, take on the new storage (shallow copy!)
		~Lock(); //!< destructor -- dereference and release (if any references remain)
		void lock(); //!< block until lock is obtained
		bool trylock(); //!< see if lock is available
		void unlock(); //!< release lock, if held
		unsigned int getInstanceLockLevel() const { return locklevel; } //!< returns the lock level of the local instance of Lock (as opposed to the lock storage structure, which might be shared with other Lock instances)
		unsigned int getLockLevel() const; //!< returns the lock level of the lock storage itself, the sum of all instance's lock levels
	protected:
		friend class MarkScope;
		virtual void useResource(Resource::Data&) { lock(); }
		virtual void releaseResource(Resource::Data&) { unlock(); }
		
		class LockStorage; //!< this internal class will hold the system-dependent lock information
		static LockStorage* glock; //!< The global lock to protect Locks sharing #mylock's
		LockStorage* mylock; //!< This lock's implementation
		static void setup(); //!< creates a new #glock if it is currently NULL (should be called by the Lock() constructor)
		unsigned int locklevel; //!< the current lock level from this Lock, may differ from #mylock's lock level if several Locks are sharing a storage!
	private:
		Lock(const Lock& l); //!< don't call
		Lock& operator=(const Lock& l); //!< don't call
	};	
}	

//! Provides a nice wrapping of pthreads library
/*! If you need to provide cleanup functions on stop(), cancelled(), etc., you
 *  should override the destructor to stop and join so that you can be assured
 *  that your cleanup will be called if the thread is auto-destructed by going out of scope */
class Thread {
public:
	typedef ThreadNS::Lock Lock; //!< shorthand for pthread lock wrapper

	Thread(); //!< constructor, does not start thread by itself (although subclasses may)
	virtual ~Thread()=0; //!< destructor, will stop and join the thread, but you should override it to do the same if you provide any cleanup functions
	
	//! requests that the thread be started, if not already running (you need to create a separate instances if you want to run multiple copies)
	virtual void start();
	
	//! requests that the thread be stopped gracefully, if running.
	/*! A cancel flag is sent, and the thread will be stopped at next cancel point, defined
	 *  by whenever testCancel(), or a set of other system functions, are called.
	 *  See your system's pthread_testcancel() manual page for a list of cancel points.
	 *  @see pushNoCancel(), popNoCancel() */
	virtual void stop();
	
	//! sends a SIGUSR1 to the thread, breaking its execution, but still allowing handle_exit (and thus cancelled()) to be called.
	/*! Beware if your thread uses mutual exclusion locks, this can cause the thread to terminate while still holding locks */
	virtual void kill();
	
	//! detaches thread and sends SIGSTOP, which immediately halts the thread without any chance for cleanup
	/*! Beware if your thread uses mutual exclusion locks, this @b will cause the thread to terminate while still holding locks. */
	virtual void murder();
	
	//! sends a signal to the thread
	virtual void sendSignal(int sig);
	
	//! blocks calling thread until this Thread has terminated, via one means or another; return value is final return value by the thread
	virtual void * join();
	
	//! indicates whether start() has been previously called
	virtual bool isRunning() const { return running; }
	
	//! returns the Thread object for the current thread (or NULL for the main thread)
	static Thread* getCurrent() ;
	
	//! should be called before any threads are created to allow some global thread-specific data to be set up
	static void initMainThread();
	//! should be called if you no longer expect to have any threads in use
	static void releaseMainThread();

	//! should be called whenever a critical section has been entered (i.e. mutex obtained) -- prevents cancel from occurring until popNoCancel() is called
	static void pushNoCancel();
	//! should be called whenever a critical section is left (i.e. mutex released) -- if it was the last one, tests cancellability as well
	static void popNoCancel();
	
protected:
	//! called by launch() when thread is first entered, return false to cancel launch (set #returnValue as well if you care)
	virtual bool launched() { return true; }
	//! called by launch() once the thread has been set up; when this returns, the thread ends, see runloop()
	/*! Default implementation repeatedly calls runloop(), usleep(), and testCancel().
	 *  If you override, you should also be sure to call testCancel occasionally in order to support stop()
	 *  If function returns a value, that value overrides #returnValue.  If cancel occurs, #returnValue is used. */
	virtual void * run();
	//! override this as a convenient way to define your thread -- return the number of *micro*seconds to sleep before the next call; return -1U to indicate end of processing
	virtual unsigned int runloop() { return -1U; }
	//! called when handle_exit() is triggered, either by the thread being cancelled, or when run() has returned voluntarily
	virtual void cancelled() {}
	
	//! checks to see if stop() has been called, and if so, will exit the thread (passing through handle_exit() first)
	virtual void testCancel();
	//! thread entry point -- calls launched() on the thread (as indicated by @a msg), and then run()
	static void * launch(void * msg);
	//! indicates kill() has been called (or SIGUSR1 was sent from some other source) while launch() was still running
	static void handle_launch_signal(int sig);
	//! indicates kill() has been called (or SIGUSR1 was sent from some other source)
	static void handle_signal(int sig);
	//! indicates the thread is exiting, either voluntary (run() returned), stop(), or kill() -- calls cancelled() for the thread as indicated by @a th
	static void handle_exit(void * th);

	//! emit a warning that the last thread exited while the self-pointer thread-specific key still exists (need to call releaseMainThread() or handle_exit())
	static void warnSelfUndestructed(void* msg);

	//! stores the actual pthread data fields
	struct Threadstorage_t * pt;
	//! set to true once start has been called, set back to false by handle_exit(), or by murder() itself
	bool running;
	//! indicates the value to be returned by the thread entry point (and thus passed back to join()) -- set this in runloop() or launched(), overridden by run()'s return value
	void * returnValue;
	//! depth of the pushNoCancel() stack
	unsigned int noCancelDepth;
	//! cancel status at root of no-cancel stack (may be no-cancel through and through)
	int cancelOrig;

private:
	Thread(const Thread& r); //!< don't call, not a well defined operation
	Thread& operator=(const Thread& r); //!< don't call, not a well defined operation
};

#endif //Aperios check

#endif

/*! @file
* @brief Describes the Thread class and its AutoThread templated subclass
* @author ejt (Creator)
*
* $Author: ejt $
* $Name: tekkotsu-3_0 $
* $Revision: 1.12 $
* $State: Exp $
* $Date: 2006/06/16 21:49:23 $
*/

