//-*-c++-*-
#ifdef PLATFORM_APERIOS
#  include <OPENR/RCRegion.h>
#else
#  ifndef INCLUDED_RCRegion_h_
#  define INCLUDED_RCRegion_h_

#include "Shared/ReferenceCounter.h"
#include "MutexLock.h"
#include "ProcessID.h"
#include "Shared/plist.h"
#include <sys/types.h>
#include <map>
#include <exception>

//do you want to use the SysV shared memory call interface
//or the POSIX shared memory interface?  They have different
//strengths/weaknesses on different platforms... :-/
#ifndef POSIX_SHM
#  define POSIX_SHM 1
#endif
#ifndef SYSV_SHM
#  define SYSV_SHM 2
#endif
#ifndef TEKKOTSU_SHM_STYLE
#  define TEKKOTSU_SHM_STYLE POSIX_SHM
#endif

#if TEKKOTSU_SHM_STYLE!=SYSV_SHM && TEKKOTSU_SHM_STYLE!=POSIX_SHM
#  error Unknown TEKKOTSU_SHM_STYLE setting
#endif

namespace ThreadNS {
	class Lock;
}

//! provides compatability with the OPEN-R type of the same name
class RCRegion {
public:

	// The type of region Identifier depends on the style of shared memory being used

#if TEKKOTSU_SHM_STYLE==SYSV_SHM
	//! contains all information needed to attach this region from a different process
	struct Identifier {
		Identifier() : key(), shmid(), size(0) {}
		key_t key; //!< key_t is defined by system headers, contains system region info
		int shmid; //!< an integer key value which identifies the region
		size_t size; //!< the size of the region
	};

#elif TEKKOTSU_SHM_STYLE==POSIX_SHM
	//! maximum guaranteed length for users' region names (might have a little leeway depending on process ID prefix or tmp path prefix)
	static const unsigned int MAX_NAME_LEN=64;

	//! contains all information needed to attach this region from a different process
	struct Identifier {
		Identifier() : size(0) {}
		char key[MAX_NAME_LEN]; //!< a string name for the key
		size_t size; //!< size of the region
	};

#  ifndef USE_UNBACKED_SHM
	static plist::Primitive<std::string> shmRoot; //!< determines location of the file backing within file system
#  endif
	static plist::Primitive<bool> useUniqueMemoryRegions; //!< if true, prefixes region names with #rootPID
	static pid_t rootPID; //!< this is the pid of the original process, used for unique names of memory regions; pid_t is from sys/types.h
#endif


	// The constructors, offering either an Aperios-compatable version,
	// or a linux-specific version offering explicit control over the
	// key value, aids better debugging

#if TEKKOTSU_SHM_STYLE==SYSV_SHM
	//! constructor (OPEN-R compatability)
	explicit RCRegion(size_t sz)
		: id(), base(NULL), references(NULL), lock(NULL)
	{ init(sz,nextKey,true); }
	//! constructor, name isn't used for sysv-style shared memory (not OPEN-R compatable)
	/*! could hash the name to generate key...? */
	RCRegion(const std::string&, size_t sz)
		: id(), base(NULL), references(NULL), lock(NULL)
	{ init(sz,nextKey,true); }

#elif TEKKOTSU_SHM_STYLE==POSIX_SHM
	//! constructor (OPEN-R compatability, name is autogenerated)
	explicit RCRegion(size_t sz)
		: id(), base(NULL), references(NULL), lock(NULL)
	{
		char name[RCRegion::MAX_NAME_LEN];
		snprintf(name,RCRegion::MAX_NAME_LEN,"Rgn.%d.%u",ProcessID::getID(),static_cast<unsigned int>(++nextKey));
		name[RCRegion::MAX_NAME_LEN-1]='\0';
		init(sz,name,true);
	}
	//! constructor, specify your own name for better debugging accountability (not OPEN-R compatable)
	RCRegion(const std::string& name, size_t sz)
		: id(), base(NULL), references(NULL), lock(NULL)
	{ init(sz,name,true); }
#endif

	//! requests that a specified RCRegion be loaded into the current process's memory space
	static RCRegion * attach(const Identifier& rid);
	
	char * Base() const { return base; } //!< the pointer to the shared memory region
	size_t Size() const { return id.size; } //!< the size of the shared memory region
	static void setNextKey(key_t k) { nextKey=k; } //!< sets the next key to be used for automatic assignment to new regions
	static key_t getNextKey() { return nextKey+1; } //!< return the next region serial number -- doesn't actually increment it though, repeated calls will return the same value until the value is actually used
	const Identifier& ID() const { return id; } //!< returns the identifier of this region
	
	int NumberOfReference() const { return references[ProcessID::NumProcesses]; } //!< number of total references to this region, total of all processes
	int NumberOfLocalReference() const { return references[ProcessID::getID()]; } //!< number of references to the region from the current process
	void AddReference(); //!< adds a reference from the current process
	void RemoveReference(); //!< removes a reference from the current process
	void AddSharedReference(); //!< adds a reference which is held by another shared memory region
	void RemoveSharedReference(); //!< removes a reference which is held by another shared memory region
	
	static void aboutToFork(ProcessID::ProcessID_t newID); //!< does housekeeping to mark the region as attached and the same number of references in the new process as well
	static void faultShutdown(); //!< try to unload all regions in a clean manner

#if TEKKOTSU_SHM_STYLE==SYSV_SHM
	//! a map from the shared memory key type to the actual region structure
	typedef std::map<key_t,RCRegion*> attachedRegions_t;
#elif TEKKOTSU_SHM_STYLE==POSIX_SHM
	//! a map from the shared memory key type to the actual region structure
	typedef std::map<std::string,RCRegion*> attachedRegions_t;
#endif
	static unsigned int NumberOfAttach() { return attachedRegions.size(); } //!< returns the number of regions which are currently attached in the process
	
	//! Returns an iterator to the beginning of the attached regions mapping -- it->first is the key, it->second is the RCRegion*
	/*! If you need thread-safety (i.e. another thread may attach/detach while you are iterating), pass true, and be sure to use attachedAdvance() to increment the iterator!
	 *  This doesn't prevent other threads from attaching/detaching regions, it only prevents detaching the one you're on.
	 *  When you're done with a thread-safe iterator, either attachedAdvance() it off the end, or manually call RemoveReference() on the iterator's final region */
	static attachedRegions_t::const_iterator attachedBegin(bool threadSafe);
	//! Returns an iterator to the end of the attached regions -- it->first is the key, it->second is the RCRegion*
	/*! If you need thread-safety (i.e. another thread may attach/detach while you are iterating), be sure to use attachedAdvance() to decrement the iterator!
	 *  This doesn't prevent other threads from attaching/detaching regions, it only prevents detaching the one you're on. */
	static attachedRegions_t::const_iterator attachedEnd();
	//! Increments the attached region iterator in a thread-safe way -- only use this if you previously passed 'true' to begin(), or are decrementing from end()
	/*! If you are using an iterator obtained without thread-safety, just increment it normally -- don't switch to this or it will screw up reference counting.
	 *  If you insist on switching back and forth between thread-safe advance (this function) and normal iterator advancing, you will need to add a reference to the current iterator's region before calling this.
	 *  When you're done, either advance off the end, or manually call RemoveReference() on the iterator's final region */
	static void attachedAdvance(attachedRegions_t::const_iterator& it, int x=1);
	
	//! returns the region's MutexLock
	MutexLock<ProcessID::NumProcesses>& getLock() const { return *lock; }
	
	//! Different methods of handling regions with conflicting keys
	enum ConflictResolutionStrategy {
		RENAME,  //!< try another key until we find one that works (better for SYSV, maybe not so smart for POSIX)
		REPLACE, //!< delete the other region and try again (better for POSIX, maybe not so smart for SYSV)
		EXIT //!< go home and cry about it
	};
	
	static void setConflictResolution(ConflictResolutionStrategy crs) { conflictStrategy=crs; } //!< sets #conflictStrategy
	static ConflictResolutionStrategy getConflictResolution() { return conflictStrategy; } //!< returns #conflictStrategy

	
protected:
	//! this protected constructor is used for attaching regions previously created by another process (see attach())
	RCRegion(const Identifier& rid)
		: id(), base(NULL), references(NULL), lock(NULL)
	{ init(rid.size,rid.key,false); }

	~RCRegion(); //!< prevents stack allocation -- needs to be heap allocated and reference counted

	//! the alignment multiple of the extra space at the end of the region
	static const unsigned int align=sizeof(unsigned int);
	//! the amount of space to leave at the end of the region for housekeeping (reference count and mutex lock)
	static const unsigned int extra=sizeof(unsigned int)*(ProcessID::NumProcesses+1)
	                                +sizeof(MutexLock<ProcessID::NumProcesses>);
	//! returns the size of the region to be allocated, given the size requested by the client
	static unsigned int calcRealSize(unsigned int size);

	//! intializes and returns #staticLock
	static ThreadNS::Lock& getStaticLock();

#if TEKKOTSU_SHM_STYLE==SYSV_SHM
	//! initializes the region's information, either creating a new shared memory region or attempting to connect to a pre-existing one
	void init(size_t sz, key_t sug_key, bool create);
#elif TEKKOTSU_SHM_STYLE==POSIX_SHM
	//! returns the qualified version of this region's key (see getQualifiedName(const std::string& key) )
	std::string getQualifiedName() const { return getQualifiedName(id.key); }
	//! wraps the region's key with a root path prefix and optionally a PID suffix (see #useUniqueMemoryRegions and #shmRoot)
	static std::string getQualifiedName(const std::string& key);
	//! opens a region either in "pure" shared memory, or in file-backed shared memory, based on whether USE_UNBACKED_SHM is defined
	int openRegion(int mode) const;
	//! unlinks a region either in "pure" shared memory, or in file-backed shared memory, based on whether USE_UNBACKED_SHM is defined
	bool unlinkRegion() const;
	//! initializes the region's information, either creating a new shared memory region or attempting to connect to a pre-existing one
	void init(size_t sz, const std::string& name, bool create);
#endif

	//! controls what to do about creating a region with a conflicting key (i.e. another region already exists with the same key)
	static ConflictResolutionStrategy conflictStrategy;
	//! set to true if we are shutting down because of an error, and trying to unload shared regions to avoid leaking beyond program scope
	static bool isFaultShutdown;
									  
	static ThreadNS::Lock* staticLock; //!< a lock over all static RCRegion members for the current process

	static key_t nextKey; //!< serial number of next key -- starts at 1024 for TEKKOTSU_SHM_STYLE==SYSV_SHM, 0 for POSIX_SHM
	static attachedRegions_t attachedRegions; //!< a mapping of key values to RCRegion pointers of the attached region
	
	Identifier id; //!< key values for the region, namely the system key type (either an integer or string depending on TEKKOTSU_SHM_STYLE) and the size of the region
	char * base; //!< pointer to the region's user data
	unsigned int * references; //!< pointer to the per-process reference counts (stored within the shared region!)
	MutexLock<ProcessID::NumProcesses> * lock; //!< region's inter-process lock (stored within the shared region!)

private:
	RCRegion(const RCRegion& r); //!< don't call
	RCRegion& operator=(const RCRegion& r); //!< don't call
};

/*! @file
 * @brief Describes RCRegion, which provides compatability with the OPEN-R type of the same name
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.5 $
 * $State: Exp $
 * $Date: 2006/08/23 19:24:20 $
 */

#  endif
#endif
