#ifndef PLATFORM_APERIOS
#include "PollThread.h"
#include <errno.h>
#include <signal.h>

//better to put this here instead of the header
using namespace std; 

void PollThread::start() {
	launching=initialPoll=true;
	startTime.Set();
	Thread::start();
}

void PollThread::stop() {
	Thread::stop();
	interrupt(); // break thread out of any long sleep commands
}

void PollThread::interrupt() {
	if(launching) //can't interrupt before thread has been launched!
		return;
	if(signal(SIGALRM,handleInterrupt)==SIG_ERR)
		perror("PollThread::run(): initial signal()");
	sendSignal(SIGALRM);
}

bool PollThread::poll() {
	unsigned int sleeptime=runloop();
	if(sleeptime==-1U)
		return false;
	period.Set(static_cast<long>(sleeptime));
	return true;
}

void PollThread::interrupted() {
	if(!initialPoll) {
		if(period-startTime.Age()<1L) {
			delay=0L;
		} else {
			while(!(period-startTime.Age()<1L))
				startTime-=period;
			startTime+=period;
			delay=period;
		}
	}
}

void * PollThread::run() {
	launching=false;
	timespec sleepSpec,remainSpec;
	if(startTime.Age()<delay) {
		sleepSpec=delay-startTime.Age();
		while(nanosleep(&sleepSpec,&remainSpec)) {
			if(errno!=EINTR) {
				perror("PollThread::run(): initial nanosleep");
				break;
			}
			//interrupted() may have changed delay to indicate remaining sleep time
			if(delay<1L || delay<startTime.Age())
				break;
			sleepSpec=delay-startTime.Age();
		}
	}
	testCancel();
	if(!poll())
		return returnValue;
	initialPoll=false;
	bool wasInterrupted=true; // need to add delay instead of period for the first time, same as if an interrupt occurred
	for(;;) {
		bool noSleep=false;
		if(TimeET(0,0)<period) {
			if(trackPollTime) {
				if(wasInterrupted) {
					if(delay<0L || startTime.Age()<delay) {}
					else
						startTime+=delay;
				} else {
					if(startTime.Age()<period) {}
					else
						startTime+=period;
				}
				//the idea with this part is that if we get behind, (because of poll() taking longer than period)
				// then we want to quick poll again as fast as we can to catch up, but we don't want to backlog
				// such that we'll be quick-calling multiple times once we do catch up
				if(period<startTime.Age()) {
					noSleep=true;
					while(period<startTime.Age())
						startTime+=period;
					startTime-=period; //back off one -- the amount we've overshot counts towards the next period
				}
				sleepSpec=period-startTime.Age();
			} else {
				sleepSpec=period;
				startTime.Set();
			}
			wasInterrupted=false;
			while(!noSleep && nanosleep(&sleepSpec,&remainSpec)) {
				wasInterrupted=true;
				if(errno!=EINTR) {
					perror("PollThread::run(): periodic nanosleep");
					break;
				}
				//interrupted() should have changed delay and/or period to indicate remaining sleep time
				if(delay<1L || delay<startTime.Age())
					break;
				sleepSpec=delay-startTime.Age();
			}
		} else {
			startTime.Set();
		}
		testCancel();
		if(!poll())
			return returnValue;
	}
	// this return is just to satisfy warnings with silly compiler
	return returnValue; //never happens -- cancel or bad poll would exit
}

//void PollThread::cancelled() {
	//signal handlers are global, not per-thread, so if we reset the handler, then the next interrupt will cause the program to end :(
	//signal(SIGALRM,SIG_DFL);
//}

void PollThread::handleInterrupt(int /*signal*/) {
	if(signal(SIGALRM,SIG_DFL)==SIG_ERR)
		perror("PollThread::handleInterrupt(): could not re-enable signal");
	PollThread * cur=dynamic_cast<PollThread*>(Thread::getCurrent());
	if(cur==NULL) {
		std::cerr << "PollThread::handleInterrupt called from non-PollThread" << endl;
		return;
	}
	if(cur->noCancelDepth==0)
		cur->testCancel();
	cur->interrupted();
}



/*! @file
 * @brief 
 * @author Ethan Tira-Thompson (ejt) (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.10 $
 * $State: Exp $
 * $Date: 2006/08/07 21:55:27 $
 */
#endif
