//-*-c++-*-
#ifndef INCLUDED_LookoutEvents_h_
#define INCLUDED_LookoutEvents_h_

#include <iostream>

#include "EventBase.h"
#include "DualCoding/LookoutRequests.h"
#include "DualCoding/Sketch.h"
#include "Shared/newmat/newmat.h"


//! Abstract base class for all Lookout Events
class LookoutEvent : public EventBase {
public:
  enum LookoutEventType_t { lookAt, sketch, ir, scan };
  virtual LookoutEventType_t getLookoutEventType() const = 0;
  //{ constructors which take exact same set of arguments as EventBase's and pass them directory to EventBase
  LookoutEvent() : EventBase() {}
  LookoutEvent(EventGeneratorID_t gid, unsigned int sid, EventTypeID_t tid, unsigned int dur=0)
    : EventBase(gid,sid,tid,dur) {}
  LookoutEvent(EventGeneratorID_t gid, unsigned int sid, EventTypeID_t tid, unsigned int dur, const std::string& n, float mag)
    : EventBase(gid,sid,tid,dur,n,mag) {}
  //}
};

// This event gives access to transformation matrix from joint specified from LookoutRequest to Base Frame
class LookoutPointAtEvent : public LookoutEvent {
public:
  NEWMAT::Matrix toBaseMatrix;
  virtual LookoutEventType_t getLookoutEventType() const { return lookAt; }
  LookoutPointAtEvent(const NEWMAT::Matrix& mat) : LookoutEvent(), toBaseMatrix(mat) {}
  LookoutPointAtEvent(const NEWMAT::Matrix& mat, EventGeneratorID_t gid, 
		     unsigned int sid, EventTypeID_t tid, unsigned int dur=0) 
    : LookoutEvent(gid,sid,tid,dur),toBaseMatrix(mat) {}
  LookoutPointAtEvent(const NEWMAT::Matrix& mat, EventGeneratorID_t gid, 
		     unsigned int sid, EventTypeID_t tid, unsigned int dur, const std::string& n, float mag)
    : LookoutEvent(gid,sid,tid,dur,n,mag),toBaseMatrix(mat) {}
  virtual EventBase* clone() const { return new LookoutPointAtEvent(*this); }
  virtual unsigned int getClassTypeID() const { return makeClassTypeID("LOLA"); }
  virtual std::string getDescription(bool showTypeSpecific=true, unsigned int verbosity=0) const;
  virtual unsigned int getBinSize() const;
  virtual unsigned int loadBinaryBuffer(const char buf[], unsigned int len);
  virtual unsigned int saveBinaryBuffer(char buf[], unsigned int len) const;
  virtual void loadXML(xmlNode* node);
  virtual void saveXML(xmlNode * node) const;
};


// Event which gives you access to the sketch stored as a result of StoreImage request
class LookoutSketchEvent : public LookoutPointAtEvent {
protected:
  DualCoding::Sketch<DualCoding::uchar> *img; // pointer to sketch existing inside lookout's requests queue
  
public:
  virtual LookoutEventType_t getLookoutEventType() const { return sketch; }
  LookoutSketchEvent(DualCoding::Sketch<DualCoding::uchar>& _img, const NEWMAT::Matrix& mat)
    : LookoutPointAtEvent(mat), img(&_img) {}
  LookoutSketchEvent(DualCoding::Sketch<DualCoding::uchar>& _img, const NEWMAT::Matrix& mat, 
		     EventGeneratorID_t gid, unsigned int sid, EventTypeID_t tid, unsigned int dur=0) 
    : LookoutPointAtEvent(mat,gid,sid,tid,dur), img(&_img) {}
  LookoutSketchEvent(DualCoding::Sketch<DualCoding::uchar>& _img, const NEWMAT::Matrix& mat, 
		     EventGeneratorID_t gid, unsigned int sid, 
		     EventTypeID_t tid, unsigned int dur, const std::string& n, float mag)
    : LookoutPointAtEvent(mat,gid,sid,tid,dur,n,mag), img(&_img) {}
  //! copy constructor (shallow copy)
  LookoutSketchEvent(const LookoutSketchEvent& lose)
    : LookoutPointAtEvent(lose), img(lose.img) {}
  
  const DualCoding::Sketch<DualCoding::uchar>& getSketch() const { return *img; }
  virtual EventBase* clone() const { return new LookoutSketchEvent(*this); }
  //  virtual std::string getDescription(bool showTypeSpecific=true, unsigned int verbosity=0) const;
private:
  LookoutSketchEvent& operator=(const LookoutSketchEvent&);
};

class LookoutIREvent : public LookoutPointAtEvent {
public:
  float distance;
  virtual LookoutEventType_t getLookoutEventType() const { return ir; }
  LookoutIREvent(float dist, const NEWMAT::Matrix& mat) : LookoutPointAtEvent(mat), distance(dist) {}
  LookoutIREvent(float dist, const NEWMAT::Matrix& mat, EventGeneratorID_t gid, 
		 unsigned int sid, EventTypeID_t tid, unsigned int dur=0) 
    : LookoutPointAtEvent(mat, gid,sid,tid,dur), distance(dist) {}
  LookoutIREvent(float dist, const NEWMAT::Matrix& mat, EventGeneratorID_t gid, unsigned int sid, 
		 EventTypeID_t tid, unsigned int dur, const std::string& n, float mag)
    : LookoutPointAtEvent(mat,gid,sid,tid,dur,n,mag), distance(dist) {}
  virtual EventBase* clone() const { return new LookoutIREvent(*this); }
  virtual unsigned int getClassTypeID() const { return makeClassTypeID("LOIR"); }
  virtual std::string getDescription(bool showTypeSpecific=true, unsigned int verbosity=0) const;
  virtual unsigned int getBinSize() const;
  virtual unsigned int loadBinaryBuffer(const char buf[], unsigned int len);
  virtual unsigned int saveBinaryBuffer(char buf[], unsigned int len) const;
  virtual void loadXML(xmlNode* node);
  virtual void saveXML(xmlNode * node) const;
};

class LookoutScanEvent : public LookoutEvent {
protected:
  //! pointer to tasks implemented during the scan
  vector<ScanRequest::Task*> *tasks;
public:
  virtual LookoutEventType_t getLookoutEventType() const { return scan; }
  LookoutScanEvent(vector<ScanRequest::Task*>& _tasks) : LookoutEvent(), tasks(&_tasks) {}
  LookoutScanEvent(vector<ScanRequest::Task*>& _tasks, EventGeneratorID_t gid, 
		   unsigned int sid, EventTypeID_t tid, unsigned int dur=0) 
    : LookoutEvent(gid,sid,tid,dur), tasks(&_tasks) {}
  LookoutScanEvent(vector<ScanRequest::Task*>& _tasks, EventGeneratorID_t gid, unsigned int sid, 
		   EventTypeID_t tid, unsigned int dur, const std::string& n, float mag)
    : LookoutEvent(gid,sid,tid,dur,n,mag), tasks(&_tasks) {}
  //! copy constructor (shallow copy)
  LookoutScanEvent(const LookoutScanEvent& lose)
    : LookoutEvent(lose), tasks(lose.tasks) {}
  //! assignment operator (shallow copy)
  const LookoutScanEvent& operator=(const LookoutScanEvent& lose) {
    if (this == &lose) return *this;
    LookoutEvent::operator=(lose);
    tasks = lose.tasks;
    return *this;
  }
  virtual EventBase* clone() const { return new LookoutScanEvent(*this); }
  const vector<ScanRequest::Task*>& getTasks() const { return *tasks; }
};

#endif
