#include "EventTranslator.h"
#include "Events/LocomotionEvent.h"
#include "Events/VisionObjectEvent.h"
#include "Events/TextMsgEvent.h"
#include "Events/EventRouter.h"
#include "Shared/debuget.h"
#include "Shared/ProjectInterface.h"
#include <iostream>

#ifdef PLATFORM_APERIOS
#  include <OPENR/OSubject.h>
#else
#  include "IPC/MessageQueue.h"
#endif
#include "IPC/RCRegion.h"

using namespace std;

EventTranslator::eventLookup_t EventTranslator::eventLookup;

EventTranslator::~EventTranslator() {
	for(eventLookup_t::iterator it=eventLookup.begin(); it!=eventLookup.end(); ++it)
		delete (*it).second;
	eventLookup.clear();
}

void
EventTranslator::encodeEvent(const EventBase& event, bool onlyReady/*=false*/) {
	event.setSaveFormat(EventBase::BINARY);
	const unsigned int headerlen=sizeof(event.getClassTypeID());
	const unsigned int bufsize=headerlen+event.getBinSize();
	char * buf=bufferRequest(bufsize);
	if(buf==NULL) {
		cerr << "ERROR: EventTranslator unable to transmit event because requested buffer was NULL" << endl;
		return;
	}
	unsigned int header=event.getClassTypeID();
	memcpy(buf,&header,headerlen);
	unsigned int used=event.saveBuffer(buf+headerlen,bufsize-headerlen);
	if(used==0) {
		cerr << "ERROR: EventTranslator unable to transmit event because EventBase::saveBuffer failed (buffer==" << (void*)(buf+headerlen) << ", size==" << bufsize-headerlen << ")" << endl;
		post(buf,0,onlyReady);
		return;
	}
	post(buf,used,onlyReady);
	return;
}

EventBase*
EventTranslator::decodeEvent(const char * entry, unsigned int size) {
	const unsigned int headerlen=sizeof(unsigned int);
	unsigned int header=0;
	memcpy(&header,entry,headerlen);
	//cout << "decodeEvent(" << (void*)entry << ","<< size << ") header is " << header << " (" << entry[0] << entry[1] << entry[2] << entry[3] << ")" << endl;
	eventLookup_t::iterator it=eventLookup.find(header);
	if(it==eventLookup.end()) {
		cerr << "ERROR: EventTranslator unable to translate buffer because header does not match a previously registered class type id" << endl;
		return NULL;
	}
	EventBase* evt=static_cast<EventBase*>((*it).second->constructTemplate());
	evt->setSaveFormat(EventBase::BINARY);
	if(evt->loadBuffer(entry+headerlen,size-headerlen)==0) {
		cerr << "ERROR: EventTranlator unable to translate buffer because data is malformed (EventBase::loadBuffer failed)" << endl;
		return NULL;
	}
	//cout << "decodes to " << evt->getDescription() << endl;
	return evt;
}

void
NoOpEventTranslator::encodeEvent(const EventBase& event, bool /*onlyReady=false*/) {
	evtRouter.postEvent(event);
}

char*
IPCEventTranslator::bufferRequest(unsigned int size) {
	ASSERT(curRegion==NULL,"WARNING: IPCEventTranslator::bufferRequest() curRegion was not NULL");
	try {
		curRegion = new RCRegion(size);
		return curRegion->Base();
	} catch(...) {
		curRegion=NULL;
		throw;
	}
}

void
IPCEventTranslator::post(const char* buf, unsigned int /*size*/, bool onlyReady) {
	ASSERTRET(curRegion!=NULL,"ERROR: IPCEventTranslator::post(buf,size) was NULL");
	if(buf!=curRegion->Base()) {
		cerr << "ERROR: IPCEventTranslator::post(buf,size) buf does not match value given from previous bufferRequest()" << endl;
		return;
	}
#ifdef PLATFORM_APERIOS
	if(!onlyReady) {
		subject.SetData(curRegion);
		subject.NotifyObservers();
	} else {
		for(ObserverConstIterator it=subject.begin(); it!=subject.end(); ++it) {
			if(subject.IsReady(*it)) {
				subject.SetData(*it,curRegion);
				subject.NotifyObserver(*it);
			}
		}
	}
#else
	if(!onlyReady || subject.getMessageSN(subject.newest())==subject.getMessagesRead()) {
		try {
			subject.sendMessage(curRegion);
		} catch(const std::exception& ex) {
			static char errmsg[256];
			strncpy(errmsg,("Occurred during IPCEventTranslator::post(), dropping interprocess event "+curName).c_str(),256);
			ProjectInterface::uncaughtException(__FILE__,__LINE__,errmsg,&ex);
		} catch(...) {
			static char errmsg[256];
			strncpy(errmsg,("Occurred during IPCEventTranslator::post(), dropping interprocess event "+curName).c_str(),256);
			ProjectInterface::uncaughtException(__FILE__,__LINE__,errmsg,NULL);
		}
	}
#endif
	curRegion->RemoveReference();
	curRegion=NULL;
}
	

/*! @file
 * @brief Implements EventTranslator, which receives events from EventRouters in non-Main processes and adds them into a SharedQueue for Main to pick up
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.22 $
 * $State: Exp $
 * $Date: 2006/09/28 21:00:27 $
 */

