//-*-c++-*-
#ifndef INCLUDED_EventRouter_h
#define INCLUDED_EventRouter_h

#include <vector>
#include <queue>
#include <map>
#include <algorithm>
#include "EventListener.h"
#include "EventTrapper.h"
#include "Shared/get_time.h"
#include "Shared/debuget.h"
#include "Shared/attributes.h"
#include <iostream>

//! This class will handle distribution of events as well as management of timers
/*! Classes must inherit from EventListener and/or EventTrapper in order to
 *  receive events.
 *
 *  Use the global ::erouter instance of EventRouter to both send (post) and receive (subscribe/listen) to
 *  events.  (except if you are posting from within a MotionCommand, in
 *  which case you should use MotionCommand::postEvent() so that it can correctly
 *  handle inter-process communication issues under Aperios (the Aibo's OS) -- under a
 *  Unix-based OS, this wouldn't be necessary.)
 *
 *  The events available for processing are listed in EventBase::EventGeneratorID_t.
 *  Each generator ID's documentation specifies how to interpret the source ID field, and whether
 *  you can expect events with that generator ID to be of a subclass of EventBase,
 *  such as TextMsgEvent or LocomotionEvent.  Many generators send plain
 *  EventBase instances.
 *
 *  When multiple listeners are subscribed, the order in which an event is
 *  distributed among them is:
 *  -# "Specific" listeners: any listener which specifies a particular source id.
 *     (doesn't matter if they specified type id as well)
 *    - older listeners get events before more recently added listeners ("FIFO")
 *  -# "General" listeners: those that subscribe to an entire generator
 *    - older listeners get events before more recently added listeners ("FIFO")
 *
 *  ...but if you're relying on that ordering, there probably should be a cleaner
 *  way to do whatever you're doing.
 *
 *  TimerEvents are generally only sent to the generator which requested them.  So if 
 *  EventListener @e A requests a timer (see addTimer()) with ID 0 at two second intervals,
 *  and @e B requests a timer with ID 0 at three second intervals,
 *  each will still only receive the timers they requested - no cross talk.
 *  The timer generator is unique in this regard, which is why it is built in
 *  as an integral component of the EventRouter.  However, EventListener @e A
 *  <b>can</b> receive @e B's timers if it specifically wants to, via addListener().  See "Timers" below.
 *
 *  If an EventListener/EventTrapper subscribes to the same event source multiple
 *  times, it will receive multiple copies of the event.  However, the first call
 *  to removeListener for a source will remove all subscriptions to that source.\n
 *  Example: EventListener @e A subscribes to (buttonEGID,*,*), and twice to
 *  (buttonEGID,0,*).
 *    - If button 0 is pressed, @e A will get three copies of the event.
 *    - If button 1 is pressed, @e A will get one copy.
 *    - If removeListener(&A,buttonEGID) is called, the (buttonEGID,*,*) is
 *      removed, <em>as well as</em> both of (buttonEGID,0,*).
 *    - If removeListener(&A,buttonEGID,0) is called, both of (buttonEGID,0,*)
 *      are removed, but (buttonEGID,*,*) would be untouched.
 *
 *  <h3>Timers</h3>
 *  addTimer() allows you to request an TimerEvent to be sent at some later point in time,
 *  possibly on a repeating basis.  Timers are specific to the behavior which requests
 *  them, and you @e do @e not (and usually should not) call addListener() in order to receive a timer
 *  event.
 *
 *  There is an important different between addTimer() and #addListener(timerEGID,...)!
 *  addTimer will "create" the timer, and will send the timer to the listener
 *  which created it when the timer expires.  This means that as long as the listener in
 *  question does @e not call addListener(timerEGID), it will @e only receive its own timers.
 *  In other words, with this usage there is no confusion with timer cross-talk between
 *  listeners, because each listener is only receiving its own timers.
 *
 *  However, if a listener calls addListener(timerEGID), it will begin receiving @e all timer events
 *  from throughout the system.  This allows you to have one behavior "eavesdrop" on
 *  another's timers.  In order to determine which listener requested/created the timer,
 *  you can use the TimerEvent::getTarget() value.
 *
 *  So beware that if you call both addTimer() and addListener(foo,timerEGID), 'foo' will get
 *  two calls to processEvent() for its own timers, and one call for all other timers, and will
 *  have to know to call TimerEvent::getTarget() to distinguish its timers from other
 *  listener's timers (if it cares about the difference...)
 *
 *  Timers are sent to the requesting listener before being broadcast -- EventTrappers cannot
 *  filter a listener's own timers, but can prevent the timer from being broadcast to other listeners.
 *
 *  <h3>Event processing examples:</h3>
 *
 *  Posting events:
 *  @code
 *  //method A: basic event posting (EventBase instance is sent)
 *  erouter->postEvent(EventBase::aiEGID, 1234, EventBase::statusETID);
 *
 *  //method B: specific event instance is posted (have to use this style to post a subclass)
 *  TextMsgEvent txt("hello world");
 *  erouter->postEvent(txt);
 *  // or can be done in one line:
 *  erouter->postEvent(TextMsgEvent("hello world"))
 *  @endcode
 *
 *  Receiving events:
 *  @code
 *  //given an EventListener subclass:
 *  class YourListener : public EventListener {
 *  public:
 *    virtual void processEvent(const EventBase& e) {
 *      std::cout << "Got: " << e.getName() << std::endl;
 *    }
 *  };
 *
 *  YourListener yourList;
 *
 *  // subscribes it to all EventBase::aiEGID events:
 *  erouter->addListener(&yourList, EventBase::aiEGID);
 *
 *  // subscribes only to button activity from the head, but not other buttons:
 *  erouter->addListener(&yourList, EventBase::buttonEGID, ERS7Info::HeadButOffset);
 *  @endcode
 *  Typically in a BehaviorBase subclass, you would just specify 'this' instead of '&yourList'.
 *
 *  <h3>Timer processing examples:</h3>
 *
 *  Requesting/Creating timers:
 *  @code
 *  YourListener yourList; // (any EventListener subclass)
 *
 *  // sends a timer with source ID 123 every 5 seconds to yourList:
 *  erouter->addTimer(&yourList, 123, 5000);
 *
 *  // sends a timer with ID 456 after 1 second, *no repeat, one time only*
 *  erouter->addTimer(&yourList, 456, 1000, false);
 *
 *  // cancels the first timer
 *  erouter->removeTimer(&yourList, 123);
 *
 *  // postpone/update the second timer's settings (*doesn't* create two timers with the same ID)
 *  erouter->addTimer(&yourList, 456, 2500, false);
 *
 *  @endcode
 *  Again, typically in a BehaviorBase subclass, you would just specify 'this' instead of '&yourList'.
 *
 *  @see EventBase::EventGeneratorID_t for a complete listing of all generators,
 *  as well as instructions on how to add new generators.
 *  @see Tutorials:
 *    - <a href="../FirstBehavior2.html">Steps 3, 4, & 5 of Tekkotsu's First Behavior Tutorial</a>
 *    - <a href="http://www.cs.cmu.edu/~dst/Tekkotsu/Tutorial/events.shtml">David Touretzky's Events Chapter</a>
 *    - <a href="http://www.cs.cmu.edu/afs/cs/academic/class/15494-s06/www/lectures/behaviors.pdf">CMU's Cognitive Robotics course slides</a>
 */
class EventRouter : public EventListener {
 public:
	EventRouter(); //!< Constructs the router
	virtual ~EventRouter() { reset(); removeAllTimers(); } //!< just calls reset and removeAllTimers()
	
	void reset() { listeners.clear(); trappers.clear(); removeAllTimers(); } //!< erases all listeners, trappers and timers, resets EventRouter
	
	
	//!@name Posting/Processing Events
	
	/*!@brief recommended to create and post an event using current buffer setting */
	void postEvent(EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid, unsigned int dur=0) { processEvent(EventBase(egid,sid,etid,dur)); }
	void postEvent(EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid, unsigned int dur, const std::string& n, float m) { processEvent(EventBase(egid,sid,etid,dur,n,m)); }
	//! deprecated -- pass by reference instead;  this version posts the specified event and then deletes it after processing is complete
	/*! moving to pass by reference instead of pass by pointer to avoid questions about who deletes the event and the event's scope,
	 *  also keeping the event on the stack is faster by avoiding unnecessary heap operations. */
	void postEvent(EventBase* e) ATTR_deprecated;
	//! posts the specified event, but doesn't delete it at the end -- equivalent to processEvent(e)
	void postEvent(const EventBase& e) { processEvent(e); }

	//! determines if timers need to be posted, and posts them if so.
	/*! Call this often to ensure accurate timers. */
	void processTimers();
	//! sends event to its trappers & listeners, but doesn't delete the event at the end (see also postEvent())
	/*! this posting method is supplied to allow an EventRouter to behave as a listener as 
	 *  well -- the 'routers' really can form a sort of network, if desired.  postEvent() is
	 *  probably a more memnomic interface to use in direct function calls however,
	 *  so that is the one you should call. */
	void processEvent(const EventBase& e);
	//@}

	
	//!@name Listener/Trapper Recall
	
	//! returns true if the specified listener/trapper would receive any events that match the specified criteria
	bool isListeningAny(EventListener* el, EventBase::EventGeneratorID_t egid) { return listeners.verifyMappingAny(el,egid); }
	bool isListeningAny(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid) { return listeners.verifyMappingAny(el,egid,sid); }
	bool isListeningAll(EventListener* el, EventBase::EventGeneratorID_t egid) { return listeners.verifyMappingAll(el,egid); }
	bool isListeningAll(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid) { return listeners.verifyMappingAll(el,egid,sid); }
	bool isListening(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) { return listeners.verifyMapping(el,egid,sid,etid); }
	bool isListening(EventListener* el, const EventBase& e) { return listeners.verifyMapping(el,e.getGeneratorID(),e.getSourceID(),e.getTypeID()); }
	bool isTrappingAny(EventTrapper* el, EventBase::EventGeneratorID_t egid) { return trappers.verifyMappingAny(el,egid); }
	bool isTrappingAny(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid) { return trappers.verifyMappingAny(el,egid,sid); }
	bool isTrappingAll(EventTrapper* el, EventBase::EventGeneratorID_t egid) { return trappers.verifyMappingAll(el,egid); }
	bool isTrappingAll(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid) { return trappers.verifyMappingAll(el,egid,sid); }
	bool isTrapping(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) { return trappers.verifyMapping(el,egid,sid,etid); }
	bool isTrapping(EventTrapper* el, const EventBase& e) { return trappers.verifyMapping(el,e.getGeneratorID(),e.getSourceID(),e.getTypeID()); }
	//@}
	
	
	//!@name Listener/Trapper Detection
	
	/*!@brief <b>counts both listeners and trappers</b>, so generators can tell if it even needs to bother generating an event...*/
	/* Generators can also subscribe to the EventBase::erouterEGID event stream if
	 * they wish to be notified when they gain or lose listeners (particularly the
	 * first or last).\n
	 * ... if a tree falls in a forest, and there's no one around to see it, does
	 * it make a sound?\n
	 * ... if Vision sees a ball in an image, and there's no listeners, does it
	 * make an event? ;) */
	bool hasListeners(EventBase::EventGeneratorID_t egid) { return trappers.hasMapping(egid) || listeners.hasMapping(egid); }
	bool hasListeners(EventBase::EventGeneratorID_t egid, unsigned int sid) { return trappers.hasMapping(egid,sid) || listeners.hasMapping(egid,sid); }
	bool hasListeners(EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) { return trappers.hasMapping(egid,sid,etid) || listeners.hasMapping(egid,sid,etid); }
	bool hasListeners(const EventBase& e) { return hasListeners(e.getGeneratorID(),e.getSourceID(),e.getTypeID()); }
	//@}

	
	//!@name Timer Management
	
	//! adds a timer if it doesn't exist, or resets the timer if it already exists.
	void addTimer(EventListener* el, unsigned int sid, unsigned int delay, bool repeat=true);
	void addTimer(EventListener* el, const EventBase& e, bool repeat=true) { addTimer(el,e.getSourceID(),e.getDuration(),repeat); } //!< calls the other addTimer() with the event's source id and duration, doesn't check to see if the generator is timerEGID
	
	//! clears all pending timers for listener @a el; see remove()
	void removeTimer(EventListener* el);
	void removeTimer(EventListener* el, unsigned int sid); //!< clears any pending timers with source id @a sid for listener @a el
	void removeAllTimers(); //!< clears all timers for all listeners
	unsigned int getNextTimer() { return (timers.size()==0 ? -1U : timers.front()->next); } //!< returns time of next timer activation
	//@}

	
	//!@name Listener Management
	
	//! Adds a listener for all events from a given event generator
	void addListener(EventListener* el, EventBase::EventGeneratorID_t egid);
	void addListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid); //!< Adds a listener for all types from a specific source and generator
	void addListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid); //!< Adds a listener for a specific source id and type from a given event generator
	void addListener(EventListener* el, const EventBase& e); //!< Uses the generator, source, and type fields of @a e to add a listener for that specific tuple

	//! stops sending ALL events to the listener -- does not remove pending timers however (may need to call removeTimer(el) as well); see remove()
	void removeListener(EventListener* el); 
	//! stops sending specified events from the generator to the listener.
	void removeListener(EventListener* el, EventBase::EventGeneratorID_t egid);
	void removeListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid); //!< stops sending specified events from the generator to the listener.
	void removeListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid); //!< stops sending specified events from the generator to the listener.
	void removeListener(EventListener* el, const EventBase& e); //!< Uses the generator, source, and type fields of @a e to remove a listener for that specific tuple
	
	//! stops all events and timers, shorthand for removeListener(el) and removeTimer(el); Note that trappers are separate, removeTrapper() is @e not called
	void remove(EventListener* el) { removeListener(el); removeTimer(el); }

	//@}

	//!@name Trapper Management
	
	//! Adds a trapper for a specific source id and type from a given event generator
	/*! Note that only the broadcasted timers can be trapped.  The EventListener which requested the timer will receive that timer before any trapping is done. */
	void addTrapper(EventTrapper* el, const EventBase& e);
	void addTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid); //!< Adds a trapper for all events from a given event generator
	void addTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid);  //!< Adds a trapper for all types from a specific source and generator
	void addTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid); //!< Adds a trapper for a specific source id and type from a given event generator

	void addTrapper(EventTrapper* el); //!< adds a trapper for ALL events

	//! stops sending specified events from the generator to the trapper.
	void removeTrapper(EventTrapper* el, const EventBase& e);
	void removeTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid); //!< stops sending specified events from the generator to the trapper.
	void removeTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid); //!< stops sending specified events from the generator to the trapper.
	void removeTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid); //!< stops sending specified events from the generator to the trapper.

	void removeTrapper(EventTrapper* el); //!< stops sending ALL events to the trapper
	//@}

 protected:
	//! Contains all the information needed to maintain a timer by the EventRouter
	struct TimerEntry {
		//! constructs an entry using the given value for next - useful for with TimerEntryPtrCmp
		explicit TimerEntry(unsigned int nxt) : el(NULL), sid(0), delay(0), next(nxt), repeat(false) {}
		//! constructs with the given values, sets next field automatically; see next
		TimerEntry(EventListener* e, unsigned int s, unsigned int d, bool r) : el(e), sid(s), delay(d), next(get_time()+delay), repeat(r) {}
		//! just does the default, i'm just being explicit since there's a pointer (no deep copy!)
		TimerEntry(const TimerEntry& t) : el(t.el), sid(t.sid), delay(t.delay), next(t.next), repeat(t.repeat) {}
		//! just does the default, i'm just being explicit since there's a pointer (no deep copy!)
		TimerEntry& operator=(const TimerEntry& t) { el=t.el; sid=t.sid; delay=t.delay; next=t.next; repeat=t.repeat; return *this; }
		//! will reset timer
		/*! @param d the time from now when the timer should go off (in milliseconds)
		 *  @param r true if the timer should automatically repeat */
		void Set(unsigned int d, bool r) { delay=d; repeat=r; next=get_time()+delay; }
		EventListener* el;  //!< the listener to fire at
		unsigned int sid;   //!< the source id to fire with
		unsigned int delay; //!< the delay until firing
		unsigned int next;  //!< the time at which this timer will go off next
		bool repeat;        //!< if true, will reset after firing, else will be deleted
	};
	/*! @brief Used by STL to sort the timer list in order of activation time
	 *  @see EventRouter::timers */
	class TimerEntryPtrCmp {
	public:
		//! Used by STL to sort the timer list in order of activation time; see timers
		/*! Since we remove NULLs before sorting, shouldn't need to check here (and I want to know if i'm wrong)
		 *  @return (a->next<b->next) */
		bool operator()(const TimerEntry* const a, const TimerEntry* const b) const { return (a->next<b->next); }
	};
	typedef std::vector<TimerEntry*>::iterator timer_it_t; //!< makes code more readable
	std::vector<TimerEntry*> timers;         //!< the list of timer entries being maintained, kept sorted by time they go active

	//! just for debugging
	void chkTimers() {
		unsigned int last=0;
		for(timer_it_t it=timers.begin(); it!=timers.end(); it++) {
			if(last>(*it)->next) {
				dispTimers();
				return;
			}
			last=(*it)->next;
		}
	}
	//! just for debugging
	void dispTimers() {
		std::cout << "out of order timers " << get_time() << " :\t";
		unsigned int last=0;
		for(timer_it_t it=timers.begin(); it!=timers.end(); it++) {
			if(last>(*it)->next)
				std::cout << "##";
			std::cout << (last=(*it)->next) << '\t';
		}
		std::cout << std::endl;
	}
	
	//! Does the actual storage of the mapping between EventBase's and the EventListeners/EventTrappers who should receive them
	/*! Actually only stores void*'s, so it's more general than just Listeners or Trappers */
	class EventMapper {
	public:
		//! constructor
		EventMapper();

		void addMapping(void* el, EventBase::EventGeneratorID_t egid) { allevents[egid].push_back(el); } //!< Adds a listener for all events from a given event generator
		void addMapping(void* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid); //!< Adds a listener for a specific source id and type from a given event generator

		//! Removes a listener for all events from a given event generator, returns true if something was actually removed
		/*! Doesn't necessarily remove the vector or mapping if this was the last listener, use clean() to do that */
		bool removeMapping(void* el, EventBase::EventGeneratorID_t egid); 

		//! Removes a listener for a specific source id and type from a given event generator, returns true if something was actually removed
		/*! Doesn't necessarily remove the vector or mapping if this was the last listener, use clean() to do that */
		bool removeMapping(void* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid);

		void clean(); //!<removes empty data structures for all event generators
		void clean(EventBase::EventGeneratorID_t egid); //!< removes empty data structures associated with a single event generator
		void clear(); //!<Resets the mapping

		//@{
		//! so stuff can tell if it even needs to bother generating an event...
		/*! ... if a tree falls in a forest, and there's no one around to see it, does it make a sound?\n
			  ... if Vision sees a ball in an image, and there's no listeners, does it make an event? ;) \n
			  @return true if it has any listeners, false otherwise */
		bool hasMapping(EventBase::EventGeneratorID_t egid) const;
		bool hasMapping(EventBase::EventGeneratorID_t egid, unsigned int sid) const;
		bool hasMapping(EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) const;
		//@}

		//! builds a list of all listeners which should receive the event, templated to typecast the pointers for you
		/*! @param e the key event
		 *  @param listeners upon return, the resulting list of listeners @a e maps to\n
		 *  @a listeners is not cleared prior to building, new listeners are pushed on end\n
		 *  Results are in the order: all specific matches first, all generator listeners second, in order they were added to the EventMapper.*/
		template<class T>
		void getMapping(const EventBase& e, std::vector<T*>& listeners) const;

		//! Used to make sure that the specified listener exists for the given event
		/*! This is needed because after we call processEvent on a lister, we can't assume
		 *  that no other listeners have been modified - one listener could cause another
		 *  to turn off.  If that has happened, we shouldn't send the event, even if it
		 *  was in the queue originally. */
		bool verifyMapping(void * listener, const EventBase& e) const { return verifyMapping(listener,e.getGeneratorID(),e.getSourceID(),e.getTypeID()); }
		//! Used to make sure that the specified listener exists for the given event
		/*! This is needed because after we call processEvent on a lister, we can't assume
		 *  that no other listeners have been modified - one listener could cause another
		 *  to turn off.  If that has happened, we shouldn't send the event, even if it
		 *  was in the queue originally. */
		bool verifyMapping(void * listener, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) const;

		//! Needed to complete EventRouter::isListening suite
		/*! Only checks #allevents */
		bool verifyMappingAll(void * listener, EventBase::EventGeneratorID_t egid) const;
		//! Needed to complete EventRouter::isListening suite
		/*! Checks both #allevents and #filteredevents */
		bool verifyMappingAny(void * listener, EventBase::EventGeneratorID_t egid) const;

		//! Needed to complete EventRouter::isListening suite
		/*! Checks both #allevents and #filteredevents, must be found in all */
		bool verifyMappingAll(void * listener, EventBase::EventGeneratorID_t egid, unsigned int sid) const;
		//! Needed to complete EventRouter::isListening suite
		/*! Checks both #allevents and #filteredevents, can be found in either */
		bool verifyMappingAny(void * listener, EventBase::EventGeneratorID_t egid, unsigned int sid) const;

	protected:
		//! a mapping from source IDs (unsigned ints), each to a vector of pointers to listeners
		/*! main use in filteredevents @see filteredevents */
		typedef std::map<unsigned int,std::vector<void*>,std::less<unsigned int> > SIDtoListenerVectorMap_t;
		
		//! an array of vectors of pointers to listeners... in other words, a vector of listener pointers for each generator
		std::vector<void*> allevents[EventBase::numEGIDs];
		//! not for the faint of heart: a matrix of mappings to vectors of pointers to listeners
		SIDtoListenerVectorMap_t* filteredevents[EventBase::numEGIDs][EventBase::numETIDs];

	private:
		EventMapper(const EventMapper&);           //!< this shouldn't be called...
		EventMapper& operator=(const EventMapper&); //!< this shouldn't be called...
	};

	EventMapper trappers;  //!< A mapping of which EventTrapper's should get a chance to trap the event
	EventMapper listeners; //!< A mapping of which EventListener's should receive events

	//! contains information regarding the progress of posting an event
	/*! This allows us to resume and complete the posting of the "current" event before processing a new incoming event */
	class PostingStatus {
	public:
		//! constructor
		PostingStatus(const EventMapper& eventTrappers, const EventMapper& eventListeners, const EventBase& event)
			: trappers(eventTrappers), listeners(eventListeners), t(), tit(), l(), lit(), e(event)
		{ trappers.getMapping(e,t); tit=t.begin(); listeners.getMapping(e,l); lit=l.begin(); }
		//! begins or resumes sending the event #e to trappers and listeners in #t and #l
		void process();
	protected:
		const EventMapper& trappers; //!< the current trapper mapping, used to verify each entry in #t is still valid before processing it
		const EventMapper& listeners; //!< the current listener mapping, used to verify each entry in #l is still valid before processing it
		std::vector<EventTrapper*> t; //!< list of trappers which were subscribed when the PostingStatus instance was constructed
		std::vector<EventTrapper*>::const_iterator tit; //!< current position within #t
		std::vector<EventListener*> l; //!< list of listeners which were subscribed when the PostingStatus instance was constructed
		std::vector<EventListener*>::const_iterator lit; //!< current position within #l
		const EventBase& e; //!< the event being processed
	};
	std::queue<PostingStatus*> postings; //!< stores calls to post() currently in progress -- may grow if one postEvent() triggers another; this allows us to finish processing of the original postEvent() before starting the second.
};

//! a global router for cross communication, probably the most common usage, although perhaps there may be times you'd rather have multiple event routers for smaller sections
extern EventRouter * erouter;

/*! @file
 * @brief Describes EventRouter class, for distribution and trapping of events to listeners
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.35 $
 * $State: Exp $
 * $Date: 2006/10/03 21:09:04 $
 */

#endif
