//-*-c++-*-
#ifndef INCLUDED_MapBuilderRequests_h_
#define INCLUDED_MapBuilderRequests_h_

#include "BlobData.h"
#include "ShapeTypes.h"
#include "VRmixin.h"

namespace DualCoding {

class MapBuilderRequest {
  typedef map<ShapeType_t,set<int> > colorMap;
  friend class MapBuilder;

public:
  enum MapBuilderRequestType_t { 
    none, 
    takeSnap, 
    localMap, 
    worldMap
  };

  virtual MapBuilderRequestType_t getRequestType() const=0;

  virtual ~MapBuilderRequest() {}

  unsigned int requestID; // set by mapbuilder when added to request queue
  colorMap objectColors, occluderColors;
  map<int, int> minBlobAreas;
  map<int, BlobData::BlobOrientation_t> blobOrientations;
  unsigned int floorColor;
  unsigned int numImages;
  enum GroundPlaneAssumption_t { onStand, onLegs } groundPlaneAssumption;

  //! Constructor
  MapBuilderRequest(unsigned int num=1)
    : requestID(0), objectColors(), occluderColors(), 
      minBlobAreas(), blobOrientations(), floorColor(0), 
      numImages(num), groundPlaneAssumption(onLegs) {}

  //! Copy constructor
  MapBuilderRequest(const MapBuilderRequest& req)
    : requestID(req.requestID), 
      objectColors(req.objectColors), occluderColors(req.occluderColors), 
      minBlobAreas(req.minBlobAreas), blobOrientations(req.blobOrientations),
      floorColor(req.floorColor), numImages(req.numImages),
      groundPlaneAssumption(req.groundPlaneAssumption) {}

private:
  MapBuilderRequest& operator=(const MapBuilderRequest& req);
};

class TakeSnapRequest : public MapBuilderRequest {
public:
  virtual MapBuilderRequestType_t getRequestType() const { return takeSnap; }
  TakeSnapRequest(unsigned int num=1)
    : MapBuilderRequest(num) {}
  TakeSnapRequest(const TakeSnapRequest& req)
    : MapBuilderRequest(req) {}
};

class TakeSnapAtRequest : public TakeSnapRequest {
public:
  TakeSnapAtRequest(const Point& pt, unsigned int num=1)
    : TakeSnapRequest(num), gazePt(pt) {}
  TakeSnapAtRequest(const TakeSnapAtRequest& req)
    : TakeSnapRequest(req), gazePt(req.gazePt) {}
  Point gazePt;
};


class LocalMapRequest : public MapBuilderRequest {
public:
  virtual MapBuilderRequestType_t getRequestType() const { return localMap; }

  //! Constructor with bounding box
  LocalMapRequest(coordinate_t maxX, coordinate_t minX, coordinate_t maxY, 
		  coordinate_t minY, unsigned int _maxDist, unsigned int num=1, 
		  bool clear=true, bool _doScan=true)
    : MapBuilderRequest(num), shs(VRmixin::localShS), 
      area(Shape<BlobData>
	   (new BlobData
	    (VRmixin::groundShS,Point(maxX,maxY),Point(maxX,minY),
	     Point(minX,maxY),Point(minX,minY),(maxX-minX)*(maxY-minY),
	     vector<BlobData::run>(),BlobData::groundplane,rgb(0,0,0)))), 
      maxDist(_maxDist), clearShapes(clear), pursueShapes(true), manualHeadMotion(false),
      removePts(true), doScan(_doScan) {}

  //! Constructor with point target
  LocalMapRequest(const Point& pt, unsigned int _maxDist, unsigned int num=1, 
		  bool clear=true, bool _doScan=true)
    : MapBuilderRequest(num), shs(VRmixin::localShS), 
      area(Shape<PointData>(new PointData(VRmixin::groundShS,pt))), 
      maxDist(_maxDist), clearShapes(clear), pursueShapes(true), manualHeadMotion(false),
      removePts(true), doScan(_doScan) {}

  //! Constructor with ShapeRoot to define scan region
  LocalMapRequest(const ShapeRoot _area, unsigned int _maxDist, unsigned int num=1, 
		  bool clear=true, bool _doScan=true)
    : MapBuilderRequest(num), shs(VRmixin::localShS), 
      area(_area), maxDist(_maxDist), clearShapes(clear), pursueShapes(true), manualHeadMotion(false),
      removePts(true), doScan(_doScan) {}

  //! Copy constructor
  LocalMapRequest(const LocalMapRequest& req)
    : MapBuilderRequest(req), shs(req.shs), area(req.area), maxDist(req.maxDist), 
      clearShapes(req.clearShapes), pursueShapes(req.pursueShapes), manualHeadMotion(req.manualHeadMotion),
      removePts(req.removePts), doScan(req.doScan) {}

  virtual ~LocalMapRequest() {} 

  ShapeSpace& shs;
  ShapeRoot area; //!< The area to search, in egocentric coords
  unsigned int maxDist; //!< Ignore objects farther than this distance
  bool clearShapes; //!< If true, clear the shape space at start of request
  bool pursueShapes; //!< If true, generate new gaze points as shapes are recognized
  bool manualHeadMotion; //!< If true, waits for !msg MoveHead before moving to next gaze point (for debugging)
  bool removePts; //!< If true, remove pending gaze points if they're visible in current image
  bool doScan; //!< If true, do a continuous scan of the area to find interest points to be examined

protected:
  //! Constructor used by subclass WorldMapRequest
  LocalMapRequest(ShapeSpace& _shs, const ShapeRoot& _area, unsigned int _maxDist, 
		  unsigned int num=1, bool clear=true, bool _doScan=true)
    : MapBuilderRequest(num), shs(_shs), area(_area), 
      maxDist(_maxDist), clearShapes(clear), pursueShapes(true), manualHeadMotion(false),
      removePts(true), doScan(_doScan) {}

  //! Constructor used by subclass WorldMapRequest
  LocalMapRequest(ShapeSpace& _shs, coordinate_t maxX, coordinate_t minX, 
		  coordinate_t maxY, coordinate_t minY, unsigned int _maxDist, 
		  unsigned int num=1, bool clear=true, bool _doScan=true)
    : MapBuilderRequest(num), shs(_shs), 
      area(Shape<BlobData>
	   (new BlobData
	    (VRmixin::groundShS,Point(maxX,maxY),Point(maxX,minY),
	     Point(minX,maxY),Point(minX,minY),(maxX-minX)*(maxY-minY),
	     vector<BlobData::run>(),BlobData::groundplane,rgb(0,0,0)))), 
      maxDist(_maxDist), clearShapes(clear), pursueShapes(true), manualHeadMotion(false),
      removePts(true), doScan(_doScan) {}
};

class LocalMapTestRequest : public LocalMapRequest {
public:
  LocalMapTestRequest(bool (*func)(), coordinate_t maxX, coordinate_t minX, 
		      coordinate_t maxY, coordinate_t minY, unsigned int _maxDist, 
		      unsigned int num=1, bool clear=true, bool _doScan=true)
    : LocalMapRequest(maxX,minX,maxY,minY,_maxDist,num,clear,_doScan), exitTest(func){}

   LocalMapTestRequest(const LocalMapTestRequest& req)
    : LocalMapRequest(req), exitTest(req.exitTest) {}

  virtual ~LocalMapTestRequest() {} 

  bool (*exitTest)();

private:
  LocalMapTestRequest& operator=(const LocalMapTestRequest&);
};

class WorldMapRequest : public LocalMapRequest {
public:
  virtual MapBuilderRequestType_t getRequestType() const { return worldMap; }

  //! Constructor using a shape to specify area
  WorldMapRequest(const ShapeRoot& _area, unsigned int _maxDist, 
		  unsigned int num=1, bool clear=true, bool _doScan=true)
    : LocalMapRequest(VRmixin::worldShS,_area,_maxDist,num,clear,_doScan) {}

  //! Constructor using a bounding box
  WorldMapRequest(coordinate_t maxX, coordinate_t minX, coordinate_t maxY, 
		  coordinate_t minY, unsigned int _maxDist, 
		  unsigned int num=1, bool clear=true, bool _doScan=true)
    : LocalMapRequest(VRmixin::worldShS,maxX,minX,maxY,minY,_maxDist,num,clear,_doScan) {}

  //! Copy constructor
  WorldMapRequest(const WorldMapRequest& req) : LocalMapRequest(req) {}

  virtual ~WorldMapRequest() {}
};

class WorldMapTestRequest : public WorldMapRequest {
public:
  //  virtual MapBuilderRequestType_t getRequestType() const { return worldMapTest; }
  WorldMapTestRequest(bool (*func)(), const ShapeRoot& _area, unsigned int _maxDist,
		      unsigned int num=1, bool clear=true, bool _doScan=true)
    : WorldMapRequest(_area,_maxDist,num,clear,_doScan), exitTest(func){}
  WorldMapTestRequest(bool (*func)(), coordinate_t maxX, coordinate_t minX, 
		      coordinate_t maxY, coordinate_t minY, unsigned int _maxDist,
		      unsigned int num=1, bool clear=true, bool _doScan=true)
    : WorldMapRequest(maxX,minX,maxY,minY,_maxDist,num,clear,_doScan), exitTest(func){}
  WorldMapTestRequest(const WorldMapTestRequest& req)
    : WorldMapRequest(req), exitTest(req.exitTest) {}
  virtual ~WorldMapTestRequest() {} 
  bool (*exitTest)();
private:
  WorldMapTestRequest& operator=(const WorldMapTestRequest&);
};

} // namespace

#endif
