//-*-c++-*-
#ifndef INCLUDED_LookoutRequests_h_
#define INCLUDED_LookoutRequests_h_

#include "Shared/newmat/newmat.h"
#include "Shared/ProjectInterface.h"
#include "Shared/WorldState.h"
#include "DualCoding.h"
#include "VRmixin.h"

namespace DualCoding {
  
class LookoutRequest {
public:
  enum HeadMotionType_t { 
    none,
    pointAt,
    scan,
    track
  };
  virtual HeadMotionType_t getHeadMotionType() const = 0;
  //! Constructor
  LookoutRequest() : requestID((unsigned int) -1) {}
  virtual ~LookoutRequest() {}
  //! Lookout assigns non-zero value when added to queue
  unsigned int requestID; 
protected:
  //! Copy constructor
  LookoutRequest(const LookoutRequest& req) : requestID(req.requestID) {}
private:
  LookoutRequest& operator=(const LookoutRequest& req);
};

// ---------- pointAt requests ----------
class PointAtBase : public LookoutRequest {
protected:
  //! Constructor
  PointAtBase(unsigned int _joint, unsigned int _duration)
    : joint(_joint), toBaseMatrix(), duration(_duration) {}

  //! Copy constructor
  PointAtBase(const PointAtBase& base)
    : LookoutRequest(base), joint(base.joint), 
      toBaseMatrix(base.toBaseMatrix), duration(base.duration) {}

public:
  enum PointAtRequestType_t { none, storeImage, measureDist };
  virtual PointAtRequestType_t getPointAtType() const = 0;//const { return none; }
  virtual HeadMotionType_t getHeadMotionType() const { return LookoutRequest::none; }
  unsigned int joint; //!< frame from which base frame transformation matrix is created, e.g., Camera, IR, etc.
  NEWMAT::Matrix toBaseMatrix; //!< transformation matrix from joint frame to base frame
  //! how much time to wait before taking measurements or throw completion event after head reaches gazePt.
  unsigned int duration;
};

class PointAtRequest : virtual public PointAtBase {
public:
  //! Constructor
  PointAtRequest(const Point& pt, unsigned int dur=1000)
    : PointAtBase(CameraFrameOffset,dur), gazePt(pt) {};

  //! Copy constructor
  PointAtRequest(const PointAtRequest& req) 
    : PointAtBase(req), gazePt(req.gazePt) {}

  virtual PointAtRequestType_t getPointAtType() const { return none; }
  virtual HeadMotionType_t getHeadMotionType() const { return pointAt; }
  Point gazePt; //!< point to look at in base frame coordinates
};

class StoreImageRequest : virtual public PointAtBase {
public:
  //! Constructor
  StoreImageRequest(Sketch<uchar> (&func)()=VRmixin::sketchFromSeg, 
		    unsigned int dur=1000)
    : PointAtBase(CameraFrameOffset,dur), image(), sketchFunc(func) {};
  //! Copy constructor
  StoreImageRequest(const StoreImageRequest& req) 
    : PointAtBase(req), image(req.image), sketchFunc(req.sketchFunc) {}

  virtual PointAtRequestType_t getPointAtType() const { return storeImage; }
  Sketch<uchar> image; //<! stores image here
  Sketch<uchar> (*sketchFunc)(); //<! function used to generate image
private:
  StoreImageRequest& operator=(const StoreImageRequest&);
};

class StoreImageAtRequest : public PointAtRequest, public StoreImageRequest {
public:
  //! Constructor
  StoreImageAtRequest(const Point& pt, Sketch<uchar> (&func)()=VRmixin::sketchFromSeg, 
		      unsigned int dur=1000)
    : PointAtBase(CameraFrameOffset,dur), PointAtRequest(pt,dur), StoreImageRequest(func,dur) {}
  //! Copy constructor
  StoreImageAtRequest(const StoreImageAtRequest& req)
    : PointAtBase(req), PointAtRequest(req), StoreImageRequest(req) {}

  virtual PointAtRequestType_t getPointAtType() const { return storeImage; }
  virtual HeadMotionType_t getHeadMotionType() const { return pointAt; }
private:
  StoreImageAtRequest& operator=(const StoreImageAtRequest&);
};
 
class StoreModeImageRequest : public StoreImageRequest  {
public:
  //! Constructor
  StoreModeImageRequest(unsigned int num, unsigned int _interval=100,
		    Sketch<uchar> (&func)()=VRmixin::sketchFromSeg, 
		    unsigned int dur=1000)
    : PointAtBase(CameraFrameOffset,dur), StoreImageRequest(func,dur), 
      numImages(num), interval(_interval) {};

  //! Copy constructor
  StoreModeImageRequest(const StoreModeImageRequest& req)
    : PointAtBase(req), StoreImageRequest(req), 
      numImages(req.numImages), interval(req.interval) {}

  unsigned int numImages; //!< number of image(sketch) over which to take mode
  unsigned int interval; //!< interval between storing images
};

class StoreModeImageAtRequest : public PointAtRequest, public StoreModeImageRequest {
public:
  //! Constructor
  StoreModeImageAtRequest (const Point& pt, unsigned int num, unsigned int _interval=100,
			   Sketch<uchar> (&func)()=VRmixin::sketchFromSeg, unsigned int dur=1000)
    : PointAtBase(CameraFrameOffset,dur), PointAtRequest(pt,dur), 
      StoreModeImageRequest(num,_interval,func) {}
  //! Copy constructor
  StoreModeImageAtRequest(const StoreModeImageAtRequest& req) 
    : PointAtBase(req), PointAtRequest(req), StoreModeImageRequest(req) {}

  virtual PointAtRequestType_t getPointAtType() const { return storeImage; }
  virtual HeadMotionType_t getHeadMotionType() const { return pointAt; }
private:
  StoreModeImageAtRequest& operator=(const StoreModeImageAtRequest&);
};

class MeasureDistanceRequest : virtual public PointAtBase {
public:
  //! Constructor
  MeasureDistanceRequest(unsigned int dur=1000)
    : PointAtBase(
#ifdef TGT_ERS7 
		 NearIRFrameOffset
#else 
		 IRFrameOffset
#endif
		 ,dur), val() {};
  //! Copy Constructor
  MeasureDistanceRequest(const MeasureDistanceRequest& req)
    : PointAtBase(req), val(req.val) {}

  virtual PointAtRequestType_t getPointAtType() const { return measureDist; }
  float val;
};

class MeasureDistanceAtRequest : public PointAtRequest, public MeasureDistanceRequest {
public:
  //! Constructor
  MeasureDistanceAtRequest(const Point& pt, unsigned int dur=1000)
    : PointAtBase(0/*overwritten by MeasureDistanceRequest constructor*/,dur), 
      PointAtRequest(pt,dur), MeasureDistanceRequest(dur) {};
  //! Copy constructor
  MeasureDistanceAtRequest(const MeasureDistanceAtRequest& req)
    : PointAtBase(req), PointAtRequest(req), MeasureDistanceRequest(req) {}
  virtual PointAtRequestType_t getPointAtType() const { return measureDist; }
  virtual HeadMotionType_t getHeadMotionType() const { return pointAt; }
};

//--------------------------------------------

// ------------SCAN requests-------------------
class ScanRequest : public LookoutRequest {
public:
  class Task;
  virtual HeadMotionType_t getHeadMotionType() const { return scan; }
  enum ScanType_t { line,area };
  virtual ScanType_t getScanType() const = 0;
  vector<Task*> tasks;
  float scanSpeed; // in rad/msec
  static const float defSpd;
  ScanRequest(const ScanRequest& req)
    : LookoutRequest(req), tasks(), scanSpeed(req.scanSpeed) {
    for (vector<Task*>::const_iterator it = req.tasks.begin();
	 it != req.tasks.end(); it++) 
      addTask(**it);
  }
  ScanRequest(float _speed=defSpd)
    : tasks(), scanSpeed(_speed) {}
  ScanRequest(const Task& _task, float _speed=defSpd)
    : tasks(), scanSpeed(_speed) { addTask(_task); }
  void addTask(const Task& t) {
    switch (t.getTaskType()) {
    case Task::visReg:
      tasks.push_back(new VisionRegionTask(*dynamic_cast<const VisionRegionTask*>(&t))); break;
    case Task::visObj:
      tasks.push_back(new VisionObjectTask(*dynamic_cast<const VisionObjectTask*>(&t))); break;
    case Task::ir:
      tasks.push_back(new IRTask(*dynamic_cast<const IRTask*>(&t))); break;
    default: return;
    };
  }
  virtual ~ScanRequest() {
    for (vector<Task*>::const_iterator it = tasks.begin();
	 it != tasks.end(); it++)
      delete *it;
  }

  // ------------Tasks may be implemented during scan -------------------
  class Task {
  public:
    enum taskType_t { none, visObj, visReg, ir };
    virtual taskType_t getTaskType() const = 0;// { return none; }
    virtual ~Task() {}
    AngPi interval; //!< interval of measurements
    vector<Point> data; //!< measured data stored here in base frame coordinates
    Task(const Task& t) : interval(t.interval), data(t.data) {}
  protected:
    Task(AngPi _interval) : interval(_interval), data() {}
  private:
    Task& operator=(const Task&);
  };

  class IRTask : public Task {
  public:
    IRTask(AngPi _interval) : Task(_interval) {}
    IRTask(const IRTask& t) : Task(t) {}
    virtual taskType_t getTaskType() const { return ir; }
    //  virtual ~IRTask() { cout << "deleting IRTask " << this << endl; }
  };

  //base class for vision tasks, should not be instantiated
  class VisionTask : public Task {
  public:
    virtual taskType_t getTaskType() const { return none; }
    set<int> index;
    VisionTask(const VisionTask& vt) : Task(vt), index(vt.index) {}
  protected:
    VisionTask(const set<int>& _index, AngPi _interval)
      : Task(_interval), index(_index) {}
    VisionTask(int _index, AngPi _interval)
      : Task(_interval), index() { index.insert(_index); }
  };

  //! Uses bult-in object detectors (like pink ball detector) via VisionObjectEvent stream
  class VisionObjectTask : public VisionTask {
  public:
    VisionObjectTask(const set<int>& sid, AngPi _interval)
      : VisionTask(sid,_interval) {}
    VisionObjectTask(const VisionObjectTask& vot) : VisionTask(vot) {}
    virtual taskType_t getTaskType() const { return visObj; }
  };

  //! Uses built-in colored region detectors via Region event stream
  class VisionRegionTask : public VisionTask {
  public:
    VisionRegionTask(const set<int>& colorIndex, AngPi _interval,
		     unsigned int _minArea=200)
      : VisionTask(colorIndex,_interval), minArea(_minArea) {}
    VisionRegionTask(int colorIndex, AngPi _interval,
		     unsigned int _minArea=200)
      : VisionTask(colorIndex,_interval), minArea(_minArea) {}
    VisionRegionTask(const VisionRegionTask& vrt)
      : VisionTask(vrt), minArea(vrt.minArea) {}
    virtual taskType_t getTaskType() const { return visReg; }
    unsigned int minArea;
  };

};

class ScanAlongLineRequest : public ScanRequest {
public:
  ScanAlongLineRequest(const Point& bPt, const Point& ePt, float _speed=defSpd)
    : ScanRequest(_speed), beginPt(bPt), endPt(ePt) {}
  ScanAlongLineRequest(const ScanAlongLineRequest& req)
    : ScanRequest(req), beginPt(req.beginPt), endPt(req.endPt) {}
  //  virtual HeadMotionType_t getHeadMotionType() const { return scanLine; }
  virtual ScanType_t getScanType() const { return line; }
  Point beginPt, endPt;
};

class ScanAreaRequest : public ScanRequest {
public:
  ScanAreaRequest(coordinate_t _left, coordinate_t _right,
		  coordinate_t _far, coordinate_t _near, 
		  float _speed=defSpd)
    : ScanRequest(_speed),left(_left),
      right(_right),far(_far),near(_near) {}
  ScanAreaRequest(const ScanAreaRequest& req)
    : ScanRequest(req), left(req.left),
      right(req.right), far(req.far), near(req.near) {}
  //  virtual HeadMotionType_t getHeadMotionType() const { return scanArea; }
  virtual ScanType_t getScanType() const { return area; }
  coordinate_t left,right,far,near;
};
//--------------------------------------------


// ------------TRACK requests-------------------
class TrackRequest : public LookoutRequest {
public:
  TrackRequest(const vector<ScanRequest::VisionTask>& _landmarks, const ShapeRoot& _area)
    : landmarks(_landmarks), area(_area), lmLocation(), lmInView(false) {}
  TrackRequest(const vector<ScanRequest::VisionTask>& _landmarks, const Point& pt, const ShapeRoot& _area)
    : landmarks(_landmarks), area(_area), lmLocation(pt), lmInView(false) {}
  virtual HeadMotionType_t getHeadMotionType() const { return track; }
  vector<ScanRequest::VisionTask> landmarks;
  ShapeRoot area;
  Point lmLocation;
  bool lmInView;
};
//--------------------------------------------

} // namespace

#endif
