//-*-c++-*-
#ifndef _BLOBDATA_H_
#define _BLOBDATA_H_

#include <vector>
#include <set>
#include <iostream>
#include <string>

#include "Shared/newmat/newmat.h"
#include "Vision/cmv_types.h"

#include "BaseData.h"      // superclass
#include "Point.h"          // Point data member
#include "ShapeTypes.h"   // blobDataType
#include "ShapeFuns.h"

namespace DualCoding {

class ShapeRoot;
template<typename T> class Sketch;

//! Blob shapes, described by bounding boxes and an optional list of runs.

class BlobData : public BaseData {
 public:

  // Bounding quadrilateral; may not be square when projected to world space.
  Point topLeft, topRight, bottomLeft, bottomRight;

  float area;

  struct run {
  public:
    unsigned short int x, y, width;
    run() : x(0), y(0), width(0) {}
    run(unsigned short int _x, unsigned short int _y, unsigned short int _width) :
      x(_x), y(_y), width(_width) {}
  };

  const std::vector<run> runvec;

  //! Assumed orientation of the blob in 3D space.
  enum BlobOrientation_t {
    groundplane,  //!< 2D shape lying flat on the ground
    pillar,       //!< 3D shape standing on the ground
    pinata       //!< 3D shape hanging vertically in space
  } orientation;

 public:
  //! Constructor
  BlobData(ShapeSpace& _space,
	   const Point &_topLeft, const Point &_topRight,
	   const Point &_bottomLeft, const Point &_bottomRight,
	   const float _area, 
	   const std::vector<run> &_runvec, 
	   const BlobOrientation_t _orientation,
	   const rgb rgbvalue);

  static ShapeType_t getStaticType() { return blobDataType; }

  DATASTUFF_H(BlobData);

  friend class Shape<BlobData>;

  //! return the centroid of the shape in point format
  virtual Point getCentroid() const;
  
  //! Area of the blob
  float getArea() { return area; }

  //! Print information about this shape.
  virtual void printParams() const;

  //! Transformations. (Virtual in BaseData.)
  virtual void applyTransform(const NEWMAT::Matrix& Tmat);
  
  //! Project to ground
  //  virtual void projectToGround(int xres, int yres, const NEWMAT::ColumnVector& groundplane);
  virtual void projectToGround(const NEWMAT::Matrix& camToBase,
			       const NEWMAT::ColumnVector& groundplane);

  //! Update derived properties
  virtual void update_derived_properties();

  //! Match blobs based on their parameters.  (Virtual in BaseData.)
  virtual bool isMatchFor(const ShapeRoot& other) const;

  virtual bool updateParams(const ShapeRoot& other, bool forceUpdate=false);

  virtual unsigned short getDimension() const { return (orientation==groundplane) ? 2 : 3; }


  //! Import blobs from Sketch<bool> as a vector of Shape<BlobData>
  static std::vector<Shape<BlobData> > 
  extractBlobs(const Sketch<bool> &sketch, 
	       const set<int>& colors, int minarea=0,
	       BlobData::BlobOrientation_t orient=BlobData::groundplane, 
	       int maxblobs=50);
  static std::vector<Shape<BlobData> > 
  extractBlobs(const Sketch<bool> &sketch, int minarea=0,
	       BlobData::BlobOrientation_t orient=BlobData::groundplane, 
	       int maxblobs=50); 

  //! Import blobs from Sketch<uchar> as a vector of Shape<BlobData>
  static std::vector<Shape<BlobData> >
  extractBlobs(const Sketch<CMVision::uchar> &sketch, 
	       const set<int>& colors, int minarea=0,
	       BlobData::BlobOrientation_t orient=BlobData::groundplane,
	       int maxblobs=50);
  static std::vector<Shape<BlobData> >
  extractBlobs(const Sketch<CMVision::uchar> &sketch, int minarea=0,
	       BlobData::BlobOrientation_t orient=BlobData::groundplane,
	       int maxblobs=50);

  static BlobData* 
  new_blob(ShapeSpace& space,
	   const CMVision::region &reg, 
	   const CMVision::run<CMVision::uchar> *rle_buff,
	   const BlobData::BlobOrientation_t orient,
	   const rgb rgbvalue);

  std::vector<Point> findCorners(unsigned int nExpected, std::vector<Point>& candidates, float &bestValue);

  std::vector<Point> findCornersDerivative();

  std::vector<Point> findCornersDiagonal();

  std::vector<Point> findCornersShapeFit(unsigned int ncorners, std::vector<Point>& candidates, float &bestValue);


 // comparison predicates

  class areaLessThan : public BinaryShapePred<BlobData> {
  public:
    bool operator() (const Shape<BlobData> &b1, const Shape<BlobData> &b2) const;
  };

private:
  //! Render into a sketch space and return reference. (Private.)
  virtual Sketch<bool>* render() const;

  BlobData& operator=(const BlobData&); //!< don't call

};

} // namespace

#endif // BLOBDATA_H_
