//-*-c++-*-
#ifndef INCLUDED_EventLogger_h_
#define INCLUDED_EventLogger_h_

#include "ControlBase.h"
#include "Events/EventListener.h"
#include <fstream>
#include <set>
#include <queue>

class FilterBankGenerator;
class BehaviorBase;
class StateNode;

//! allows logging of events to the console or a file
class EventLogger : public ControlBase, public EventListener {
public:
	//!constructor
	EventLogger();
	//!destructor
	virtual ~EventLogger();

	//!opens a custom (embedded) menu to toggle individual EGIDs
	virtual ControlBase* doSelect();
	
	virtual void refresh();

	//!sends all events received to stdout and/or logfile
	virtual void processEvent(const EventBase& event);

	//!returns #logSocket
	static class Socket* getLogSocket() { return logSocket; }
	
	//! returns #port
	static int getLogSocketPort() { return port; }
	
	//! sets #port
	static void setLogSocketPort(int p) { port=p; }

	//! send the current camera image over the log socket
	static void logImage(FilterBankGenerator& fbg, unsigned int layer, unsigned int channel, const BehaviorBase* source=NULL);

	//! send a string over the log socket
	static void logMessage(std::string msg, const BehaviorBase* source=NULL, const char* icon=NULL, unsigned int placement=0);
	
	//! request that the desktop side take a picture with the webcam (if available)
	static void logWebcam(const BehaviorBase* source=NULL);
	
	static int callback(char *buf, int bytes); //!< called by wireless when there's new data

protected:
	static EventLogger * theOne; //!< the instance which will handle network communication

	//! a separate processEvent to distinguish between events requested for logging and events requested by a remote monitor
	class StateMachineListener : public EventListener {
		//! forwards any events received to EventLogger::theOne's EventLogger::processStateMachineEvent()
		/*! EventLogger::runCommand() is responsible for maintaining which events this is listening to */
		virtual void processEvent(const EventBase& event) {
			EventLogger::theOne->processStateMachineEvent(event);
		}
	};
	static class StateMachineListener smProcess; //!< handles state machine transitions if the Storyboard GUI (or other remote monitor) is listening for state machine events

	virtual void clearSlots();

	//!sets the status char of slot @a i to @a c
	void setStatus(unsigned int i, char c);

	//!checks to see if logfilePath differs from the StringInputControl's value and switches it if it is
	void checkLogFile();
	
	//! dumps all of the transitions and subnodes of a given statenode
	void spider(const StateNode* n, unsigned int depth=0);

	//! returns true iff @a n or one of its parents is found in #listen
	bool isListening(const StateNode* n);

	//! parses commands sent from callback()
	void runCommand(const std::string& s);

	//!just to prettify the data sent out - probably should make this a null-op to save bandwidth after debugging is done
	void indent(unsigned int level);
	
	//!searches currently instantiated StateNodes to find the one named @a name
	const StateNode * find(const std::string& name);

	//!if there is a remote monitor listening for state machine transitions, this will send them over
	/*!this is called by the StateMachineListener, which is subscribed to only
	 * those machines which have been requested by the remote monitor */
	virtual void processStateMachineEvent(const EventBase& event);

	//!address of the logfile, if any (empty string is no logfile)
	std::string logfilePath;

	//!if a filename is given, events are logged to here
	std::ofstream logfile;
	
	//! events which are logged will be sent over this port in an xml format.  See eventlog.dtd in the docs directory
	static class Socket* logSocket;
	
	//! port number #logSocket will listen on
	static int port;
	
	//! reference count for #logSocket -- when this hits 0, close the socket
	static unsigned int logSocketRefCount;
	
	//!controls the level of verbosity - currently 0 through 2
	unsigned int verbosity;

	typedef std::set<BehaviorBase*> registry_t; //!< the type of the behavior registry (BehaviorBase::registry)

	typedef std::multiset<const StateNode*> expected_t; //!< the type of #expected
	expected_t expected; //!< a set of behaviors which are involved with an impending transition - their next stateMachineEGID event should be ignored

	typedef std::set<std::string> listen_t; //!< the type of #listen
	listen_t listen; //!< a set of state machine names which should have their subnodes monitored

	typedef std::queue<EventBase> queuedEvents_t; //!< the type of #queuedEvents
	queuedEvents_t queuedEvents; //!< used if a transition causes other transitions, those transitions need to be remembered
};

/*! @file
 * @brief Describes EventLogger, which allows logging of events to the console or a file
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.9 $
 * $State: Exp $
 * $Date: 2006/09/16 06:28:06 $
 */

#endif
