#include "EventLogger.h"
#include "Events/EventRouter.h"
#include "Motion/MMAccessor.h"
#include "Motion/LedMC.h"
#include "ValueEditControl.h"
#include "StringInputControl.h"
#include "NullControl.h"
#include <sstream>
#include "Sound/SoundManager.h"
#include "Vision/FilterBankGenerator.h"
#include "Vision/JPEGGenerator.h"
#include "Shared/Base64.h"
#include "Behaviors/StateNode.h"

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

Socket* EventLogger::logSocket=NULL;
unsigned int EventLogger::logSocketRefCount=0;
int EventLogger::port=10080;
EventLogger * EventLogger::theOne=NULL;

EventLogger::StateMachineListener EventLogger::smProcess;

EventLogger::EventLogger()
	: ControlBase("Event Logger","Allows you to see/log all of the un-trapped events as they are generated"),
		logfilePath(), logfile(), verbosity(0), expected(), listen(), queuedEvents() {
	for(unsigned int i=0; i<EventBase::numEGIDs; i++) {
		std::string tmp=EventBase::EventGeneratorNames[i];
		pushSlot(new NullControl(("[ ] "+tmp).c_str(),"Show/hide events from "+tmp));
	}
	pushSlot(NULL);
	pushSlot(new ValueEditControl<unsigned int>("Verbosity","Controls verbosity level: 0=(gen,source,type); 1=0+gen_id,source_id,type_id; 2=1+duration,timestamp; 3=2+magnitude; additional columns may be added for subclass info","Please enter a new verbosity level: 0=(gen,source,type); 1=0+gen_id,source_id,type_id; 2=1+duration,timestamp; 3=2+magnitude; additional columns may be added for subclass info",&verbosity));
	pushSlot(new ControlBase("[X] Console Output","If selected, outputs events to the console"));
	pushSlot(new StringInputControl("[ ] File Output","Please enter the filename to log to (in /ms/...)"));
	if(logSocket==NULL) {
		theOne=this;
		ASSERT(logSocketRefCount==0,"logSocket is NULL, ref count is non-zero");
		logSocket=wireless->socket(SocketNS::SOCK_STREAM,1024,1<<15);
		wireless->setDaemon(logSocket);
		wireless->setReceiver(logSocket, callback);
		wireless->listen(logSocket,port);
	}
	logSocketRefCount++;
}

EventLogger::~EventLogger() {
	expected.clear();
	while(!queuedEvents.empty())
		queuedEvents.pop();
	clearSlots();
	if(--logSocketRefCount==0) {
		wireless->setDaemon(logSocket,false);
		wireless->close(logSocket);
		logSocket=NULL;
	}
	if(theOne==this)
		theOne=NULL;
}

ControlBase* EventLogger::doSelect() {
	ControlBase* ans=this;
	for(unsigned int i=0; i<hilights.size(); i++) {
		unsigned int cur=hilights[i];
		if(cur<EventBase::numEGIDs) {
			if(options[cur]->getName()[1]!=' ') {
				erouter->removeListener(this,(EventBase::EventGeneratorID_t)(cur));
				setStatus(cur,' ');
			} else {
				erouter->addListener(this,(EventBase::EventGeneratorID_t)(cur));
				setStatus(cur,'X');
			}
		} else if(cur==EventBase::numEGIDs+1) {
			ans=options[cur];
		} else if(cur==EventBase::numEGIDs+2) {
			if(options[cur]->getName()[1]!=' ') {
				setStatus(cur,' ');
			} else {
				setStatus(cur,'X');
			}
		} else if(cur==EventBase::numEGIDs+3) {
			if(options[cur]->getName()[1]!=' ') {
				logfile.close();
				options[cur]->setName("[ ] File Output");
			} else {
				ans=options[cur];
			}
		}
		sndman->playFile(config->controller.select_snd);
	}
	if(ans==this)
		refresh();
	return ans;
}

void EventLogger::refresh() {
	checkLogFile();
	ControlBase::refresh();
}

//!sends all events received to stdout and/or logfile
void EventLogger::processEvent(const EventBase& event) {
	std::string logdata = event.getDescription(true,verbosity);
	if(options[EventBase::numEGIDs+2]->getName()[1]=='X')
		sout->printf("EVENT: %s\n",logdata.c_str());
	if(logSocket!=NULL && wireless->isConnected(logSocket->sock)) {
		xmlDoc * doc = xmlNewDoc((const xmlChar*)"1.0");
		xmlNode * cur = xmlNewNode(NULL,(const xmlChar*)"");
		xmlSetProp(cur,(const xmlChar*)"type",(const xmlChar*)"log");
		xmlNode * desc = xmlNewNode(NULL,(const xmlChar*)"param");
		event.saveXML(cur);
		xmlAddChild(cur,desc);
		xmlSetProp(desc,(const xmlChar*)"name",(const xmlChar*)"description");
		xmlSetProp(desc,(const xmlChar*)"value",(const xmlChar*)event.getDescription(true,3).c_str());
		xmlBuffer* buf=xmlBufferCreate();
		int n=xmlNodeDump(buf,doc,cur,0,1);
		xmlFreeDoc(doc);
		byte * nbuf = logSocket->getWriteBuffer(n+1);
		if(nbuf!=NULL) {
			memcpy(nbuf,xmlBufferContent(buf),n);
			nbuf[n]='\n';
			logSocket->write(n+1);
		}
		xmlBufferFree(buf);
	}
	checkLogFile();
	if(logfile)
		logfile << logdata << endl;
}

void EventLogger::logImage(FilterBankGenerator& fbg, unsigned int layer, unsigned int channel, const BehaviorBase* source/*=NULL*/) {
	if(logSocket!=NULL && wireless->isConnected(logSocket->sock)) {

		char * binbuf;
		unsigned int len;
		if(JPEGGenerator* jpeg=dynamic_cast<JPEGGenerator*>(&fbg)) {
			binbuf=(char*)jpeg->getImage(layer,channel);
			len=jpeg->getImageSize(layer,channel);
		} else {
			fbg.selectSaveImage(layer,channel);
			len=fbg.getBinSize();
			binbuf=new char[len];
			fbg.saveBuffer(binbuf,len);
		}
		string b64buf=base64::encode(binbuf,len);
		if(binbuf!=(char*)fbg.getImage(layer,channel)) //cached, should be a simple return
			delete [] binbuf;
		
		xmlDoc * doc = xmlNewDoc((const xmlChar*)"1.0");
		xmlNode * cur = xmlNewNode(NULL,(const xmlChar*)"event");
		xmlSetProp(cur,(const xmlChar*)"type",(const xmlChar*)"image");
		if(source!=NULL)
			xmlSetProp(cur,(const xmlChar*)"sid",(const xmlChar*)source->getName().c_str());
		char timebuf[20];
		snprintf(timebuf,20,"%d",get_time());
		xmlSetProp(cur,(const xmlChar*)"time",(const xmlChar*)timebuf);
		xmlNewChild(cur,NULL,(const xmlChar*)"image",(const xmlChar*)b64buf.c_str());
		xmlBuffer* buf=xmlBufferCreate();
		int n=xmlNodeDump(buf,doc,cur,0,1);
		xmlFreeDoc(doc);
		byte * nbuf = logSocket->getWriteBuffer(n+1);
		if(nbuf!=NULL) {
			memcpy(nbuf,xmlBufferContent(buf),n);
			nbuf[n]='\n';
			logSocket->write(n+1);
		}
		xmlBufferFree(buf);
	}		
}

void EventLogger::logMessage(std::string msg, const BehaviorBase* source/*=NULL*/, const char* icon/*=NULL*/, unsigned int placement/*=0*/) {
	if(logSocket!=NULL && wireless->isConnected(logSocket->sock)) {
		xmlDoc * doc = xmlNewDoc((const xmlChar*)"1.0");
		xmlNode * cur = xmlNewNode(NULL,(const xmlChar*)"event");
		xmlSetProp(cur,(const xmlChar*)"type",(const xmlChar*)"userlog");
		if(source!=NULL)
			xmlSetProp(cur,(const xmlChar*)"sid",(const xmlChar*)source->getName().c_str());
		if(icon!=NULL)
			xmlSetProp(cur,(const xmlChar*)"icon",(const xmlChar*)icon);
		const unsigned int len=20;
		char sbuf[len];
		snprintf(sbuf,len,"%d",placement);
		xmlSetProp(cur,(const xmlChar*)"voff",(const xmlChar*)sbuf);
		snprintf(sbuf,len,"%d",get_time());
		xmlSetProp(cur,(const xmlChar*)"time",(const xmlChar*)sbuf);
		xmlNodeSetContent(cur,(const xmlChar*)msg.c_str());
		xmlBuffer* buf=xmlBufferCreate();
		int n=xmlNodeDump(buf,doc,cur,0,1);
		xmlFreeDoc(doc);
		byte * nbuf = logSocket->getWriteBuffer(n+1);
		if(nbuf!=NULL) {
			memcpy(nbuf,xmlBufferContent(buf),n);
			nbuf[n]='\n';
			logSocket->write(n+1);
		}
		xmlBufferFree(buf);
	}		
}

void EventLogger::logWebcam(const BehaviorBase* source/*=NULL*/) {
	if(logSocket!=NULL && wireless->isConnected(logSocket->sock)) {
		xmlDoc * doc = xmlNewDoc((const xmlChar*)"1.0");
		xmlNode * cur = xmlNewNode(NULL,(const xmlChar*)"event");
		xmlSetProp(cur,(const xmlChar*)"type",(const xmlChar*)"webcam");
		if(source!=NULL)
			xmlSetProp(cur,(const xmlChar*)"sid",(const xmlChar*)source->getName().c_str());
		const unsigned int len=20;
		char sbuf[len];
		snprintf(sbuf,len,"%d",get_time());
		xmlSetProp(cur,(const xmlChar*)"time",(const xmlChar*)sbuf);
		xmlNodeSetContent(cur,(const xmlChar*)" ");
		xmlBuffer* buf=xmlBufferCreate();
		int n=xmlNodeDump(buf,doc,cur,0,1);
		xmlFreeDoc(doc);
		byte * nbuf = logSocket->getWriteBuffer(n+1);
		if(nbuf!=NULL) {
			memcpy(nbuf,xmlBufferContent(buf),n);
			nbuf[n]='\n';
			logSocket->write(n+1);
		}
		xmlBufferFree(buf);
	}		
}

void EventLogger::clearSlots() {
	erouter->removeListener(this);
	ControlBase::clearSlots();
}

void EventLogger::setStatus(unsigned int i, char c) {
	std::string tmp=options[i]->getName();
	tmp[1]=c;
	options[i]->setName(tmp);
}

void EventLogger::checkLogFile() {
	unsigned int cur=EventBase::numEGIDs+3;
	StringInputControl * strin=dynamic_cast<StringInputControl*>(options[cur]);
	ASSERTRET(strin!=NULL,"The StringInputControl is misplaced");
	if(strin->getLastInput()!=logfilePath) {
		logfile.close();
		logfilePath=strin->getLastInput();
		logfile.clear();
		if(logfilePath.size()!=0) {
			sout->printf("Opening `%s'\n",(config->portPath(logfilePath)).c_str());
			logfile.open((config->portPath(logfilePath)).c_str());
			if(!logfile.fail()) {
				setStatus(cur,'X');
				strin->setName(strin->getName()+": "+logfilePath);
			} else {
				serr->printf("Opening `%s' failed\n",(config->portPath(logfilePath)).c_str());
			}
		}
	}
}


void EventLogger::spider(const StateNode* n, unsigned int depth/*=0*/) {
	if(n==NULL)
		return;

	const std::vector<StateNode*>& subnodes=n->getNodes();
	if(subnodes.size()==0) {
		// it's a leaf node, no subnodes or transitions between them
		indent(depth);
		logSocket->printf("<state class=\"%s\" id=\"%s\" />\n", n->getClassName().c_str(), n->getName().c_str());
	} else {

		// first output current node's info
		indent(depth);
		logSocket->printf("<state class=\"%s\" id=\"%s\">\n", n->getClassName().c_str(), n->getName().c_str());

		std::set<const Transition*> transitions;
		// now recurse on sub-nodes, extracting all of the subnodes transitions
		for(unsigned int i=0; i<subnodes.size(); i++) {
			spider(subnodes[i],depth+1);
			const std::vector<Transition*>& curt=subnodes[i]->getTransitions();
			transitions.insert(curt.begin(),curt.end());
		}

		// now output transitions between subnodes we collected in previous step
		for(std::set<const Transition*>::const_iterator it=transitions.begin(); it!=transitions.end(); it++) {
			indent(depth+1);
			logSocket->printf("<transition class=\"%s\" id=\"%s\">\n", (*it)->getClassName().c_str(), (*it)->getName().c_str());
			const std::vector<StateNode*>& incoming=(*it)->getSources();
			for(unsigned int i=0; i<incoming.size(); i++) {
				indent(depth+2);
				logSocket->printf("<source>%s</source>\n",incoming[i]->getName().c_str());
			}
			const std::vector<StateNode*>& outgoing=(*it)->getDestinations();
			for(unsigned int i=0; i<outgoing.size(); i++) {
				indent(depth+2);
				logSocket->printf("<destination>%s</destination>\n",outgoing[i]->getName().c_str());
			}
			indent(depth+1);
			logSocket->printf("</transition>\n");
		}

		indent(depth);
		logSocket->printf("</state>\n");
	}
}
	
bool EventLogger::isListening(const StateNode* n) {
	while(n!=NULL) {
		if(listen.find(n->getName())!=listen.end())
			return true;
		n=n->getParent();
	}
	return false;
}

void EventLogger::indent(unsigned int level) {
	for(unsigned int i=0; i<level; i++)
		logSocket->printf("  ");
}

const StateNode * EventLogger::find(const std::string& sname) {
	const registry_t& registry=BehaviorBase::getRegistry();
	for(registry_t::const_iterator it=registry.begin(); it!=registry.end(); it++) {
		const StateNode * cur=dynamic_cast<const StateNode*>(*it);
		if(cur!=NULL && cur->getName()==sname)
			return cur;
	}
	//serr->printf("WARNING: EventLogger Could not find StateNode named `%s'\n",sname.c_str());
	return NULL;
}

void EventLogger::runCommand(const std::string& s) {
	if(s==std::string("list")) {
		const registry_t& registry=BehaviorBase::getRegistry();
		unsigned int numstate=0;
		for(registry_t::const_iterator it=registry.begin(); it!=registry.end(); it++) {
			const StateNode * cur=dynamic_cast<const StateNode*>(*it);
			if(cur!=NULL)
				numstate++;
		}
		logSocket->printf("%d\n",numstate);
		for(registry_t::const_iterator it=registry.begin(); it!=registry.end(); it++) {
			const StateNode * cur=dynamic_cast<const StateNode*>(*it);
			if(cur!=NULL)
				logSocket->printf("%s\n",cur->getName().c_str());
		}

	} else if(s.find("spider ")==0) {
		const StateNode * n=find(s.substr(7));
		if(n==NULL) {
			serr->printf("WARNING: EventLogger could not find \"%s\" for spidering\n",s.substr(7).c_str());
			logSocket->printf("<model></model>\n");
		} else {
			logSocket->printf("<model>\n");
			spider(n);
			logSocket->printf("</model>\n");
		}

	} else if(s.find("listen ")==0) {
		if(listen.size()==0) {
			erouter->addListener(&smProcess,EventBase::stateMachineEGID);
			erouter->addListener(&smProcess,EventBase::stateTransitionEGID);
		}
		listen.insert(s.substr(7));

	} else if(s.find("ignore ")==0) {
		listen.erase(s.substr(7));
		if(listen.size()==0)
			erouter->removeListener(&smProcess);

	} else if(s=="clear") {
		listen.clear();
		erouter->removeListener(&smProcess);

	} else {
		serr->printf("EventLogger::runCommand() - bad message: '%s'\n",s.c_str());
	}
}

// The command packet reassembly mechanism
int EventLogger::callback(char *buf, int bytes) {
	if(EventLogger::theOne==NULL)
		return 0;
	static std::string cmd;
	for(int i=0; i<bytes; i++) {
		if(buf[i]=='\n') {
			EventLogger::theOne->runCommand(cmd);
			cmd.clear();
		} else if(buf[i]!='\r')
			cmd+=buf[i];
	}
  return 0;
}

void EventLogger::processStateMachineEvent(const EventBase& event) {
	if(!wireless->isConnected(logSocket->sock) || listen.size()==0)
		return;

	if(event.getGeneratorID()==EventBase::stateTransitionEGID) {
		bool care=false;
		const Transition * trans = reinterpret_cast<Transition*>(event.getSourceID());
		const std::vector<StateNode*>& incoming=trans->getSources();
		const std::vector<StateNode*>& outgoing=trans->getDestinations();
		for(std::vector<StateNode*>::const_iterator it=incoming.begin(); it!=incoming.end() && !care; it++)
			care=isListening(*it);
		for(std::vector<StateNode*>::const_iterator it=outgoing.begin(); it!=outgoing.end() && !care; it++)
			care=isListening(*it);
		if(!care)
			return;

		if(expected.size()!=0) {
			queuedEvents.push(event);
		} else {
			logSocket->printf("<event>\n");
			indent(1);
			logSocket->printf("<fire id=\"%s\" time=\"%d\" />\n",trans->getName().c_str(),event.getTimeStamp());
			expected.insert(incoming.begin(),incoming.end());
			expected.insert(outgoing.begin(),outgoing.end());
			while(queuedEvents.size()>0) {
				EventBase qe=queuedEvents.front();
				queuedEvents.pop();
				processEvent(qe);
			}
		}

	} else if(event.getGeneratorID()==EventBase::stateMachineEGID) {
		if(event.getTypeID()==EventBase::statusETID)
			return;
		const StateNode * beh=reinterpret_cast<StateNode*>(event.getSourceID());
		expected_t::iterator it=expected.find(beh);
		char * format;
		if(isListening(beh)) {
			if(it==expected.end()) { //if not found
				if(queuedEvents.size()==0)
					format="<event><state%s id=\"%s\" time=\"%d\" /></event>\n"; // unexpected
				else {
					queuedEvents.push(event);
					return;
				}
			} else
				format="  <state%s id=\"%s\" time=\"%d\" />\n"; // expected as part of transition
			if(event.getTypeID()==EventBase::activateETID)
				logSocket->printf(format,"start",beh->getName().c_str(),event.getTimeStamp());
			else if(event.getTypeID()==EventBase::deactivateETID)
				logSocket->printf(format,"stop",beh->getName().c_str(),event.getTimeStamp());
			else
				serr->printf("WARNING: Unrecognized TypeID %d\n",event.getTypeID());
		}
		if(it!=expected.end()) { //was found
			expected.erase(it);
			if(expected.size()==0) {
				logSocket->printf("</event>\n");
				while(queuedEvents.size()>0) {
					EventBase qe=queuedEvents.front();
					queuedEvents.pop();
					processEvent(qe);
				}
			}
		}

	} else {
		serr->printf("WARNING: Unknown event %s (%s)\n",event.getName().c_str(),event.getDescription().c_str());
	}
}





/*! @file
 * @brief Describes EventLogger, which allows logging of events to the console or a file
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-3_0 $
 * $Revision: 1.22 $
 * $State: Exp $
 * $Date: 2006/09/18 18:07:54 $
 */
