#include "Simulator.h"
#include "IPC/RegionRegistry.h"
#include "Shared/string_util.h"
#include "Shared/RobotInfo.h"
#include <iostream>
#include "SimConfig.h"

using namespace std;

Simulator::Simulator()
: Process(getID(),getClassName()),
cameraQueue(ipc_setup->registerRegion(Simulator::getCameraQueueID(),sizeof(sim::CameraQueue_t))),
sensorQueue(ipc_setup->registerRegion(Simulator::getSensorQueueID(),sizeof(sim::SensorQueue_t))),
vision("images", ".*\\.jpg", 30, *cameraQueue),
sensors("sensors", ".*\\.txt", 1000.f/NumFrames/FrameTime, *sensorQueue),
visionLayer(-2)
{
	new (&(*cameraQueue)) sim::CameraQueue_t;
	new (&(*sensorQueue)) sim::SensorQueue_t;
	sim::config.writeParseTree();
	sim::config.addEntry("Sensors",sensors);
	sim::config.addEntry("Vision",vision);
	vision.addEntry("Layer",visionLayer,"Controls at what resolution layer the image should be processed.\n0 and positive numbers indicate the resolution layer directly as it would be accessed by behaviors.\nNegative values are relative to the number of layers marked available by the vision setup, so that typically -1 would correspond to the \"double\" layer, and -2 would correspond to the \"full\" layer.");
	sim::config.readParseTree();
	for(unsigned int i=0; i<sim::assignments.size(); i++) {
		vector<string> setarg;
		setarg.push_back("set");
		setarg.push_back(sim::assignments[i]);
		cmdSet(setarg);
	}
	if(globals->timeScale<0) {
		cameraQueue->setOverflowPolicy(MessageQueueBase::WAIT);
		sensorQueue->setOverflowPolicy(MessageQueueBase::WAIT);
	} else {
		cameraQueue->setOverflowPolicy(MessageQueueBase::DROP_OLDEST);
		sensorQueue->setOverflowPolicy(MessageQueueBase::DROP_OLDEST);
	}
	globals->timeScale.addPrimitiveListener(this);
	processRunlevel(SharedGlobals::CONSTRUCTING);
}

Simulator::~Simulator() {
	globals->timeScale.removePrimitiveListener(this);
	processRunlevel(SharedGlobals::DESTRUCTING);
}

void Simulator::DoStart() {
	Process::DoStart();
	processRunlevel(SharedGlobals::STARTING);
}

void Simulator::run() {
	for(unsigned int i=0; i<ProcessID::NumProcesses; ++i)
		cout << globals->processNames[i] << " pid=" << globals->pids[i] << ";  ";
	cout << endl;

	vision.addStatusListener(this);
	vision.addStatusListener(this);
	vision.loadFileList();
	vision.start();

	processRunlevel(SharedGlobals::RUNNING);

	vision.stop();
	vision.removeStatusListener(this);
	vision.removeStatusListener(this);

	globals->signalShutdown();
	cout << "So long buddy!" << endl;
}

void Simulator::plistValueChanged(const plist::PrimitiveBase& pl) {
	if(&pl==&globals->timeScale) {
		get_time(); // force SharedGlobals to notice the change and update its state
		if(globals->timeScale<0) {
			cameraQueue->setOverflowPolicy(MessageQueueBase::WAIT);
			sensorQueue->setOverflowPolicy(MessageQueueBase::WAIT);
			unsigned int next=std::min(vision.nextTimestamp(),sensors.nextTimestamp());
			if(next!=-1U)
				globals->simulatorTime=next;
		} else {
			cameraQueue->setOverflowPolicy(MessageQueueBase::DROP_OLDEST);
			sensorQueue->setOverflowPolicy(MessageQueueBase::DROP_OLDEST);
		}
	} else {
		cerr << "WARNING: Simulator got a plistValueChanged for an unknown plist primitive";
	}
}

void Simulator::messagesRead(MessageQueueBase& mq, unsigned int /*n*/) {
	if(globals->timeScale<0) {
		if(&mq==&(*cameraQueue)) {
			unsigned int next=vision.nextTimestamp();
			if(next!=-1U)
				globals->simulatorTime=next;
		} else if(&mq==&(*sensorQueue)) {
			unsigned int next=sensors.nextTimestamp();
			if(next!=-1U)
				globals->simulatorTime=next;
		}
	}
}

void Simulator::DoStop() {
	processRunlevel(SharedGlobals::STOPPING);
	Process::DoStop();
}

void Simulator::processRunlevel(SharedGlobals::runlevel_t curLevel) {
	while(sim::config.tgtRunlevel==curLevel && (!globals->isShutdown() || curLevel>SharedGlobals::RUNNING)) {
		string line;
		cout << sim::config.cmdPrompt << flush;
		getline(cin,line);
		if(line=="shutdown" || line=="quit" || line=="exit" || !cin) {
			sim::config.tgtRunlevel=SharedGlobals::DESTRUCTED;
			break;
		}
		vector<string> args;
		vector <unsigned int> offs;
		if(!string_util::parseArgs(line,args,offs)) {
			cerr << "Mismatched quotes" << endl;
			continue;
		}
		if(args.size()==0)
			continue;
		if(args[0]=="load") {
			cmdLoad(args);
		} else if(args[0]=="save") {
			cmdSave(args);
		} else if(args[0]=="runlevel") {
			cmdRunlevel(args, curLevel);
		} else if(args[0]=="get_time") {
			cout << "Current time is " << get_time() << endl;
		} else if(args[0]=="set") {
			cmdSet(args);
		} else if(args[0]=="runto") {
			cmdRun(args,curLevel,false);
		} else if(args[0]=="runfor") {
			cmdRun(args,curLevel,true);
		} else if(args[0]=="run") {
			cmdRun(args,curLevel);
		} else if(args[0]=="pause") {
			cmdPause(args,curLevel);
		} else if(args[0]=="help") {
			cmdHelp(args);
		} else {
			cout << "Unknown command '" << args[0] << "'" << endl;
		}
	}
}

void Simulator::cmdLoad(const std::vector<std::string>& args) {
	if(args.size()>1)
		for(unsigned int i=1; i<args.size(); i++)
			sim::config.LoadFile(args[i].c_str());
	else
		sim::config.LoadFile(sim::config.getLastFile().c_str());
}
void Simulator::cmdSave(const std::vector<std::string>& args) {
	if(args.size()>1)
		for(unsigned int i=1; i<args.size(); i++)
			sim::config.SaveFile(args[i].c_str());
	else
		sim::config.SaveFile(sim::config.getLastFile().c_str());
}
void Simulator::cmdRunlevel(const std::vector<std::string>& args, SharedGlobals::runlevel_t curLevel) {
	if(args.size()==1) {
		sim::config.tgtRunlevel=static_cast<SharedGlobals::runlevel_t>(sim::config.tgtRunlevel+1);
		cout << "Moving to next runlevel: " << SharedGlobals::runlevel_names[sim::config.tgtRunlevel] << endl;
		return;
	}
	try {
		sim::config.tgtRunlevel=string_util::makeUpper(args[1]);
	} catch(...) {
		cout << "Invalid runlevel specification.  Try one of:\n\t";
		for(unsigned int i=0; i<SharedGlobals::NUM_RUNLEVELS; i++)
			cout << i << ' ' << SharedGlobals::runlevel_names[i] << ", ";
		cout << "\nCurrently at " << SharedGlobals::runlevel_names[curLevel] << endl;
		return;
	}
	if(sim::config.tgtRunlevel<curLevel) {
		sim::config.tgtRunlevel=curLevel;
		cout << "Cannot reduce runlevel, currently at " << curLevel << ' ' << SharedGlobals::runlevel_names[curLevel] << "\n\t";
		for(unsigned int i=0; i<SharedGlobals::NUM_RUNLEVELS; i++)
			cout << i << ' ' << SharedGlobals::runlevel_names[i] << ", ";
		cout << endl;
	} else if(sim::config.tgtRunlevel==curLevel) {
		cout << "Already at " << curLevel << ' ' << SharedGlobals::runlevel_names[curLevel] << "\n\t";
		for(unsigned int i=0; i<SharedGlobals::NUM_RUNLEVELS; i++)
			cout << i << ' ' << SharedGlobals::runlevel_names[i] << ", ";
		cout << endl;
	}
}
void Simulator::cmdSet(const std::vector<std::string>& args) {
	if(args.size()==1) {
		cout << sim::config << endl;
		return;
	}
	string arg;
	for(unsigned int i=1; i<args.size(); i++) {
		arg+=args[i];
		if(i!=args.size()-1)
			arg+=" ";
	}
	if(arg.rfind("=")==string::npos) {
		plist::ObjectBase* ob=sim::config.findEntry(arg);
		if(ob==NULL) {
			cout << "'" << arg << "' is unknown" << endl;
			return;
		}
		cout << *ob << endl;
	} else {
		string value=string_util::trim(arg.substr(arg.find("=")+1));
		string key=string_util::trim(arg.substr(0,arg.find("=")));
		plist::ObjectBase* ob=sim::config.findEntry(key);
		if(ob==NULL) {
			cout << "'" << key << "' is unknown" << endl;
			return;
		}
		if(plist::PrimitiveBase* pbp=dynamic_cast<plist::PrimitiveBase*>(ob)) {
			try {
				pbp->set(value);
			} catch(const XMLLoadSave::bad_format& bf) {
				cout << "'" << value << "' is a bad value for '" << key << "'" << endl;
				cout << bf.what() << endl;
			} catch(const std::exception& e) {
				cout << "An exception occured: " << e.what() << endl;
			}
		} else {
			cout << "Cannot assign to a dictionary" << endl;
			return;
		}
	}
}
void Simulator::cmdRun(const std::vector<std::string>& args, SharedGlobals::runlevel_t /*curLevel*/, bool isRelative) {
	if(args.size()<=1)
		return;
	if(isRelative)
		globals->setAutoPauseTime(get_time()+atoi(args[1].c_str()));
	else
		globals->setAutoPauseTime(atoi(args[1].c_str()));
}
void Simulator::cmdRun(const std::vector<std::string>& /*args*/, SharedGlobals::runlevel_t /*curLevel*/) {
	globals->timeScale=1;
}
void Simulator::cmdPause(const std::vector<std::string>& /*args*/, SharedGlobals::runlevel_t /*curLevel*/) {
	globals->timeScale=0;
}
void Simulator::cmdHelp(const std::vector<std::string>& args) {
	map<string,string> syntax;
	syntax["load"]="[file]";
	syntax["save"]="[file]";
	syntax["runlevel"]="[";
	for(unsigned int i=0; i<SharedGlobals::NUM_RUNLEVELS; i++) {
		stringstream ss;
		ss << i << "|" << SharedGlobals::runlevel_names[i];
		if(i!=SharedGlobals::NUM_RUNLEVELS-1)
			ss << " | ";
		syntax["runlevel"]+=ss.str();
	}
	syntax["runlevel"]+="]";
	syntax["get_time"]+="";
	syntax["set"]+="[var=value]";
	syntax["runto"]+="time";
	syntax["runfor"]+="time";
	syntax["run"]+="";
	syntax["pause"]+="";
	
	map<string,string> help;
	help["load"]="Load simulator configuration from file; if file unspecified, defaults to 'simulator.plist'.";
	help["save"]="Save simulator configuration to file; if file unspecified, defaults to 'simulator.plist'.";
	help["runlevel"]="You can specify a runlevel to move to, or if unspecified, the next one.\n"
		"You can only move forward runlevels, not backward.  Usually you'll only need RUNNING,\n"
		"unless you are debugging startup/shutdown code or the simulator itself.";
	help["get_time"]="Displays the simulator time.";
	help["set"]="Sets simulator configuration variables.  Without any arguments, displays\n"
		"all available variables and their current values.  Type 'help set <variable>' to get\n"
		"more information about a particular variable.";
	help["runto"]="Will advance the simulator time to a the specified value and then set Speed to 0.";
	help["runfor"]="Will advance the simulator time forward the specified number of milliseconds and then set Speed to 0.";
	help["run"]="Equivalent to 'set Speed=1'";
	help["pause"]="Equivalent to 'set Speed=0'";
	
	if(args.size()==1) {
		cout << "Available commands: " << endl;
		for(map<string,string>::const_iterator it=help.begin(); it!=help.end(); ++it) {
			cout << '\t' << it->first << " " << syntax[it->first] << endl;
		}
		cout << "type 'help <command>' for more information" << endl;
	} else {
		if(help.find(args[1])==help.end()) {
			cout << "The command '"<< args[1] << "' was not found" << endl;
			return;
		}
		if(args.size()==2) {
			cout << args[1] << " " << syntax[args[1]] << endl;
			cout << help[args[1]] << endl;
		} else {
			if(args[1]=="set") {
				plist::ObjectBase* ob=sim::config.findEntry(args[2]);
				if(ob==NULL) {
					cout << "'" << args[2] << "' is unknown" << endl;
					return;
				}
				size_t n=args[2].rfind('.');
				if(n==string::npos)
					cout << sim::config.getComment(args[2]) << endl;
				else {
					ob=sim::config.findEntry(args[2].substr(0,n));
					if(const plist::Dictionary * dict=dynamic_cast<const plist::Dictionary*>(ob))
						cout << dict->getComment(args[2].substr(n+1)) << endl;
					else
						cout << "'" << args[2].substr(0,n) << "' is not a dictionary" << endl;
				}
			} else {
				cout << args[1] << " " << syntax[args[1]] << endl;
				cout << help[args[1]] << endl;
			}
		}
	}
}


/*! @file
 * @brief 
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4 $
 * $Revision: 1.9 $
 * $State: Exp $
 * $Date: 2005/08/03 06:37:41 $
 */

