//-*-c++-*-
#ifndef INCLUDED_CLASSNAME_h_
#define INCLUDED_CLASSNAME_h_

#include "IPC/MutexLock.h"
#include "IPC/SemaphoreManager.h"
#include "IPC/ProcessID.h"
#include "Shared/plist.h"
#include "Shared/TimeET.h"

//! A class to hold various simulator parameters which need to be accessed from multiple processes
class SharedGlobals {
public:
	//! constructor
	SharedGlobals()
		: simulatorTime(0), timeScale(1), advanceOnAccess(true), lock(),
		bootTime(), timeOffset(0), lastTimeScale(1), autoPauseTime(-1U),
		semgr(), running(semgr.getSemaphore())
	{
		for(unsigned int i=0; i<NUM_RUNLEVELS; i++)
			level_count[i]=0;
		semgr.raise(running,1);
	}
	//! destructor
	~SharedGlobals() {
		semgr.releaseSemaphore(running);
	}

	//       ****************
	//!@name Shutdown Control
	//       ****************

	//! call this to cause "system shutdown" -- clean halt of the simulator (not actually the host system)
	void signalShutdown() {
		semgr.setValue(running,0);
	}
	//! test to see if the shutdown flag has been set (non-blocking)
	bool isShutdown() {
		return semgr.testZero(running,false);
	}
	//! blocks until shutdown flag has been set
	bool waitShutdown() {
		return semgr.testZero(running,true);
	}
	
	//! access to #semgr, returns SemaphoreManager::hadFault()
	bool hadFault() const { return semgr.hadFault(); }
	
	//! access to #semgr's SemaphoreManager::faultShutdown() -- call this *after* a fault has occured from the signal handler; doesn't signal the fault itself
	void faultShutdown() { semgr.faultShutdown(); }
	
	//@}


	//       ************
	//!@name Time Control
	//       ************

	//! returns the current simulator time, in milliseconds since startup
	/*! the simulator should set project_get_time::get_time_callback to call this,
	 *  so calls to ::get_time() will be forwarded here.  That wall all processes
	 *  will share the same time */
	unsigned int get_time();

	//! the current time within the simulation, only applicable when #isRealTime is false
	unsigned int simulatorTime;
	
	//! The speed at which time from get_time() will move when #isRealTime is true
	/*! You can use this to pretend your hardware is faster or slower
	 *  than it actually is.  For instance, a value of .5 means time
	 *  will move at half speed (pretending your hardware is twice as
	 *  fast)  This can be useful for "slow motion" analysis, or you
	 *  can speed up time to simulate a more processor-restrictive platform.
	 *
	 *  Negative values indicate full-speed processing -- time will be
	 *  incremented only as quickly as it can be without dropping any
	 *  video or sensor frames. (may be faster or slower than realtime)
	 *
	 *  A value of zero halts time. */
	plist::Primitive<double> timeScale;

	//! If true, and #timeScale is negative, simulator will automatically advance simulatorTime after each data update is processed, allowing full-speed processing
	/*! This functionality is actually achieved by having the receiver mark the
	 *  message read when it is ready for another message.  If #advanceOnAccess is
	 *  false, the receiver will not automatically mark the message read when it
	 *  is accessed, and you will have to make a manual call within your code to
	 *  signal completion.  For example, typically you will be doing this from
	 *  Main process, which would look like this:
	 *
	 *  @code
	 *  #include "local/sim/Main.h"
	 *  
	 *  Main::advanceVision(); //only legal if your code is
	 *  Main::advanceSensor(); // actually running in Main!
	 *  @endcode
	 *
	 *  This flag applies to both sensors and vision, but ignored when in realtime
	 *  mode (0 or positive #timeScale, or loading from a realtime source, such as
	 *  live telemetry)
	 */
	plist::Primitive<bool> advanceOnAccess;

	//! sets #autoPauseTime
	void setAutoPauseTime(unsigned int t) { autoPauseTime=t; }
	
	//@}


	//       **********************
	//!@name Runlevel Communication
	//       **********************

	//! defines the runlevels that each process passes through; runlevels should monotonically increase (can't go backwards)
	enum runlevel_t {
		CREATED=0,    //!< corresponding element of #level_count is incremented prior to each fork -- not strictly a runlevel per se
		CONSTRUCTING, //!< currently initializing
		STARTING,     //!< setting up shared memory regions with other processes
		RUNNING,      //!< full activity, stay here until the #running semaphore is set to 0
		STOPPING,     //!< dereferencing shared regions, waiting for threads to finish
		DESTRUCTING,  //!< destructors are in progress
		DESTRUCTED,   //!< destruction has completed, corresponding element of #level_count is incremented immediately prior to process completion
	};
	static const unsigned int NUM_RUNLEVELS=DESTRUCTED+1; //!< symbolic access to the total number of runlevel stages

	//! string versions of runlevel_t for runtime user-feedback
	static const char * const runlevel_names[NUM_RUNLEVELS];

	//! a count of the number of processes which have passed through each runlevel
	unsigned int level_count[NUM_RUNLEVELS];

	//@}

	//! allows mutually exclusive access to the fields of SharedObject
	MutexLock<ProcessID::NumProcesses> lock;

	//! holds the host system's process ID for each simulator process
	pid_t pids[ProcessID::NumProcesses];

	//! maximum storage size of strings in #processNames
	static const unsigned int MAX_PROCESS_NAME_LEN=32;

	//! each process should set a string version of its name for user feedback
	char processNames[ProcessID::NumProcesses][MAX_PROCESS_NAME_LEN];
	
protected:
	//! this returns time since boot (#bootTime), scaled by @a scale, relative to #timeOffset
	unsigned int get_real_time(double scale) const {
		return static_cast<unsigned int>(bootTime.Age().Value()*scale*1000-timeOffset);
	}

	//! real time since simulator startup (or, at least, since SharedGlobals was constructed... close enough)
	TimeET bootTime; 

	//! the scaled value of #bootTime at which isRealTime was last activated, allows you to start and stop realtime fluidly
	double timeOffset; 
	
	//! updated by each call to get_time(), if timeScale differs, allows timeOffset to be updated fluidly
	double lastTimeScale;
	
	//! if simulatorTime is about to move past this value, timeScale is set to 0 instead, and simulatorTime is set to this
	unsigned int autoPauseTime;
	
	SemaphoreManager semgr; //!< a semaphore set, only used for #running
	SemaphoreManager::semid_t running; //!< the semaphore within #semgr to communicate shutdown status between processes -- when the semaphore is set to 0, shutdown is requested
};

const unsigned int MAX_SUBJECTS=50; //!< maximum number of message queues the simulator can maintain
const unsigned int MAX_SUBJECT_NAME=50; //!< maximum storage capacity of subject names

// just a forward definition of RegionRegistry
template<unsigned int MAX_SUBJECTS, unsigned int MAX_SUBJECT_NAME> class RegionRegistry;
//! the type to use for the inter-process communication registry
typedef RegionRegistry<MAX_SUBJECTS,MAX_SUBJECT_NAME> ipc_setup_t;

extern ipc_setup_t * ipc_setup; //!< a global pointer to the inter-process message queue registry (a RegionRegistry)
extern SharedGlobals * globals; //!< a global pointer to the SharedGlobals instance

/*! @file
 * @brief A class to hold various simulator parameters which need to be accessed from multiple processes
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4 $
 * $Revision: 1.8 $
 * $State: Exp $
 * $Date: 2005/07/28 18:22:26 $
 */

#endif
