//-*-c++-*-
#ifndef INCLUDED_LoadFileThread_h_
#define INCLUDED_LoadFileThread_h_

#include "IPC/Thread.h"
#include "Shared/plist.h"
#include "IPC/MessageQueue.h"
#include <list>

//! Provides resources for loading time-based data from disk
/*! Runs in a separate thread, preloads data into shared memory buffers, and
 *  then sends the buffer when the time is right.  Thread doesn't start until
 *  you set the source or call loadFileList(). */
class LoadFileThread : public Thread, public plist::Dictionary, public plist::PrimitiveListener {
public:

	//! constructor
	/*! @param source the path to either a directory to load files from, or a single specific data file
	 *  @param filter a regular expression (POSIX.2 extended format) to select which files to load from @a source, if @a is a directory
	 *  @param fps frames per second, see #frameRate
	 *  @param messages the MessageQueue through which to send the data
	 *
	 *  The file list is not actually loaded until you call loadFileList -- this
	 *  gives you a chance to reset the default values you supplied in the
	 *  constructor with actual values from, for example, a preference file, and
	 *  then load the file list after the settings are correct.
	 */
	LoadFileThread(std::string source, std::string filter, float fps, MessageQueueBase& messages)
	: Thread(), plist::Dictionary(),
		src(source),
		filenameFilter(filter),
		framerate(fps),
		startTimeOffset(0),
		verbose(false),
		loop(true),
		sent(), loaded(), timestamps(), files(), curfile(files.begin()), msgr(messages),
		timeOffset(0), lock(), loopRemainder(0), isIndexed(false), statusListeners()
	{
		addEntry("Source",src,"The directory from which data samples should be loaded, or a single specific file.\nIn the future, this could also be network addresses for teleoperation and remote processing.");
		addEntry("FileFilter",filenameFilter,"If Source is a directory, only files matching the filter will be loaded from it.");
		addEntry("Framerate",framerate,"The rate at which images should be loaded");
		addEntry("StartTime",startTimeOffset,"The time at which the file list should start being processed");
		addEntry("Verbose",verbose,"If true, input status information will be dumped on the console");
		addEntry("Loop",loop,"If true, restart file list at the beginning when the end is reached; otherwise just stop loading data");
		src.addPrimitiveListener(this);
		verbose.addPrimitiveListener(this);
	}
	//! destructor
	~LoadFileThread();
	
	//! source the path to either a directory to load files from, or a single specific data file
	plist::Primitive<std::string> src;

	//! a regular expression (POSIX.2 extended format) to select which files to load from @a source, if @a is a directory
	plist::Primitive<std::string> filenameFilter;

	//! frames per second to send -- only referenced when setting up #files during loadFileList -- later modifications won't resent the already loaded file list
	plist::Primitive<float> framerate;

	//! the requested time of start of run through #files -- if this is modified, it will be noticed during runloop() and #curfile will be updated appropriately
	plist::Primitive<int> startTimeOffset;

	//! controls whether to give feedback messages on the console regarding progress
	plist::Primitive<bool> verbose; 

	//! controls whether to restart #curfile at the beginning of #files when it reaches the end
	plist::Primitive<bool> loop; 


	//! call this to (re)load the list of available file names from disk
	/*! If @a clearCurrent is set, then the current file list will be cleared;
	 *  otherwise, the loaded files will be appended to the current queue */
	virtual void loadFileList(bool clearCurrent=true);
	//! sets the next frame to be sent (e.g. pass 0 to reset to the first frame)
	virtual void setFrame(unsigned int f);
	
	//! returns the timestamp of the next message in the queue ready to be sent
	virtual unsigned int nextTimestamp() { return timestamps.size()>0 ? timestamps.front() : -1U; }
	
	//! Request updates to StatusListener callbacks
	virtual void addStatusListener(MessageQueueBase::StatusListener* l) {
		if(l==NULL)
			return;
		statusListeners.push_back(l);
	}
	//! Unsubscribes a StatusListener from future updates
	virtual void removeStatusListener(MessageQueueBase::StatusListener* l) {
		std::list<MessageQueueBase::StatusListener*>::iterator it=find(statusListeners.begin(),statusListeners.end(),l);
		if(it!=statusListeners.end())
			statusListeners.erase(it);
	}
	
	virtual void plistValueChanged(const plist::PrimitiveBase& pl) {
		if(&pl==&src)
			loadFileList();
		else if(&pl==&verbose)
			msgr.setReportDroppings(verbose);
	}
	
protected:
	//! removes our reference to a region created by loadFile()
	virtual void freeRegion(RCRegion* rcr) { rcr->RemoveReference(); }
	//! loads the next data element from file and provides its timestamp, skipping any files which have already expired
	/*! @param[in] data a pre-allocated region to use, or NULL to request creation of a new one
	 *  @param[out] data the address of the region with loaded data
	 *  @param[out] t the timestamp at which @a data should be consumed */
	virtual void getNextData(RCRegion*& data, unsigned int& t);
	//! does the actual work of loading and processing data from disk, should be provided by subclass
	/*! @param[in] file full path of file to load
	 *  @param[in] data a pre-allocated region to use, or NULL to request creation of a new one
	 *  @param[out] data the address of the region which has been filled in */
	virtual bool loadFile(const std::string& file, RCRegion*& data)=0;
	//! increments #curfile, looping if necessary; returns false if end of list reached and not #loop
	/*! @param[in] loopTime is the time of a single loop (the time of the last frame, plus inter-frame time)
	 *  @param[in] curTime is the current time, relative to beginning of current loop (#timeOffset)
	 *  @param[out] modified curTime if loop has occurred, subtracting loopTime */
	virtual bool incrementCurfile(float loopTime, int& curTime);
	
	//! load a list of files from a directory specified by #src
	virtual void loadFileListFromDirectory();
	//! load a single file
	virtual void loadSingleFile(const std::string& file);
	// ! NOT DONE -- load a list of files from an index file specified by #src
	//virtual void loadFileListFromIndex();

	//! monitor #msgr, send new messages when their timestamp indicates they are due, then load upcoming messages
	virtual unsigned int runloop();

	//! Notifies statusListeners that a message has been read by all MessageQueue receivers
	virtual void fireMessagesRead(unsigned int howmany);

	//! assign a new value @a t to #timeOffset -- clears loaded data queue (to be reloaded on next call to getNextData())
	virtual void resetTimeOffset(int t);
	
	//! computes the time of an entire loop through the files (including inter-frame time at the end of the loop)
	virtual float calcLoopTime();

	static const unsigned int NUM_PRELOAD=2; //!< number of data elements to preload
	typedef std::list<RCRegion* > msgbuf_t; //!< type of collection of shared data regions
	typedef std::map<unsigned int,std::string> files_t; //!< type of #files, the list of files to load

	msgbuf_t sent; //!< for efficiency, reuse old buffers -- oldest at front, most recently used at back
	msgbuf_t loaded; //!< for efficiency, reuse old buffers -- oldest at front, most recently used at back
	std::list<unsigned int> timestamps; //!< the timestamps to send data queued in #loaded
	files_t files; //!< the list of files to load ('second'), and their timestamps ('first')
	files_t::const_iterator curfile; //!< an iterator referencing #files -- indicates next file to load
	MessageQueueBase& msgr; //!< the MessageQueue through which to send the data
	int timeOffset; //!< the starting time of the run through #files, see #startTimeOffset
	Thread::Lock lock; //!< allows mutual exclusion over this object's member variables
	float loopRemainder; //!< a bit ugly -- this is the fractional leftover accumulated from looping so we won't "drift" over time
	bool isIndexed; //!< if we're using an index file, we have to handle the loopRemainder differently

	std::list<MessageQueueBase::StatusListener*> statusListeners; //!< MessageQueueBase::StatusListeners currently subscribed from addStatusListener()
};

/*! @file
 * @brief 
 * @author Ethan Tira-Thompson (ejt) (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4 $
 * $Revision: 1.10 $
 * $State: Exp $
 * $Date: 2005/08/04 21:32:16 $
 */

#endif
