#include "LoadFileThread.h"
#include "Shared/get_time.h"
#include <sys/stat.h>
#include <regex.h>
#include <dirent.h>

LoadFileThread::~LoadFileThread() {
	Thread::Lock l(lock);
	src.removePrimitiveListener(this);
	verbose.removePrimitiveListener(this);
	while(sent.size()>0) {
		freeRegion(sent.front());
		sent.pop_front();
	}
	while(loaded.size()>0) {
		freeRegion(loaded.front());
		loaded.pop_front();
	}
}

void LoadFileThread::loadFileList(bool clearCurrent/*=true*/) {
	Thread::Lock l(lock);
	struct stat sb;
	if(stat(src.c_str(),&sb)) {
		std::cerr << "Could not open source " << src << std::endl;
		return;
	}
	if(clearCurrent)
		files.clear();
	isIndexed=false;
	if(sb.st_mode&S_IFDIR) {
		loadFileListFromDirectory();
	} else {
		//Test to see if the file matches the filter
		regex_t re;
		if(int err=regcomp(&re,filenameFilter.c_str(),REG_EXTENDED | REG_NOSUB)) {
			char msg[128];
			regerror(err,&re,msg,128);
			std::cerr << "Bad filter '" << filenameFilter << "': " << msg << std::endl;
			regfree(&re);
			return;
		}
		int match=regexec(&re,src.c_str(),0,NULL,0);
		regfree(&re);
		if(match==0) {
			loadSingleFile(src.c_str());
		} else if(match==REG_NOMATCH) {
			//if it doesn't match the image RE, assume it's an index file
			//loadFileListFromIndex();
		} else {
			char msg[128];
			regerror(match,&re,msg,128);
			std::cerr << "Regex error on '" << src << "': " << msg << std::endl;
		}
	}
	setFrame(0);
}

void LoadFileThread::setFrame(unsigned int f) {
	Thread::Lock l(lock);
	loopRemainder=0;
	startTimeOffset=(int)(get_time()-f*1000/framerate);
	resetTimeOffset(startTimeOffset);
	curfile=files.begin();
	advance(curfile,f);
}

void LoadFileThread::loadSingleFile(const std::string& file) {
	Thread::Lock l(lock);
	files[(unsigned int)calcLoopTime()]=file;
}

void LoadFileThread::loadFileListFromDirectory() {
	Thread::Lock l(lock);
	regex_t re;
	if(int err=regcomp(&re,filenameFilter.c_str(),REG_EXTENDED | REG_NOSUB)) {
		char msg[128];
		regerror(err,&re,msg,128);
		std::cerr << "Bad filter '" << filenameFilter << "': " << msg << std::endl;
		regfree(&re);
		return;
	}
	DIR * d=opendir(src.c_str());
	if(d==NULL) {
		std::cerr << "Could not open directory " << src << std::endl;
		regfree(&re);
		return;
	}
	float tinc=1000.f/framerate;
	float time=calcLoopTime();
	struct dirent* res;
	
#ifdef HAVE_READDIR_R
	struct dirent cur;
	if(readdir_r(d,&cur,&res)) {
		std::cerr << "Error reading files from " << src << std::endl;
		closedir(d);
		regfree(&re);
		return;
	}
#else
	res=readdir(d);
#endif
	while(res!=NULL) {
		int match=regexec(&re,res->d_name,0,NULL,0);
		if(match==0) {
			files[static_cast<unsigned int>(time)]=(src+"/")+res->d_name;
			//std::cout << "Enqueuing " << res->d_name << std::endl;
			time+=tinc;
		} else if(match!=REG_NOMATCH) {
			char msg[128];
			regerror(match,&re,msg,128);
			std::cerr << "Regex error on '" << res->d_name << "': " << msg << std::endl;
		} // else std::cout << "Skipping " << res->d_name << std::endl;
#ifdef HAVE_READDIR_R
		if(readdir_r(d,&cur,&res)) {
			std::cerr << "Error reading files from " << src << std::endl;
			closedir(d);
			regfree(&re);
			return;
		}
#else
		res=readdir(d);
#endif
	}
	closedir(d);
	regfree(&re);
}

/*
void LoadFileThread::loadFileListFromIndex() {
	isIndexed=true;
	regex_t re;
	if(int err=regcomp(&re,filenameFilter.c_str(),REG_EXTENDED | REG_NOSUB)) {
		char msg[128];
		regerror(err,&re,msg,128);
		std::cerr << "Bad filter '" << filenameFilter << "': " << msg << std::endl;
		regfree(&re);
		return;
	}
	// todo read index file
	regfree(&re);
}
*/

bool LoadFileThread::incrementCurfile(float loopTime, int& curTime) {
	if(++curfile == files.end()) {
		if(!loop) {
			if(verbose)
				std::cout << "Out of data from '" << src << "' -- set 'loop' to true, or use 'restart' command to manually loop" << std::endl;
			return false;
		} else {
			if(verbose)
				std::cout << "Looping data from '" << src << "'" << std::endl;
			unsigned int loopTimeI = (unsigned int)(loopTime);
			loopRemainder+=loopTime-loopTimeI;
			unsigned int loopRemainderI=(unsigned int)(loopRemainder);
			loopRemainder-=loopRemainderI;
			timeOffset=(startTimeOffset+=loopTimeI+loopRemainderI);
			curTime-=loopTimeI;
			curfile=files.begin();
		}
	}
	return true;
}

void LoadFileThread::getNextData(RCRegion*& data, unsigned int& t) {
	if(files.size()==0 || curfile==files.end())
		return;
	if(startTimeOffset!=timeOffset)
		resetTimeOffset(startTimeOffset);

	//skip old data -- current time is already past these frames, no point in loading them
	unsigned int realt=get_time();
	int curt=realt-timeOffset;
	float loopTime=calcLoopTime();
	if(curt>(int)(loopTime)) {
		if(!loop) {
			if(verbose)
				std::cout << "Way out of data from '" << src << "' -- set 'loop' to true, or use 'restart' command to manually loop" << std::endl;
			curfile=files.end();
			return;
		}
		if(verbose)
			std::cout << "Mass looping data from '" << src << "'" << std::endl;
		unsigned int loops=(unsigned int)(curt/loopTime);
		float massLoopTime=loops*loopTime;
		unsigned int massLoopTimeI=(unsigned int)(massLoopTime);
		loopRemainder+=massLoopTime-massLoopTimeI;
		unsigned int loopRemainderI=(unsigned int)(loopRemainder);
		loopRemainder-=loopRemainderI;
		startTimeOffset+=massLoopTimeI+loopRemainderI;
		resetTimeOffset(startTimeOffset);
		curt=realt-timeOffset;
		curfile=files.begin();
	}
	while(static_cast<int>(curfile->first)<curt)
		if(!incrementCurfile(loopTime,curt))
			return;

	if(verbose)
		std::cout << "Loading frame from " << curfile->second << " dt="<<curfile->first << " scheduled:" << (curfile->first+timeOffset) << std::endl;
	msgbuf_t::iterator it;
	for(it=sent.begin();it!=sent.end(); ++it)
		if((*it)->NumberOfReference()==1)
			break;
	if(it!=sent.end()) {
		if(!loadFile(curfile->second, *it)) {
			std::cerr << "Bad load on " << curfile->second << std::endl;
			++curfile;
			return;
		}
		data=*it;
		sent.erase(it);
	} else {
		if(!loadFile(curfile->second, data)) {
			std::cerr << "Bad load on " << curfile->second << std::endl;
			++curfile;
			return;
		}
	}
	t=curfile->first+timeOffset;
	incrementCurfile(loopTime,curt);
}

unsigned int LoadFileThread::runloop() {
	Thread::Lock l(lock);
	if(unsigned int read=msgr.pollStatus())
		fireMessagesRead(read);
	unsigned int curt=get_time();
	if(timestamps.size()>0 && curt>=timestamps.front()) {
		if(verbose)
			std::cout << "Sending frame " << timestamps.front() << " at " << curt << std::endl;
		msgr.sendMessage(loaded.front());
		sent.push_back(loaded.front());
		loaded.pop_front();
		timestamps.pop_front();
	}
	if(loaded.size()<NUM_PRELOAD) {
		RCRegion* rcr=NULL;
		unsigned int t=-1U;
		getNextData(rcr,t);
		if(t==-1U) {
			//out of data, skip a frame and then check again in case we get reset
			return (1000000/(unsigned int)framerate);
		}
		if(rcr==NULL) {
			std::cerr << "Could not allocate new shared memory region or bad initial load" << std::endl;
			return (1000000/(unsigned int)framerate);
		}
		if(timestamps.size()==0 || t>timestamps.back()) {
			loaded.push_back(rcr);
			timestamps.push_back(t);
		} else {
			std::cerr << "ERROR: LoadFileThread received old data from getNextData" << std::endl;
			sent.push_front(rcr);
		}
	}
	return 1000;
}

void LoadFileThread::fireMessagesRead(unsigned int howmany) {
	std::list<MessageQueueBase::StatusListener*>::iterator it=statusListeners.begin();
	while(it!=statusListeners.end()) {
		std::list<MessageQueueBase::StatusListener*>::iterator cur=it++; //increment early in case the listener changes subscription
		(*cur)->messagesRead(msgr,howmany);
	}
}

void LoadFileThread::resetTimeOffset(int t) {
	while(loaded.size()>0) {
		sent.push_front(loaded.front());
		loaded.pop_front();
	}
	timestamps.clear();
	if(timeOffset<t) {
		//need to rewind curfile
		curfile=files.begin(); //getNextData will fast forward to current time
	}
	timeOffset=t;
}

float LoadFileThread::calcLoopTime() {
	if(files.size()==0)
		return 0;
	float tinc=1000.f/framerate;
	if(isIndexed) {
		files_t::iterator lastfile=files.end();
		--lastfile;
		unsigned int last=lastfile->first;
		return last+tinc;
	} else {
		return files.size()*tinc;
	}
}


/*! @file
 * @brief 
 * @author Ethan Tira-Thompson (ejt) (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4 $
 * $Revision: 1.12 $
 * $State: Exp $
 * $Date: 2005/08/04 21:32:16 $
 */
