//-*-c++-*-
#ifndef INCLUDED_OldHeadPointerMC_h
#define INCLUDED_OldHeadPointerMC_h

#include "MotionCommand.h"
#include "OutputCmd.h"
#include "Shared/RobotInfo.h"

//! This class gives some quick and easy functions to point the head at things
class OldHeadPointerMC : public MotionCommand {
public:
  //! constructor, defaults to active, BodyRelative, all joints at 0
  OldHeadPointerMC();

  //! destructor
  virtual ~OldHeadPointerMC() {}

  //! Various modes the head can be in.  In the future may want to add ability to explicitly track an object or point in the world model
  enum CoordFrame_t {
    BodyRelative,    //!<holds neck at a specified position, like a PostureEngine, but neck specific
    GravityRelative  //!<uses accelerometers to keep a level head, doesn't apply for pan joint, but in future could use localization for pan
  };

  void   setWeight(double w);                               //!< sets the weight values for all the neck joints

  //! set a specific head joint weight, pass one of RobotInfo::TPROffset_t, not a full offset!
  void setWeight(RobotInfo::TPROffset_t i, double weight) {
    dirty=true; targetReached=false; headCmds[i].weight=weight; }

  void   setActive(bool a) { active=a; } //!< sets #active flag; see isDirty()
  bool   getActive() const { return active; } //!< returns #active flag, see isDirty()

  //! sets #maxSpeed to 0 (no maximum)
  void noMaxSpeed() { for(unsigned int i=0; i<NumHeadJoints; i++) maxSpeed[i]=0; }

  void defaultMaxSpeed(); //!< restores #maxSpeed to default settings from Config::Motion_Config

  void setMaxSpeed(TPROffset_t i, float x) { maxSpeed[i]=x*FrameTime/1000; } //!< sets #maxSpeed in rad/sec
  float getMaxSpeed(TPROffset_t i) { return maxSpeed[i]*1000/FrameTime; } //!< returns #maxSpeed in rad/sec

  //! converts a value @a v in @a srcmode to a value in @a tgtmode that would leave the joint angle for joint @a i constant (you probably won't need to call this directly)
  double convert(RobotInfo::TPROffset_t i, double v, CoordFrame_t srcmode, CoordFrame_t tgtmode) const {
    return (srcmode==tgtmode)?v:convFromBodyRelative(i,convToBodyRelative(i, v, srcmode),tgtmode);
  }

  //!Note that none of these are @c virtual, so you don't have to checkout to use them, you @e should be able to use MotionManager::peekMotion()
  //!@name Joint Accessors

  //! Directly sets the neck values (radians), uses current mode
  void setJoints(double tilt, double pan, double roll);

  //! sets all the joints to the given mode, will convert the values to the new mode if @a convert is true
  void setMode(CoordFrame_t m, bool convert=true); 

    //! sets a specific head joint's mode; will convert from previous mode's value to next mode's value if convert is true.  Pass one of RobotInfo::TPROffset_t, not a full offset!
  void setJointMode(RobotInfo::TPROffset_t i, CoordFrame_t m, bool convert=true);

  //! set a specific head joint's value (in radians, for whatever mode it's in), pass one of RobotInfo::TPROffset_t, not a full offset!
  void setJointValue(RobotInfo::TPROffset_t i, double value)
	{ dirty=true; targetReached=false; headTargets[i]=(headModes[i]==BodyRelative)?clipAngularRange(i+HeadOffset,value):value; }

  //! set a specific head joint (in radians), pass one of RobotInfo::TPROffset_t, not a full offset!
  void setJointValueAndMode(RobotInfo::TPROffset_t i, double value, CoordFrame_t m)
  { dirty=true; targetReached=false; headTargets[i]=(m==BodyRelative)?clipAngularRange(i+HeadOffset,value):value; headModes[i]=m; }

  //! set a specific head joint (in radians) auto converting @a value from mode @a m to the current mode, pass one of RobotInfo::TPROffset_t, not a full offset!
  void setJointValueFromMode(RobotInfo::TPROffset_t i, double value, CoordFrame_t m)
  { dirty=true; targetReached=false; headTargets[i]=convert(i,value,m,headModes[i]); }

  //! returns the current mode for joint @a i (use RobotInfo::TPROffset_t, not global offset)
  CoordFrame_t getJointMode(RobotInfo::TPROffset_t i) const { return headModes[i]; }

  //! returns the target value (relative to the current mode) of joint @a i.  Use getOutputCmd() if you want to know the current @b commanded joint value; To get the current joint @b position, look in WorldState
  double getJointValue(RobotInfo::TPROffset_t i) const { return headTargets[i]; }
  //@}


  //!@name Inherited:
  virtual int updateOutputs(); //!< Updates where the head is looking
  virtual const OutputCmd& getOutputCmd(unsigned int i);  //!< returns one of the #headCmds entries or ::unusedJoint if not a head joint
  virtual int isDirty() { return ((dirty || !targetReached) && active)?1:0; } //!< true if a change has been made since the last updateJointCmds() and we're active
  virtual int isAlive() { return true; }
  virtual void DoStart() { MotionCommand::DoStart(); dirty=true; targetReached=false; }
  //@}

 protected:
  double convToBodyRelative(TPROffset_t i, double v, CoordFrame_t mode) const;   //!< converts to a body relative measurement for joint @a i
  double convFromBodyRelative(TPROffset_t i, double v, CoordFrame_t mode) const; //!< converts from a body relative measurement for joint @a i
	static float normalizeAngle(float x) { return x-rint(x/(2*M_PI))*(2*M_PI); } //!< puts x in the range (-pi,pi)
	//! if @a x is outside of the range of joint @a i, it is set to either the min or the max, whichever is closer
	static float clipAngularRange(unsigned int i, float x) {
		float min=outputRanges[i][MinRange];
		float max=outputRanges[i][MaxRange];
		if(x<min || x>max) {
			float mn_dist=fabs(normalizeAngle(min-x));
			float mx_dist=fabs(normalizeAngle(max-x));
			if(mn_dist<mx_dist)
				return min;
			else
				return max;
		} else
			return x;
	}

  bool dirty;                            //!< true if a change has been made since last call to updateJointCmds()
  bool active;                           //!< set by accessor functions, defaults to true
  bool targetReached;                  //!< false if the head is still moving towards its target
  OutputCmd headCmds[NumHeadJoints] ;  //!< stores the last values we sent from updateOutputs
  float headTargets[NumHeadJoints];       //!< stores the target value of each joint, relative to #headModes
  CoordFrame_t headModes[NumHeadJoints]; //!< stores the current mode of each joint, for instance so tilt can be GravityRelative while pan is static
  float maxSpeed[NumHeadJoints];         //!< initialized from Config::motion_config, but can be overridden by setMaxSpeed(); rad per frame
};

/*! @file
 * @brief Describes OldHeadPointerMC, a class for various ways to control where the head is looking
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4 $
 * $Revision: 1.2 $
 * $State: Exp $
 * $Date: 2005/08/07 04:11:03 $
 */

#endif

