#include "EventTranslator.h"
#include "Events/LocomotionEvent.h"
#include "Events/VisionObjectEvent.h"
#include "Events/TextMsgEvent.h"
#include "Events/EventRouter.h"
#include "Shared/debuget.h"
#include <iostream>

#ifdef PLATFORM_APERIOS
#  include <OPENR/OSubject.h>
#else
#  include "IPC/MessageQueue.h"
#endif
#include "IPC/RCRegion.h"

using namespace std;

EventTranslator::eventLookup_t EventTranslator::eventLookup;

EventTranslator::~EventTranslator() {
	for(eventLookup_t::iterator it=eventLookup.begin(); it!=eventLookup.end(); ++it)
		delete (*it).second;
	eventLookup.clear();
}

void
EventTranslator::encodeEvent(const EventBase& event) {
	event.setSaveFormat(EventBase::BINARY);
	const unsigned int headerlen=sizeof(event.getClassTypeID());
	const unsigned int bufsize=headerlen+event.getBinSize();
	char * buf=bufferRequest(bufsize);
	if(buf==NULL) {
		cerr << "ERROR: EventTranslator unable to transmit event because requested buffer was NULL" << endl;
		return;
	}
	unsigned int header=event.getClassTypeID();
	memcpy(buf,&header,headerlen);
	unsigned int used=event.SaveBuffer(buf+headerlen,bufsize-headerlen);
	if(used==0) {
		cerr << "ERROR: EventTranslator unable to transmit event because EventBase::SaveBuffer failed (buffer==" << (void*)(buf+headerlen) << ", size==" << bufsize-headerlen << ")" << endl;
		post(buf,0);
		return;
	}
	post(buf,used);
	return;
}

EventBase*
EventTranslator::decodeEvent(const char * entry, unsigned int size) {
	const unsigned int headerlen=sizeof(unsigned int);
	unsigned int header=0;
	memcpy(&header,entry,headerlen);
	eventLookup_t::iterator it=eventLookup.find(header);
	if(it==eventLookup.end()) {
		cerr << "ERROR: EventTranslator unable to translate buffer because header does not match a previously registered class type id" << endl;
		return NULL;
	}
	EventBase* evt=static_cast<EventBase*>((*it).second->constructTemplate());
	evt->setSaveFormat(EventBase::BINARY);
	if(evt->LoadBuffer(entry+headerlen,size-headerlen)==0) {
		cerr << "ERROR: EventTranlator unable to translate buffer because data is malformed (EventBase::LoadBuffer failed)" << endl;
		return NULL;
	}
	return evt;
}

void
NoOpEventTranslator::encodeEvent(const EventBase& event) {
	evtRouter.postEvent(event.clone());
}

char*
IPCEventTranslator::bufferRequest(unsigned int size) {
	ASSERT(curRegion==NULL,"WARNING: IPCEventTranslator::bufferRequest() curRegion was not NULL");
	curRegion = new RCRegion(size);
	return curRegion->Base();
}

//#include "Shared/TimeET.h"

void
IPCEventTranslator::post(const char* buf, unsigned int /*size*/) {
	ASSERTRET(curRegion!=NULL,"ERROR: IPCEventTranslator::post(buf,size) was NULL");
	if(buf!=curRegion->Base()) {
		cerr << "ERROR: IPCEventTranslator::post(buf,size) buf does not match value given from previous bufferRequest()" << endl;
		return;
	}
	//cout << TimeET() << endl;
#ifdef PLATFORM_APERIOS
	subject.SetData(curRegion);
	subject.NotifyObservers();
#else
	subject.sendMessage(curRegion);
	curRegion->RemoveReference();
#endif
	curRegion=NULL;
}
	

/*! @file
 * @brief Implements EventTranslator, which receives events from EventRouters in non-Main processes and adds them into a SharedQueue for Main to pick up
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4 $
 * $Revision: 1.15 $
 * $State: Exp $
 * $Date: 2005/07/26 03:11:23 $
 */

