//-*-c++-*-
#ifndef INCLUDED_EventRouter_h
#define INCLUDED_EventRouter_h

#include <vector>
#include <list>
#include <map>
#include <algorithm>
#include "EventListener.h"
#include "EventTrapper.h"
#include "Shared/get_time.h"
#include "Shared/debuget.h"
#include <iostream>

//! This class will handle distribution of events as well as management of timers
/*! Classes must inherit from EventListener and/or EventTrapper in order to
 *  receive events.
 *
 *  Use the global @c ::erouter EventRouter to post and subscribe to
 *  events, except if you are posting from within a MotionCommand, in
 *  which case you should use MotionCommand::postEvent() so that it
 *  will automatically be sent from the Motion process to the Main
 *  process.
 *
 *  When multiple listeners are subscribed, the order in which an event is
 *  distributed among them looks like this:
 *  -# "Specific" listeners: any listener which specifies a particular source id.
 *     (It doesn't matter if they specify a type id or not.)
 *    - older listeners get events before younger listeners
 *  -# "General" listeners: those that subscribe to an entire generator
 *    - older listeners get events before younger listeners
 *
 *  ...but if you're relying on that ordering, there should be a cleaner
 *  way to do whatever you're doing.
 *
 *  If one behaviors unsubscribes another one during a processEvent(),
 *  that behavior may still get the "current" event before the
 *  unsubscription takes place.  This is not a prescribed behavior,
 *  and also should not be relied on one way or the other.
 *
 *  Timer events are only sent to the generator which requested them.  So if 
 *  EventListener @e A requests a timer with ID 0 at two second intervals,
 *  and EventListener @e B requests a timer with ID 0 at three second intervals,
 *  each will still only receive the timers they requested - no cross talk.
 *  The timer generator is unique in this regard, which is why it is built in
 *  as an integral component of the EventRouter.  All other events are broadcast.
 *
 *  If an EventListener/EventTrapper subscribes to the same event source multiple
 *  times, it will receive multiple copies of the event.  However, the first call
 *  to removeListener for a source will remove all subscriptions to that source.\n
 *  Example: EventListener @e A subscribes to (buttonEGID,*,*), and twice to
 *  (buttonEGID,0,*).
 *    - If button 0 is pressed, @e A will get three copies of the event.
 *    - If button 1 is pressed, @e A will get one copy.
 *    - If removeListener(&A,buttonEGID) is called, the (buttonEGID,*,*) is
 *      removed, <em>as well as</em> both of (buttonEGID,0,*).
 *    - If removeListener(&A,buttonEGID,0) is called, both of (buttonEGID,0,*)
 *      are removed, but (buttonEGID,*,*) would be untouched.
 *
 *  The buffered event distribution has not been tested thoroughly,
 *  and should be considered deprecated.
 *
 *  @see EventBase::EventGeneratorID_t for a complete listing of all generators,
 *  as well as instructions on how to add new generators.
 */
class EventRouter : public EventListener {
 public:
	EventRouter(); //!< Constructs the router, #buffertime defaults to 1
	virtual ~EventRouter() { reset(); removeAllTimers(); } //!< just calls reset and removeAllTimers()
	void reset() { listeners.clear(); trappers.clear(); removeAllTimers(); } //!< erases all listeners, trappers and timers, resets EventRouter
	
	void setBufferTime(unsigned int t) { buffertime=t; } /*!< @brief sets the time to wait between buffer clears, see EventRouter::buffertime. @param t time to wait between buffer clears @see buffertime */
	unsigned int getBufferTime() { return buffertime; } /*!< @brief returns the time to wait between buffer clears, see EventRouter::buffertime. @return time to wait between buffer clears @see buffertime */

	//!@name Posting/Processing Events
	/*!@brief recommended to create and post an event using current buffer setting */
	void postEvent(EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid, unsigned int dur) { if(buffertime>0) events.push_back(new EventBase(egid,sid,etid,dur)); else processEvent(EventBase(egid,sid,etid,dur)); }
	void postEvent(EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid, unsigned int dur, const std::string& n, float m) { if(buffertime>0) events.push_back(new EventBase(egid,sid,etid,dur,n,m)); else processEvent(EventBase(egid,sid,etid,dur,n,m)); }
	void postEvent(EventBase* e) { if(buffertime>0) events.push_back(e); else { processEvent(*e); delete e; } }

	//! determines if timers need to be posted, and posts them if so.
	/*! Call this often to ensure accurate timers.  Also, will clear event buffer before sending
			timer events in order to ensure correct ordering of event reception */
	void processTimers();
	void processEventBuffer(); //!< clears the event buffer, deletes events as it does so.
	void processEvent(const EventBase& e); //!< forces unbuffered - sends event *now*.  Will clear the buffer first if needed to ensure proper event ordering
	//@}

	//!@name Listener/Trapper Recall
	//! returns true if the specified listener/trapper would receive any events that match the specified criteria
	bool isListeningAny(EventListener* el, EventBase::EventGeneratorID_t egid) { return listeners.verifyMappingAny(el,egid); }
	bool isListeningAny(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid) { return listeners.verifyMappingAny(el,egid,sid); }
	bool isListeningAll(EventListener* el, EventBase::EventGeneratorID_t egid) { return listeners.verifyMappingAll(el,egid); }
	bool isListeningAll(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid) { return listeners.verifyMappingAll(el,egid,sid); }
	bool isListening(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) { return listeners.verifyMapping(el,egid,sid,etid); }
	bool isListening(EventListener* el, const EventBase& e) { return listeners.verifyMapping(el,e.getGeneratorID(),e.getSourceID(),e.getTypeID()); }
	bool isTrappingAny(EventTrapper* el, EventBase::EventGeneratorID_t egid) { return trappers.verifyMappingAny(el,egid); }
	bool isTrappingAny(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid) { return trappers.verifyMappingAny(el,egid,sid); }
	bool isTrappingAll(EventTrapper* el, EventBase::EventGeneratorID_t egid) { return trappers.verifyMappingAll(el,egid); }
	bool isTrappingAll(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid) { return trappers.verifyMappingAll(el,egid,sid); }
	bool isTrapping(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) { return trappers.verifyMapping(el,egid,sid,etid); }
	bool isTrapping(EventTrapper* el, const EventBase& e) { return trappers.verifyMapping(el,e.getGeneratorID(),e.getSourceID(),e.getTypeID()); }
	//@}
	
	//!@name Listener/Trapper Detection
	/*!@brief <b>counts both listeners and trappers</b>, so generators can tell if it even needs to bother generating an event...*/
	/* Generators can also subscribe to the EventBase::erouterEGID event stream if
	 * they wish to be notified when they gain or lose listeners (particularly the
	 * first or last).\n
	 * ... if a tree falls in a forest, and there's no one around to see it, does
	 * it make a sound?\n
	 * ... if Vision sees a ball in an image, and there's no listeners, does it
	 * make an event? ;) */
	bool hasListeners(EventBase::EventGeneratorID_t egid) { return (egid==EventBase::timerEGID) ? timers.size()>0 : trappers.hasMapping(egid) || listeners.hasMapping(egid); }
	bool hasListeners(EventBase::EventGeneratorID_t egid, unsigned int sid) { return (egid==EventBase::timerEGID) ? timers.size()>0 : trappers.hasMapping(egid,sid) || listeners.hasMapping(egid,sid); }
	bool hasListeners(EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) { return (egid==EventBase::timerEGID) ? timers.size()>0 : trappers.hasMapping(egid,sid,etid) || listeners.hasMapping(egid,sid,etid); }
	bool hasListeners(const EventBase& e) { return (e.getGeneratorID()==EventBase::timerEGID) ? timers.size()>0 : hasListeners(e.getGeneratorID(),e.getSourceID(),e.getTypeID()); }
	//@}

	//!@name Timer Management
	void addTimer(EventListener* el, unsigned int sid, unsigned int delay, bool repeat=true); //!< adds a timer if it doesn't exist, or resets the timer if it already exists.
	void addTimer(EventListener* el, const EventBase& e, bool repeat=true) { addTimer(el,e.getSourceID(),e.getDuration(),repeat); } //!< calls the other addTimer() with the event's source id and duration, doesn't check to see if the generator is timerEGID
	void removeTimer(EventListener* el); //!< clears all pending timers for listener @a el
	void removeTimer(EventListener* el, unsigned int sid); //!< clears any pending timers with source id @a sid for listener @a el
	void removeAllTimers(); //!< clears all timers for all listeners
	//@}

	//!@name Listener Management
	//! Adds a listener for all events from a given event generator
	void addListener(EventListener* el, EventBase::EventGeneratorID_t egid);
	void addListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid); //!< Adds a listener for all types from a specific source and generator
	void addListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid); //!< Adds a listener for a specific source id and type from a given event generator
	//! Adds a listener for a specific source id and type from a given event generator, adding a Timer event will invoke addTimer(@a el, @a e.getSourceID(), @a e.getDuration(), @c true)
	void addListener(EventListener* el, const EventBase& e); 

	//! stops sending ALL events to the listener, including timers
	void removeListener(EventListener* el); 
	//! stops sending specified events from the generator to the listener.
	void removeListener(EventListener* el, EventBase::EventGeneratorID_t egid);
	void removeListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid); //!< stops sending specified events from the generator to the listener.
	void removeListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid); //!< stops sending specified events from the generator to the listener.
	//! stops sending specified events from the generator to the listener.  If a timer is passed it will invoke removeTimer(@a el, @a e.getSourceID())
	void removeListener(EventListener* el, const EventBase& e);

	//@}

	//!@name Trapper Management
	//! Adds a trapper for a specific source id and type from a given event generator
	/*! Note that since timers are not broadcast, they cannot be trapped.  Only the EventListener which requested the timer will receive that timer. */
	void addTrapper(EventTrapper* el, const EventBase& e);
	void addTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid); //!< Adds a trapper for all events from a given event generator
	void addTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid);  //!< Adds a trapper for all types from a specific source and generator
	void addTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid); //!< Adds a trapper for a specific source id and type from a given event generator

	void addTrapper(EventTrapper* el); //!< adds a trapper for ALL events

	//! stops sending specified events from the generator to the trapper.
	void removeTrapper(EventTrapper* el, const EventBase& e);
	void removeTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid); //!< stops sending specified events from the generator to the trapper.
	void removeTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid); //!< stops sending specified events from the generator to the trapper.
	void removeTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid); //!< stops sending specified events from the generator to the trapper.

	void removeTrapper(EventTrapper* el); //!< stops sending ALL events to the trapper
	//@}

 protected:
	void doSendBuffer(); //!< does the work of clearing the buffer (calls doSendEvent() )
	
	//! does the work of sending an event
	/*! Be aware this is an O(n^2) where n is the number of listeners for a particular event.
	 *  This is because after each call to processEvent, the list of listeners could have changed
	 *  so each listener much be verified before it is sent an event.  New listeners won't get the
	 *  current event, but neither should listeners which have just be removed */
	void doSendEvent(const EventBase& e); 

	//! Contains all the information needed to maintain a timer by the EventRouter
	struct TimerEntry {
		//! constructors an entry using the given value for next - useful for with TimerEntryPtrCmp
		explicit TimerEntry(unsigned int nxt) : el(NULL), sid(0), delay(0), next(nxt), repeat(false) {}
		//! constructs with the given values, sets next field automatically; see next
		TimerEntry(EventListener* e, unsigned int s, unsigned int d, bool r) : el(e), sid(s), delay(d), next(get_time()+delay), repeat(r) {}
		//! just does the default, i'm just being explicit since there's a pointer (no deep copy!)
		TimerEntry(const TimerEntry& t) : el(t.el), sid(t.sid), delay(t.delay), next(t.next), repeat(t.repeat) {}
		//! just does the default, i'm just being explicit since there's a pointer (no deep copy!)
		TimerEntry& operator=(const TimerEntry& t) { el=t.el; sid=t.sid; delay=t.delay; next=t.next; repeat=t.repeat; return *this; }
		//! will reset timer
		/*! @param d the time from now when the timer should go off (in milliseconds)
		 *  @param r true if the timer should automatically repeat */
		void Set(unsigned int d, bool r) { delay=d; repeat=r; next=get_time()+delay; }
		EventListener* el;  //!< the listener to fire at
		unsigned int sid;   //!< the source id to fire with
		unsigned int delay; //!< the delay until firing
		unsigned int next;  //!< the time at which this timer will go off next
		bool repeat;        //!< if true, will reset after firing, else will be deleted
	};
	/*! @brief Used by STL to sort the timer list in order of activation time
	 *  @see EventRouter::timers */
	class TimerEntryPtrCmp {
	public:
		//! Used by STL to sort the timer list in order of activation time; see timers
		/*! Since we remove NULLs before sorting, shouldn't need to check here (and I want to know if i'm wrong)
		 *  @return (a->next<b->next) */
		bool operator()(const TimerEntry* const a, const TimerEntry* const b) const { return (a->next<b->next); }
	};
	typedef std::vector<TimerEntry*>::iterator timer_it_t; //!< makes code more readable
	std::vector<TimerEntry*> timers;         //!< the list of timer entries being maintained, kept sorted by time they go active

	//! just for debugging
	void chkTimers() {
		unsigned int last=0;
		for(timer_it_t it=timers.begin(); it!=timers.end(); it++) {
			if(last>(*it)->next) {
				dispTimers();
				return;
			}
			last=(*it)->next;
		}
	}
	//! just for debugging
	void dispTimers() {
		std::cout << "out of order timers " << get_time() << " :\t";
		unsigned int last=0;
		for(timer_it_t it=timers.begin(); it!=timers.end(); it++) {
			if(last>(*it)->next)
				std::cout << "##";
			std::cout << (last=(*it)->next) << '\t';
		}
		std::cout << std::endl;
	}

	std::vector<EventBase*> events;     //!< used to store buffered events
	bool doSendBufferLock;              //!< in case of recursive calls to processEventBuffer()/doSendBuffer()
	unsigned int lastBufClear;          //!< time of last event buffer clear
	unsigned int buffertime;            //!< The time between clearings of the buffer
																			/*!< - 0 will not use the buffer, events are routed upon posting
																			 *   - 1 will clear the buffer at next call to processTimers() or processEventBuffer()
																			 *   - a larger value will cause a delay of that number of milliseconds since the last clearing */
	
	//! Does the actual storage of the mapping between EventBase's and the EventListeners/EventTrappers who should receive them
	/*! Actually only stores void*'s, so it's more general than just Listeners or Trappers */
	class EventMapper {
	public:
		//! constructor
		EventMapper();

		void addMapping(void* el, EventBase::EventGeneratorID_t egid) { allevents[egid].push_back(el); } //!< Adds a listener for all events from a given event generator
		void addMapping(void* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid); //!< Adds a listener for a specific source id and type from a given event generator

		//! Removes a listener for all events from a given event generator, returns true if something was actually removed
		/*! Doesn't necessarily remove the vector or mapping if this was the last listener, use clean() to do that */
		bool removeMapping(void* el, EventBase::EventGeneratorID_t egid); 

		//! Removes a listener for a specific source id and type from a given event generator, returns true if something was actually removed
		/*! Doesn't necessarily remove the vector or mapping if this was the last listener, use clean() to do that */
		bool removeMapping(void* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid);

		void clean(); //!<removes empty data structures for all event generators
		void clean(EventBase::EventGeneratorID_t egid); //!< removes empty data structures associated with a single event generator
		void clear(); //!<Resets the mapping

		//@{
		//! so stuff can tell if it even needs to bother generating an event...
		/*! ... if a tree falls in a forest, and there's no one around to see it, does it make a sound?\n
			  ... if Vision sees a ball in an image, and there's no listeners, does it make an event? ;) \n
			  @return true if it has any listeners, false otherwise */
		bool hasMapping(EventBase::EventGeneratorID_t egid);
		bool hasMapping(EventBase::EventGeneratorID_t egid, unsigned int sid);
		bool hasMapping(EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid);
		//@}

		//! builds a list of all listeners which should receive the event, templated to typecast the pointers for you
		/*! @param e the key event
		 *  @param listeners upon return, the resulting list of listeners @a e maps to\n
		 *  @a listeners is not cleared prior to building, new listeners are pushed on end\n
		 *  Results are in the order: all specific matches first, all generator listeners second, in order they were added to the EventMapper.*/
		template<class T>
		void getMapping(const EventBase& e, std::vector<T*>& listeners);

		//! Used to make sure that the specified listener exists for the given event
		/*! This is needed because after we call processEvent on a lister, we can't assume
		 *  that no other listeners have been modified - one listener could cause another
		 *  to turn off.  If that has happened, we shouldn't send the event, even if it
		 *  was in the queue originally. */
		bool verifyMapping(void * listener, const EventBase& e) { return verifyMapping(listener,e.getGeneratorID(),e.getSourceID(),e.getTypeID()); }
		//! Used to make sure that the specified listener exists for the given event
		/*! This is needed because after we call processEvent on a lister, we can't assume
		 *  that no other listeners have been modified - one listener could cause another
		 *  to turn off.  If that has happened, we shouldn't send the event, even if it
		 *  was in the queue originally. */
		bool verifyMapping(void * listener, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid);

		//! Needed to complete EventRouter::isListening suite
		/*! Only checks #allevents */
		bool verifyMappingAll(void * listener, EventBase::EventGeneratorID_t egid);
		//! Needed to complete EventRouter::isListening suite
		/*! Checks both #allevents and #filteredevents */
		bool verifyMappingAny(void * listener, EventBase::EventGeneratorID_t egid);

		//! Needed to complete EventRouter::isListening suite
		/*! Checks both #allevents and #filteredevents, must be found in all */
		bool verifyMappingAll(void * listener, EventBase::EventGeneratorID_t egid, unsigned int sid);
		//! Needed to complete EventRouter::isListening suite
		/*! Checks both #allevents and #filteredevents, can be found in either */
		bool verifyMappingAny(void * listener, EventBase::EventGeneratorID_t egid, unsigned int sid);

	protected:
		//! a mapping from source IDs (unsigned ints), each to a vector of pointers to listeners
		/*! main use in filteredevents @see filteredevents */
		typedef std::map<unsigned int,std::vector<void*>,std::less<unsigned int> > SIDtoListenerVectorMap_t;
		
		//! an array of vectors of pointers to listeners... in other words, a vector of listener pointers for each generator
		std::vector<void*> allevents[EventBase::numEGIDs];
		//! not for the faint of heart: a matrix of mappings to vectors of pointers to listeners
		SIDtoListenerVectorMap_t* filteredevents[EventBase::numEGIDs][EventBase::numETIDs];

	private:
		EventMapper(const EventMapper&);           //!< this shouldn't be called...
		EventMapper operator=(const EventMapper&); //!< this shouldn't be called...
	};

	EventMapper trappers;  //!< A mapping of which EventTrapper's should get a chance to trap the event
	EventMapper listeners; //!< A mapping of which EventListener's should receive events

};

//! a global router for cross communication, probably the most common usage, although perhaps there may be times you'd rather have multiple event routers for smaller sections
extern EventRouter * erouter;

/*! @file
 * @brief Describes EventRouter class, for distribution and trapping of events to listeners
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4 $
 * $Revision: 1.21 $
 * $State: Exp $
 * $Date: 2005/08/07 04:11:03 $
 */

#endif
