#include "EventBase.h"
#include <stdio.h>
#include <sstream>
#include <libxml/tree.h>
#include "Shared/debuget.h"

const char* const EventBase::EventGeneratorNames[numEGIDs] = {
	"unknownEGID",
	"aiEGID",
	"audioEGID",
	"buttonEGID",
	"erouterEGID",
	"estopEGID",
	"locomotionEGID",
	"micOSndEGID",
	"micRawEGID",
	"micFFTEGID",
	"motmanEGID",
	"powerEGID",
	"sensorEGID",
	"stateMachineEGID",
	"stateTransitionEGID",
	"textmsgEGID",
	"timerEGID",
	"visOFbkEGID",
	"visRawCameraEGID",
	"visInterleaveEGID",
	"visJPEGEGID",
	"visSegmentEGID",
	"visRLEEGID",
	"visRegionEGID",
	"visObjEGID",
	"wmVarEGID",
	"worldModelEGID",
};

const char* const EventBase::EventTypeNames[numETIDs] = {
	"activate",
	"status",
	"deactivate"
};

const char* const EventBase::EventTypeAbbr[numETIDs] = { "A", "S", "D" };

EventBase::EventBase()
	: XMLLoadSave(), stim_id(), magnitude(0), timestamp(get_time()), saveFormat(XML), nameisgen(true), genID(unknownEGID), typeID(statusETID), sourceID((unsigned int)-1), duration(0)
{
	genName();
}

EventBase::EventBase(EventGeneratorID_t gid, unsigned int sid, EventTypeID_t tid, unsigned int dur)
	: XMLLoadSave(), stim_id(), magnitude(0), timestamp(get_time()), saveFormat(XML), nameisgen(true), genID(gid), typeID(tid), sourceID(sid), duration(dur)
{
	genName();
	if(tid==deactivateETID)
		setMagnitude(0.0);
	else
		setMagnitude(1.0);
}

EventBase::EventBase(EventGeneratorID_t gid, unsigned int sid, EventTypeID_t tid, unsigned int dur, const std::string& n)
	: XMLLoadSave(), stim_id(), magnitude(0), timestamp(get_time()), saveFormat(XML), nameisgen(true), genID(gid), typeID(tid), sourceID(sid), duration(dur)
{
	setName(n);
	if(tid==deactivateETID)
		setMagnitude(0.0);
	else
		setMagnitude(1.0);
}

EventBase::EventBase(EventGeneratorID_t gid, unsigned int sid, EventTypeID_t tid, unsigned int dur, const std::string& n, float mag)
	: XMLLoadSave(), stim_id(), magnitude(mag), timestamp(get_time()), saveFormat(XML), nameisgen(true), genID(gid), typeID(tid), sourceID(sid), duration(dur)
{
	setName(n);
}

EventBase&
EventBase::setName(const std::string& sourcename) {
	stim_id='(';
	if(genID<numEGIDs) {
		stim_id+=EventGeneratorNames[genID];
	} else {
		stim_id+=std::string("InvalidGen");
		char tmp2[16];
		snprintf(tmp2,16,"(%d)",genID);
		stim_id+=tmp2; 
	}
	stim_id+=',';
	stim_id+=sourcename;
	stim_id+=',';
	stim_id+=EventTypeAbbr[getTypeID()];
	stim_id+=')';
	nameisgen=false;
	return *this;
}

std::string
EventBase::getDescription(bool /*showTypeSpecific=true*/, unsigned int verbosity/*=0*/) const {
	std::ostringstream logdata;
	logdata << getName();
	if(verbosity>=1)
		logdata << '\t' << getGeneratorID() << '\t' << getSourceID() << '\t' << getTypeID();
	if(verbosity>=2)
		logdata << '\t' << getDuration() << '\t' << getTimeStamp();
	if(verbosity>=3)
		logdata << '\t' << getMagnitude();
	return logdata.str();
}

unsigned int
EventBase::getBinSize() const {
	if(saveFormat==XML)
		return XMLLoadSave::getBinSize();
	unsigned int used=0;
	used+=creatorSize("EventBase");
	used+=stim_id.size()+stringpad;
	used+=sizeof(magnitude);
	used+=sizeof(timestamp);
	used+=sizeof(nameisgen);
	used+=sizeof(char);
	used+=sizeof(char);
	used+=sizeof(sourceID);
	used+=sizeof(duration);
	return used;
}

unsigned int
EventBase::LoadBinaryBuffer(const char buf[], unsigned int len) {
	unsigned int origlen=len;
	unsigned int used=0;
	if(0==(used=checkCreator("EventBase",buf,len,true))) return 0;
	len-=used; buf+=used;
	if(0==(used=decode(stim_id,buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=decode(magnitude,buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=decode(timestamp,buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=decode(nameisgen,buf,len))) return 0;
	len-=used; buf+=used;
	char tmp;
	if(0==(used=decode(tmp,buf,len))) return 0;
	genID=(EventGeneratorID_t)tmp;
	len-=used; buf+=used;
	if(0==(used=decode(tmp,buf,len))) return 0;
	typeID=(EventTypeID_t)tmp;
	len-=used; buf+=used;
	if(0==(used=decode(sourceID,buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=decode(duration,buf,len))) return 0;
	len-=used; buf+=used;
	return origlen-len;	
}

unsigned int
EventBase::SaveBinaryBuffer(char buf[], unsigned int len) const {
	unsigned int origlen=len;
	unsigned int used=0;
	if(0==(used=saveCreator("EventBase",buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=encode(stim_id,buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=encode(magnitude,buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=encode(timestamp,buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=encode(nameisgen,buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=encode((char)genID,buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=encode((char)typeID,buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=encode(sourceID,buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=encode(duration,buf,len))) return 0;
	len-=used; buf+=used;
	return origlen-len;
}

void EventBase::LoadXML(xmlNode* node) {
	if(xmlStrcmp(node->name, (const xmlChar *)"event"))
		throw bad_format(node,"Load of the wrong type -- expecting 'event' node");
	unsigned int i;
	xmlChar* str = xmlGetProp(node,(const xmlChar*)"egid");
	if(str==NULL)
		throw bad_format(node,"missing generator id");
	for(i=0; i<numEGIDs; i++) {
		if(xmlStrcmp(str,(const xmlChar*)EventGeneratorNames[i])==0)
			break;
	}
	xmlFree(str);
	if(i==numEGIDs)
		throw bad_format(node,"bad event generator name");
	genID=static_cast<EventGeneratorID_t>(i);
	str = xmlGetProp(node,(const xmlChar*)"sid");
	if(str==NULL)
		throw bad_format(node,"missing source id");
	sourceID=atoi((const char*)str);
	xmlFree(str);
	str = xmlGetProp(node,(const xmlChar*)"etid");
	if(str==NULL)
		throw bad_format(node,"missing type id");
	for(i=0; i<numETIDs; i++) {
		if(xmlStrcmp(str,(const xmlChar*)EventTypeAbbr[i])==0)
			break;
		if(xmlStrcmp(str,(const xmlChar*)EventTypeNames[i])==0)
			break;
	}
	xmlFree(str);
	if(i==numETIDs)
		throw bad_format(node,"bad event type name");
	typeID=static_cast<EventTypeID_t>(i);
	str = xmlGetProp(node,(const xmlChar*)"time");
	if(str==NULL)
		throw bad_format(node,"missing timestamp");
	timestamp=atoi((const char*)str);
	xmlFree(str);
	str = xmlGetProp(node,(const xmlChar*)"duration");
	if(str==NULL)
		throw bad_format(node,"missing duration");
	duration=atoi((const char*)str);
	xmlFree(str);
	str = xmlGetProp(node,(const xmlChar*)"magnitude");
	if(str==NULL)
		throw bad_format(node,"missing magnitude");
	magnitude=atoi((const char*)str);
	xmlFree(str);
	str = xmlGetProp(node,(const xmlChar*)"custom_name");
	if(str!=NULL) {
		setName((const char*)str);
		xmlFree(str);
	} else
		genName();
}
void EventBase::SaveXML(xmlNode * node) const {
	xmlNodeSetName(node,(const xmlChar*)"event");
	xmlSetProp(node,(const xmlChar*)"egid",(const xmlChar*)EventGeneratorNames[genID]);
	char buf[20];
	snprintf(buf,20,"%x",sourceID);
	xmlSetProp(node,(const xmlChar*)"sid",(const xmlChar*)buf);
	xmlSetProp(node,(const xmlChar*)"etid",(const xmlChar*)EventTypeAbbr[typeID]);
	snprintf(buf,20,"%u",timestamp);
	xmlSetProp(node,(const xmlChar*)"time",(const xmlChar*)buf);
	snprintf(buf,20,"%u",duration);
	xmlSetProp(node,(const xmlChar*)"duration",(const xmlChar*)buf);
	snprintf(buf,20,"%g",magnitude);
	xmlSetProp(node,(const xmlChar*)"magnitude",(const xmlChar*)buf);
	if(!nameisgen) {
		//extract custom name
		size_t b=stim_id.find(',');
		ASSERTRET(b!=std::string::npos,"malformed event name! (no comma)");
		size_t e=stim_id.find(',',++b);
		ASSERTRET(e!=std::string::npos,"malformed event name! (no 2nd comma)");
		xmlSetProp(node,(const xmlChar*)"custom_name",(const xmlChar*)stim_id.substr(b,e-b).c_str());
	}
}

void
EventBase::genName() {
	if(!nameisgen)
		return;
	char tmp[16];
	snprintf(tmp,16,"%d",sourceID);
	setName(tmp);
	nameisgen=true;
}

/*! @file
 * @brief Implements EventBase, the basic class for sending events around the system
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4 $
 * $Revision: 1.25 $
 * $State: Exp $
 * $Date: 2005/06/01 05:47:46 $
 */

