/** @file TrainCanvas.java
 *  @brief Canvas for EasyTrain        
 *
 *	@author editted by: Eric Durback
 *  @bug No known bugs.
 */

import java.awt.image.*;
import java.awt.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.event.*;
import java.util.*;
import java.awt.geom.*;
import java.io.*;
import java.util.prefs.Preferences;

public class TrainCanvas extends Canvas implements MouseListener,
MouseMotionListener
{
	BufferedImage image;
	BufferedImage cachedplot;
	
	ColorControlPanel ControlPanel;

	Graphics2D graphics;
	Polygon curPoly;
	int lastx, lasty;
	float f_width, f_height, f_offset;
	int numImages, curImage;
	String [] files;
	String defaultName;

	Area curColorArea;
	Area [] curImageArea;
	Area [] originalImageArea;

	String curColor;

	Point imageHint, imagePlace;

	Hashtable colorAreasHash;
	Hashtable imageAreasHash;
	Hashtable originalImageAreasHash;

	Hashtable colorMaskHash;

	boolean inverted, colorshow, autoSelect, dirtyOriginal;
	
	public final static int NOTHING = 0;
	public final static int IMAGE_AREA = 1;
	public final static int COLOR_AREA = 2;
	
	Object undoObj;
	int lastAction;
	int uImg;
	
	float[] xy;
	int[] rgb;

	SegmentedImage segImage;
	ImageShowArea imageShow;
	ImageData imageData;
	ImageData YUVimageData;
	HelpBox helpBox;

	boolean areaInImage = false;
	
	static Preferences prefs = Preferences.userNodeForPackage(EasyTrain.class);


	public TrainCanvas(SegmentedImage _segImage, ImageShowArea _imageShow, ImageData _imageData, ImageData _YUVimageData, ColorControlPanel c_panel, HelpBox _helpBox, String [] files_ ) 
	{

		//setBackground(Color.black);
		addMouseListener(this);
		addMouseMotionListener(this);
	
		colorAreasHash=new Hashtable();
		imageAreasHash = new Hashtable();
		colorMaskHash = new Hashtable();
		originalImageAreasHash = new Hashtable();
		ControlPanel = c_panel;
		helpBox = _helpBox;
		
		inverted=false;
		colorshow=false;
		dirtyOriginal = true;
		imageHint=null;
		imagePlace=null;
		segImage = _segImage;
		imageShow = _imageShow;
		imageData = _imageData;
		YUVimageData = _YUVimageData;
		numImages = files_.length;
		curImage = 0;
		undoObj = null;
		lastAction = NOTHING;
		uImg = 0;		
		
		curImageArea=new Area[numImages];
		for(int i=0; i<numImages; i++)
		{
			curImageArea[i] = new Area();
		}
		originalImageArea = new Area[numImages];

		files = files_;    

	}

	public void setControlPanel(ColorControlPanel c)
	{
		ControlPanel = c;
	}


	public void plotImage() 
	{
		plotImage(this.xy,this.rgb);
	}

	public void plotImage(float[] xy, int[] rgb) 
	{
		
		this.xy=xy;
		this.rgb=rgb;
		
		Dimension d=getSize();
		f_width=(float)d.width-41;
		f_height=(float)d.height-41;
		f_offset=20;

		cachedplot=new BufferedImage(d.width, d.height, BufferedImage.TYPE_INT_RGB);
		graphics=(Graphics2D)cachedplot.getGraphics();
		
		if(inverted)
		{
			graphics.setColor(Color.white);
			graphics.fillRect(0,0,d.width, d.height);
		}
		
		for (int i=0; i<rgb.length; i++) 
		{
		 
		  graphics.setColor(new Color(rgb[i]));
		  graphics.drawRect((int)(xy[i*2]*f_width+f_offset),
		                    (int)(xy[i*2+1]*f_height+f_offset), 1,1);
		}

		image=new BufferedImage(d.width, d.height, BufferedImage.TYPE_INT_RGB);
		graphics=(Graphics2D)image.getGraphics();
				
		redrawScene();
		repaint();
	}

	public void plotImageArea(Area [] area)
	{ 
				
		float[][] xy = imageData.getAllHS();		
		int[][] rgb = imageData.getAllPixels();
		
		double xMod = imageShow.resizedX;
		double yMod = imageShow.resizedY;

		double width = imageShow.getSize().getWidth();
		double height = imageShow.getSize().getHeight();		
		
		Rectangle curRect;

		Dimension d=getSize();
		image = new BufferedImage(d.width, d.height, BufferedImage.TYPE_INT_RGB);
		Graphics2D graphics = (Graphics2D)image.getGraphics();		
		
		if(inverted)
		{
			graphics.setColor(Color.white);
			graphics.fillRect(0,0,d.width, d.height);
		}

		for(double y=0; y<height; y+=2.0 * yMod)
		{
		    for (double x=0; x<width; x+=2.0 * xMod)
			{
				for(int k=0; k<numImages; k++)
				{
				    if(area[k].contains((int)x,(int)y))
					{
					    int i=(int) ((((double)y/yMod)/2.0)*(((double)width/xMod)/2.0)+((double)x/xMod)/2.0);
					    
					    int w = (int)(xy[k][i*2]*f_width+f_offset);	//int w = (int)(xy[k][i*2]*f_width+f_offset);
					    int h = (int)(xy[k][i*2+1]*f_height+f_offset);	//int h = (int)(xy[k][i*2+1]*f_height+f_offset);
					     
					    graphics.setColor(new Color(rgb[k][i]));	//graphics.setColor(new Color(rgb[k][i]));
					    graphics.drawRect(w, h, 1,1);
						
						/*if(autoSelect && !curColorArea.contains(w,h))
						{						
							curRect=new Rectangle(w -1, h-1,3,3); 								    
					    	curColorArea.add(new Area(curRect));
						}*/
					}
				}
			}
		}

		repaint();
	}


	
	public void paint(Graphics g) 
	{
		update(g);
	}

	public void update(Graphics g) 
	{
		Graphics2D g2d=(Graphics2D) g;
		if (image!=null) 
			g.drawImage(image, 0, 0, this);

		if (inverted)
			g2d.setColor(Color.black);
		else
			g2d.setColor(Color.white);

		if (curColorArea!=null)
			g2d.draw(curColorArea);

		if (curPoly!=null && !areaInImage)
			g2d.draw(curPoly);

		if (imageHint!=null) 
		{
			g2d.drawRect(imageHint.x-2, imageHint.y-2, 5, 5);
		}
	}

	public void setCurColor(String color) 
	{

		if (color==null) 
		{
			curImageArea=null;
			curColorArea = null;
			return;
		}
		
		undoObj = null;
		lastAction = NOTHING;
		ControlPanel.disableUndo();

		curImage = imageShow.getCurImage();

		curColorArea = (Area)colorAreasHash.get(color);
		curImageArea = (Area[])imageAreasHash.get(color);
		originalImageArea = (Area[])originalImageAreasHash.get(color);

		if (curImageArea==null) 
		{
			curImageArea=new Area[numImages];
			for(int i=0; i<numImages; i++)
			{
				curImageArea[i] = new Area();
			}

			imageAreasHash.put(color, curImageArea);
			curColorArea = new Area();

			colorAreasHash.put(color, curColorArea);
		}

		imageShow.setCurArea(curImageArea[curImage]);

		if(!colorshow)
		{
			plotImageArea(curImageArea);
		}
		else
		{
			plotImage();
		}

		redrawScene();
	}
	
	public void undo ()
	{
	
		if(lastAction == IMAGE_AREA)
		{		
			curImageArea[uImg] = ((Area)undoObj).createTransformedArea(AffineTransform.getScaleInstance(imageShow.resizedX,imageShow.resizedY));
			imageShow.setCurArea(curImageArea[curImage]);
			plotImageArea(curImageArea);	
			dirtyOriginal = true;	
		}
		else if(lastAction == COLOR_AREA)
		{
			curColorArea.reset();
			curColorArea.add((Area)((Area)undoObj));
						
		}
		
		repaint();
		ControlPanel.disableUndo();
		segImage.reSegment(makeTmap(), getAverageColors());
		
	}
	
	public void help()
	{
		
		helpBox.setVisible(true);
		helpBox.setState(helpBox.NORMAL);
		
	}
	
	public void clear () 
	{
		if (curImageArea==null) return;

		curImage = imageShow.getCurImage();
		curImageArea[curImage].reset();
		curColorArea.reset();
		
		ControlPanel.disableUndo();
		
		
		imageShow.setCurArea(null);
		redrawScene();
		segImage.reSegment(makeTmap(), getAverageColors());		
		
		if(colorshow)
		{
			plotImage();
		}
	}

	public void invert () 
	{
		inverted=!inverted;
		
		if(inverted)
		{
			setBackground(Color.white);
		}
		else
		{
			setBackground(Color.black);
		}
		
		if(curImageArea != null && !colorshow)
		{
			plotImageArea(curImageArea);
		}
		else
		{
			plotImage();
		}

	}

	public void showColors () 
	{
		colorshow=!colorshow;
		
		if(colorshow)
		{
			plotImage();
		}
		else if(curImageArea != null)
		{
			plotImageArea(curImageArea);
		}		
		repaint();
	}

	public void autoSelect () //not being used
	{
		autoSelect=true;

		if(autoSelect)
		{
			plotImageArea(curImageArea);
			segImage.reSegment(makeTmap(), getAverageColors());
		}
		
		if(colorshow)
		{
			plotImage();
		}

		autoSelect = false;

	}

	public void remove(String color) 
	{
		curColor=null;
		curColorArea=null;
		curImageArea=null;
		colorAreasHash.remove(color);
		imageAreasHash.remove(color);		

		imageShow.setCurArea(null);

		redrawScene();
	}

	public void redrawScene()
	{
		graphics.drawImage(cachedplot,0,0,this);
		  		
		if (curColorArea != null)
		graphics.draw(curColorArea);
		
	}

	
	public byte[] makeTmap()
	{
		ArrayList areas = makeAreas();
		//boolean maps[][][] = makeMasks();

		int size_y=16, size_u=64, size_v=64;
		byte[] tmap=new byte[size_y*size_u*size_v];

		float[] hsb=new float[3];
		int y, u, v, r, g, b, h, s;
		int iy, iu, iv, i;

		for (iy=0; iy<16; iy++) 
		{
			for (iu=0; iu<64; iu++) 
			{
				for (iv=0; iv<64; iv++) 
				{
				    y=iy<<4; u=iu<<2; v=iv<<2;
				    u=u*2-255;
				    v=v*2-255;
				    r=y+u;
				    b=y+v;
				    u=u>>1;
				    v=(v>>2)-(v>>4);
				    g=y-u-v;
				    if (r<0) r=0; 
				    if (g<0) g=0; 
				    if (b<0) b=0;
				    if (r>255) r=255; 
				    if (g>255) g=255; 
				    if (b>255) b=255;
				    
				    float[] xy=EasyTrain.colorConverter.getColor(r, g, b);
				    h=(int)(xy[0]*f_width+f_offset);
				    s=(int)(xy[1]*f_height+f_offset);
				    Point point=new Point(h, s);
				    
				    for (i=0; i<areas.size(); i++) 
				    {
						
						Area area = (Area) areas.get(i);
						if (area.contains(point)) 
						{
						    tmap[(iy*size_u+iu)*size_v+iv]=(byte)(i+1);
						    break;
						}
				    }
				}
	    	}
		}

		return tmap;
	
	}   
	//TODO make a weighted average version
	public int[][] getAverageColors()
	{
		ArrayList areas = makeAreas();

		rgb=imageData.getSpecialPixels();

		int[][] avgcolors=new int[areas.size()][4]; 

		int skip=(rgb.length/10000)+1;
		
		for (int i=0; i<rgb.length; i+=skip) 
		{
		    for (int a=0; a<areas.size(); a++) 
		    {
			
				Area area=(Area)areas.get(a);
				Point p=new Point((int)(xy[i*2]*f_width+f_offset),
						  (int)(xy[i*2+1]*f_height+f_offset));
				
				if (area.contains(p)) 
				{
				    avgcolors[a][0]+=(rgb[i]>>16)&0xff;
				    avgcolors[a][1]+=(rgb[i]>>8)&0xff;
				    avgcolors[a][2]+=rgb[i]&0xff;
				    avgcolors[a][3]++;
				    
				}
		    }	    
		}

		for (int i=0; i<avgcolors.length; i++) 
		{
		    if (avgcolors[i][3]>0) 
		    {
				avgcolors[i][0]=avgcolors[i][0]/avgcolors[i][3];
				avgcolors[i][1]=avgcolors[i][1]/avgcolors[i][3];
				avgcolors[i][2]=avgcolors[i][2]/avgcolors[i][3];
		    } 
		    else 
		    {
				avgcolors[i][0]=0;
				avgcolors[i][1]=0;
				avgcolors[i][2]=0;
		    }
		}
		return avgcolors;
	}

	public ArrayList makeAreas()
	{
	
		Enumeration colors=colorAreasHash.keys();
		ArrayList colornames=new ArrayList(20);

		while (colors.hasMoreElements()) colornames.add(colors.nextElement());

		Collections.sort(colornames);
		ArrayList areas=new ArrayList(20);

		for (Iterator i=colornames.iterator(); i.hasNext();
		     areas.add(colorAreasHash.get(i.next())));

		return areas;

	}	

	public ArrayList makeImageAreas()
	{

		Enumeration colors=imageAreasHash.keys();
		ArrayList colornames=new ArrayList(20);

		while (colors.hasMoreElements()) colornames.add(colors.nextElement());

		Collections.sort(colornames);
		ArrayList areas=new ArrayList(20);

		for (Iterator i=colornames.iterator(); i.hasNext();
		     areas.add(imageAreasHash.get(i.next())));

		return areas;
	}

	public boolean[][][] makeMasks()
	{
		
		Enumeration colors=imageAreasHash.keys();
		ArrayList colornames=new ArrayList(20);
		while (colors.hasMoreElements()) colornames.add(colors.nextElement());
		Collections.sort(colornames);
		ArrayList areas=new ArrayList(20);

		for (Iterator i=colornames.iterator(); i.hasNext();
		     areas.add(imageAreasHash.get(i.next())));

		boolean[][][] masks = new boolean[areas.size()][(int)f_width+1][(int)f_height+1];

		for (int i=0; i<areas.size(); i++){
		    for(int x=0; x<f_width; x++){
				for(int y=0; y<f_height; y++)
			    {
					masks[i][x][y]=false;
			    }
			}
		}
		//something wrong here
		//xy = hue saturation, this is probably from segmenter
		float xy[][] = imageData.getAllHS();
		double imageWidth = imageShow.getSize().getWidth();
		double imageHeight = imageShow.getSize().getHeight();
		for (int i=0; i<areas.size(); i++)
		{
			for (int y=0; y<imageHeight; y+=2)
			{
			    for (int x=0; x<imageWidth; x+=2)
				{
				    Area[] a = (Area[])areas.get(i);
				    for(int k=0; k<numImages; k++)
					{
				    	if (a[k].contains(x,y))
						{
					    	int j=(y/2)*(((int)imageWidth)/2)+x/2;
					    	masks[i][(int)(xy[k][2*j]*f_width)][(int)(xy[k][2*j+1]*f_height)] = true;
					    
					    	//int j=(y/2)*(((int)imageWidth))+x/2;
					    	//masks[i][(int)(xy[j]*f_width)][(int)(xy[j+1]*f_height)] = true;
						}
					}
				}	
			}
		}

		return masks;
	}

	public void load (String filename) throws FileNotFoundException, IOException
	{
		int dotpos=filename.lastIndexOf('.');
		if (dotpos>0) filename=filename.substring(0,dotpos);

		colorAreasHash = new Hashtable();
		imageAreasHash = new Hashtable();

		curImageArea[curImage].reset();
		curColorArea.reset();

		BufferedReader saveFile=new BufferedReader(new FileReader(filename + ".save"));
		BufferedReader areaFile;

		saveFile.readLine();

		int numColors = Integer.parseInt(saveFile.readLine());
		String [] colors = new String[numColors];
		Polygon curPoly = new Polygon();;
		StringTokenizer inTok;
		int x,y;

		for(int j=0; j<numColors; j++)
		{
			String s = saveFile.readLine();
			colors[j] = s;
		}

		if(!saveFile.readLine().equals("NEXT"))
		{
			System.out.println("Error File Format is incorrect");
			return;
		}

		for(int i=0; i<numColors; i++)
		{
			
			for(String in = saveFile.readLine(); !((in.equals("NEXT")) || in.equals("")); in = saveFile.readLine())
			{
				inTok = new StringTokenizer(in);
				in = inTok.nextToken();
				
				if(in.equals("START"))
				{
					curPoly = new Polygon();
					x = (int)Double.parseDouble(inTok.nextToken());
					y = (int)Double.parseDouble(inTok.nextToken());
					curPoly.addPoint(x,y);
				}
				else if(in.equals("LINE"))
				{
					x = (int)Double.parseDouble(inTok.nextToken());
					y = (int)Double.parseDouble(inTok.nextToken());
					curPoly.addPoint(x,y);
				}
				else if(in.equals("CLOSE"))
				{
					//System.out.println("adding area");	
					curColorArea.add(new Area(curPoly));
				}
				
				//System.out.println("S");				
			}
			
			colorAreasHash.put(colors[i],curColorArea);
			curColorArea = new Area();
		}

			
		for(int l=0; l<files.length; l++)
		{
				
			try
			{
				areaFile = new BufferedReader(new FileReader(filename +"_"+files[l]+ ".area"));
			
				areaFile.readLine();
				for(int k=0; k<numColors; k++)
				{		
				
					curImageArea =(Area[])imageAreasHash.get(colors[k]);
					
					if(curImageArea == null)
					{
						curImageArea = new Area[numImages];
						for(int i=0; i<numImages; i++)
					    {
					      	curImageArea[i] = new Area();
					    }
		    		}
				    
				    
					for(String in = areaFile.readLine(); (!(in.equals("NEXT")) && areaFile.ready()); in = areaFile.readLine())
					{
						inTok = new StringTokenizer(in);
						in = inTok.nextToken();
						
						if(in.equals("START"))
						{
							curPoly = new Polygon();
							x = (int)Double.parseDouble(inTok.nextToken());
							y = (int)Double.parseDouble(inTok.nextToken());
							curPoly.addPoint(x,y);
						}
						else if(in.equals("LINE"))
						{
							x = (int)Double.parseDouble(inTok.nextToken());
							y = (int)Double.parseDouble(inTok.nextToken());
							curPoly.addPoint(x,y);
						}
						else if(in.equals("CLOSE"))
						{
							curImageArea[l].add(new Area(curPoly));
						}
						
					}
					imageAreasHash.put(colors[k],curImageArea);
				}
				
			}
			catch(Exception e)
			{
				for(int k=0; k<numColors; k++)
				{		
				
					curImageArea =(Area[])imageAreasHash.get(colors[k]);
					
					if(curImageArea == null)
					{
						curImageArea = new Area[numImages];
						for(int i=0; i<numImages; i++)
					    {
					      	curImageArea[i] = new Area();
					    }
		    		}
		    		
		    		imageAreasHash.put(colors[k],curImageArea);
		    	}
							
			}
			
		}
		segImage.reSegment(makeTmap(), getAverageColors());
		//setCurColor(null);
	}

	public void save (String filename) 
	{
		int dotpos=filename.lastIndexOf('.');
		if (dotpos>0) filename=filename.substring(0,dotpos);

		Enumeration colors=colorAreasHash.keys();
		ArrayList colornames=new ArrayList(20);
		while (colors.hasMoreElements()) colornames.add(colors.nextElement());
		Collections.sort(colornames);
		ArrayList areas=makeAreas();
		ArrayList imageAreas = makeImageAreas();

		int size_y=16, size_u=64, size_v=64;

		int i;
		byte[] tmap = makeTmap();

		try 
		{
			FileOutputStream file_tm_fos=new FileOutputStream(filename + ".tm");
			OutputStreamWriter file_tm_osw=new OutputStreamWriter(file_tm_fos);
			file_tm_osw.write("TMAP\nYUV8\n" +
			  size_y + " " + size_u + " " + size_v + "\n");
			file_tm_osw.flush();
			file_tm_fos.write(tmap);
			file_tm_osw.close();
		} 
		catch (Exception ex) 
		{
			System.out.println("Error saving to "+filename +".tm: " + ex);
		}
		
		int [][] avgcolors = getAverageColors();

		try 
		{
			FileWriter file_col_fw=new FileWriter(filename + ".col");
			file_col_fw.write("0 (128 128 128) \"unclassified\" 8 1.00\n");
			
			for (i=0; i<areas.size(); i++) 
			{
				file_col_fw.write((i+1)+ " (" +
				              avgcolors[i][0] + " " +
				              avgcolors[i][1] + " " +
				              avgcolors[i][2] + ") " +
				              "\"" + colornames.get(i)+ "\" 8 0.75\n");
			}
			
			file_col_fw.close();
		} 
		catch (Exception ex) 
		{
			System.out.println("Error saving to "+filename + ".col: " + ex);
		}

		try 
		{
			FileWriter file_save_fw=new FileWriter(filename + ".save");
			FileWriter file_area_fw;
			file_save_fw.write(filename + "\n");
			file_save_fw.write(areas.size() + "\n");

			for(int j=0; j<colornames.size(); j++)
			{
				file_save_fw.write(colornames.get(j) + "\n");	
			}      

			for (i=0; i<areas.size(); i++) 
			{
				Area a = (Area)areas.get(i);
				PathIterator path = a.getPathIterator(null);

				String pathString = createPathString(path);
				file_save_fw.write(pathString);  

			}

			file_save_fw.write("\n");
			file_save_fw.close();


			for(int l=0; l<files.length; l++)
			{
				
				String areaname = filename + "_"+ files[l];
	
				file_area_fw=new FileWriter(areaname + ".area");	

				for (i=0; i<imageAreas.size(); i++) 
				{
				    Area [] a = (Area[])imageAreas.get(i);
				    PathIterator path = a[l].getPathIterator(null);
				      	
				    String pathString = createPathString(path);
				    file_area_fw.write(pathString);  
				   
				}

				file_area_fw.write("\n");
				file_area_fw.close();

			}		  

		} catch (Exception ex) {
		  System.out.println("Error saving to "+filename + ".area: " + ex);
		}


	}

	String createPathString(PathIterator path)
	{

		String retPath = "NEXT\n";

		double[] coords = new double[6];
		int type;	

		while(!path.isDone())
		{
		  	type = path.currentSegment(coords); 
		  	
		  	
		  	if( (type == PathIterator.SEG_MOVETO)) //new polygon
		  	{
		  		retPath += "START " + coords[0] + " " + coords[1] + "\n";		
		  	}
		  	else if( (type == PathIterator.SEG_LINETO)) //line 
		  	{
		  		retPath += "LINE " + coords[0] + " " + coords[1] + "\n";
		  	}
		  	else if( (type == PathIterator.SEG_CLOSE))  //end of curr polygon
		  	{
		  		retPath += "CLOSE\n";
			}
			
			path.next();
			
		}

		return retPath;

	}

	byte[] readThresholdMap(String thresholdfile) 
	{
		try 
		{
			byte[] tmdata=new byte[65536];
			FileInputStream file_tm_fis=new FileInputStream(thresholdfile);
			file_tm_fis.read(tmdata,0,19);
			file_tm_fis.read(tmdata);
			file_tm_fis.close();
			return tmdata;
		} 
		catch (Exception ex) 
		{
			System.out.println("Error reading file "+thresholdfile+": "+ex);
			return null;
		}
	}



	void resizeAreas(double x, double y, double changeX, double changeY)
	{
		if(curImageArea != null)
		{
			Area[] curAreaList;
			Area curArea;
			
			updateAreas(x,y,changeX,changeY);			
			
			imageShow.setCurArea(curImageArea[curImage]);
			plotImageArea(curImageArea);

			repaint();
		}	  
	}
	
	void updateAreas(double x, double y, double changeX, double changeY)
	{
		Area[] curAreaList;
		Area curArea;
		Area[] origAreaList;
		Area origArea;
		Object o;
		
		Enumeration colors=imageAreasHash.keys();
		ArrayList colornames=new ArrayList();
		while (colors.hasMoreElements()) colornames.add(colors.nextElement());
		
		for (int i=0; i<colornames.size(); i++)
		{
			curAreaList = (Area[])imageAreasHash.get(colornames.get(i));
			
			o = originalImageAreasHash.get(colornames.get(i));
			
			if(o == null)
			{
				origAreaList = new Area[numImages];
			}
			else
			{
				origAreaList = (Area[])o;
			}
				
			for(int j = 0; j < numImages; j++)
			{
				if(dirtyOriginal == true)
				{
					curAreaList[j] = curAreaList[j].createTransformedArea(AffineTransform.getScaleInstance(changeX,changeY));
					origAreaList[j] = curAreaList[j].createTransformedArea(AffineTransform.getScaleInstance(1.0/x,1.0/y));
				}
				else
				{
					curAreaList[j] = origAreaList[j].createTransformedArea(AffineTransform.getScaleInstance(x,y));
				}
				
			}
				
			originalImageAreasHash.put(colornames.get(i),origAreaList);
			imageAreasHash.put(colornames.get(i),curAreaList);
		}	
		
		dirtyOriginal = false;
	}
	
	//******* MOUSE CONTROLS ********
    
    public void mousePressed(MouseEvent e) 
	{
		if (e.getButton()!=MouseEvent.BUTTON1) return;

		if (curImageArea==null || curColorArea==null)
		{
			ControlPanel.addDefaultColor();    
		}

		if (e.getSource() instanceof ImageShowArea)
		{
		    areaInImage = true;
		    curPoly=new Polygon();
		    lastx=e.getX();
		    lasty=e.getY();
		    curPoly.addPoint(e.getX(), e.getY());
		    imageShow.setCurPoly(curPoly);		    
		}
		else
		{
		    areaInImage = false;		    
		    curPoly=new Polygon();
		    lastx=e.getX();
		    lasty=e.getY();
		    curPoly.addPoint(e.getX(), e.getY());
		    repaint();
		}
	}

	public void mouseDragged(MouseEvent e) 
	{	  
		if (e.getSource() instanceof ImageShowArea)
		{   	      
			int x=e.getX();
			int y=e.getY();

			if ((Math.abs(x-lastx)+Math.abs(y-lasty))>2) 
			{
				curPoly.addPoint(e.getX(), e.getY());
				lastx=x;
				lasty=y;
				imageShow.setCurPoly(curPoly);
			}
		}
		else
		{	      
			int x=e.getX();
			int y=e.getY();

			if ((Math.abs(x-lastx)+Math.abs(y-lasty))>2) 
			{
				curPoly.addPoint(e.getX(), e.getY());
				lastx=x;
				lasty=y;
				repaint();
			}
		}
	}

	public void mouseReleased(MouseEvent e) 
	{
		if (e.getButton()!=MouseEvent.BUTTON1) return;

		if (e.getSource() instanceof ImageShowArea)
		{
		    if (curImageArea==null)
		    {
		    	ControlPanel.addDefaultColor();		    
		    } 
		    
		    curImage = imageShow.getCurImage();

		    curPoly.addPoint(e.getX(), e.getY());
		    
		    if (curPoly.npoints>=3) 
		    {
				
				if(originalImageArea != null && originalImageArea[curImage] != null)
				{
					undoObj = new Area(originalImageArea[curImage]);
				}
				else
				{
					undoObj = curImageArea[curImage].createTransformedArea(AffineTransform.getScaleInstance(1.0/imageShow.resizedX,1.0/imageShow.resizedY));
				}
				
				ControlPanel.enableUndo();
				lastAction = IMAGE_AREA;
				uImg = curImage; 
				 
				if (e.isControlDown())
				{ 
				    curImageArea[curImage].subtract(new Area(curPoly));
				}
				else if (e.isShiftDown())
				{ 
					curImageArea[curImage].add(new Area(curPoly));
				}
				else
				{
					curImageArea[curImage].reset();
				 	
					curImageArea[curImage].add(new Area(curPoly));
				}
				dirtyOriginal = true;
		    }
		    
		    curPoly=null;
		    imageShow.setCurArea(curImageArea[curImage]);
		    imageShow.setCurPoly(null);
		    
		    

		    if(!colorshow)
		    {
		    	plotImageArea(curImageArea);
		    }
		}
		else //color area
		{
		    if (curColorArea==null) return;

		    curPoly.addPoint(e.getX(), e.getY());
		    
		    if (curPoly.npoints>=3) 
		    {
				
				undoObj = new Area(curColorArea);
				ControlPanel.enableUndo();
				lastAction = COLOR_AREA;
				
				
				if (e.isControlDown())
				{ 
				    curColorArea.subtract(new Area(curPoly));
				}
				else if (e.isShiftDown())
				{ 
				    curColorArea.add(new Area(curPoly));
				}
				else
				{
					curColorArea.reset();
					curColorArea.add(new Area(curPoly));
				}
			}
			
		    curPoly=null;
		    repaint();
		}
		
		segImage.reSegment(makeTmap(), getAverageColors());
	}

	public void mouseMoved(MouseEvent e)
	{
		Object source=e.getSource();
		curImage = imageShow.getCurImage();
		if (source instanceof ImageShowArea) 
		{
		  	
			ImageShowArea imageShow=(ImageShowArea)source;
			int rgb=imageShow.getPixel(e.getX(), e.getY());
			float[] xy=EasyTrain.colorConverter.getColor(rgb);
			int x=(int)(xy[0]*f_width+f_offset);
			int y=(int)(xy[1]*f_height+f_offset);
			imageHint=new Point(x,y);
			
			segImage.showHint(new Point((int) ((double)e.getX() / imageShow.resizedX),(int) ( (double)e.getY() / imageShow.resizedY)));
				
			repaint();
		}
		/* this currently is not working
		else if (source instanceof SegmentedImage) 
		{
			SegmentedImage seg=(SegmentedImage)source;
			int rgb=seg.getPixel(e.getX(), e.getY());
			float[] xy=TrainingTool.colorConverter.getColor(rgb);
			int x=(int)(xy[0]*f_width+f_offset);
			int y=(int)(xy[1]*f_height+f_offset);
			imageHint=new Point(x,y);
				
			segImage.showHint(null);
			
			repaint();
		 
		}*/
		else
		{
			segImage.showHint(null);
			imageHint = null;			
			repaint();
		}    
	}

	public void mouseClicked(MouseEvent e){}
	
	public void mouseExited(MouseEvent e)
	{
		if (e.getSource() instanceof ImageShowArea) 
		{
			imageHint=null;
			repaint();
		}
		else if(e.getSource() instanceof SegmentedImage)
		{
			imagePlace = null;
			repaint();
		}
	}
	
	public void mouseEntered(MouseEvent e){}
	public void updateLocation(MouseEvent e){}

 	

}
