/** @file SegmentedImage.java
 *  @brief Frame for displaying the segmented image        
 *
 *	@author editted by: Eric Durback
 *  @bug No known bugs.
 */

import java.awt.image.*;
import java.awt.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.event.*;
import java.util.*;
import java.awt.geom.*;
import java.io.*;
import java.util.prefs.Preferences;


public class SegmentedImage extends JFrame implements KeyListener, MouseListener, ComponentListener
{
	BufferedImage image;
	ImageData imageData;
	ImageShowArea imageShow;
	byte[] tmap;
	int [][] averageColors;
	Point hint;
	
	static Preferences prefs = Preferences.userNodeForPackage(SegmentedImage.class);

  
	public SegmentedImage (ImageData _imageData, ImageShowArea _imageShow) 
	{
		//STUB
		
		imageData = _imageData;
		image = null;
		imageShow = _imageShow;
		
		setBackground(Color.GRAY);
		setLocation(prefs.getInt("SegmentedImage.location.x",50),prefs.getInt("SegmentedImage.location.y",50));
        setTitle("Segmented " + imageShow.getNewTitle());
	
		addKeyListener(this);
		addComponentListener(this);
		addMouseListener(this);

		setVisible(true);
		repaint();
	}

	public void setImageData(ImageData _imageData)
	{
	
		imageData = _imageData;
		reSegment(tmap,averageColors);
	}

	public void reSegment(byte[] _tmap, int[][]_averageColors)
	{

		tmap = _tmap;
		averageColors = _averageColors;

		int[] data=imageData.getPixels();

		IndexColorModel cmodel=makeColorModel(averageColors);

		if (tmap==null || cmodel==null) return;

		setSize(imageData.image_width*2, imageData.image_height*2);
		image = new BufferedImage(imageData.image_width, imageData.image_height,
		    BufferedImage.TYPE_BYTE_INDEXED,cmodel);

		segmentImage(data, tmap, imageData.image_width, imageData.image_height);

		repaint();
	}

	public void paint(Graphics graphics) 
	{
		
		Dimension sz=getSize();
		if (image!=null)
		    graphics.drawImage(image, 0, 0, sz.width, sz.height, null);
	  

		if(hint!=null && image!=null)
		{
			graphics.drawRect(hint.x-2, hint.y-2, 5, 5);
		}



	}

	public void showHint(Point p)
	{

		hint = p;
		repaint();
	}


	public void segmentImage(int[] data, byte[] tmap, int width, int height) 
	{
		
		int size_y=16, size_u=64, size_v=64;
		byte[] imgdata=new byte[data.length];

		for (int i=0; i<data.length; i++) 
		{
		    int y=(data[i]>>16)&0xff;
		    int u=(data[i]>>8)&0xff;
		    int v=data[i]&0xff;
		    y=y>>4;
		    u=u>>2;
		    v=v>>2;
		    imgdata[i]=tmap[(y*size_u+u)*size_v+v];
		}

		image.getRaster().setDataElements(0,0,width,height,imgdata);
		repaint();
	}

	public IndexColorModel makeColorModel(int[][] averageColors) 
	{
		if (averageColors == null)
		    return null;
		
		byte[] byte_cmap=new byte[(averageColors.length+1)*3];


		int i;
		for (i=1; i<averageColors.length+1; i++) 
		{
		    //byte_cmap[i*3]=(byte) ((cmap[i]>>16) & 0xff);
		    //byte_cmap[i*3+1]=(byte) ((cmap[i]>>8) & 0xff);
		    //byte_cmap[i*3+2]=(byte) (cmap[i] & 0xff);
		    byte_cmap[i*3] = (byte)(averageColors[i-1][0]&0xff);
		    byte_cmap[i*3+1] = (byte)(averageColors[i-1][1]&0xff);
		    byte_cmap[i*3+2] = (byte)(averageColors[i-1][2]&0xff);
		}

		byte_cmap[0] = (byte)128;
		byte_cmap[1] = (byte)128;
		byte_cmap[2] = (byte)128;

		IndexColorModel cmodel=new IndexColorModel(7, averageColors.length + 1, byte_cmap,
						   0, false); 
		return cmodel;
	}

	public int getPixel(int x, int y) 
	{
		Dimension sz=getSize();
		x=(x * image.getWidth()) / sz.width;
		y=(y * image.getHeight()) / sz.height;
		return image.getRGB(x, y);
	}

	public void keyPressed(KeyEvent e) 
	{
		if (e.getKeyCode()==KeyEvent.VK_LEFT ||
		    e.getKeyCode()==KeyEvent.VK_UP ||
		    e.getKeyCode()==KeyEvent.VK_PAGE_UP ||
		    e.getKeyCode()==KeyEvent.VK_KP_UP ||
		    e.getKeyCode()==KeyEvent.VK_KP_LEFT) 
		{
			imageShow.curimg--;

			if (imageShow.curimg<0) imageShow.curimg+=imageShow.imglist.length;
				
			imageShow.imageChange(imageShow.curimg);
		} 
		else if (e.getKeyCode()==KeyEvent.VK_RIGHT ||
		    e.getKeyCode()==KeyEvent.VK_DOWN ||
		    e.getKeyCode()==KeyEvent.VK_PAGE_DOWN ||
		    e.getKeyCode()==KeyEvent.VK_KP_DOWN ||
		    e.getKeyCode()==KeyEvent.VK_KP_RIGHT) 
		{
			imageShow.curimg++;

			if (imageShow.curimg>=imageShow.imglist.length) imageShow.curimg-=imageShow.imglist.length;

			imageShow.imageChange(imageShow.curimg);
		}
	}
	public void keyReleased(KeyEvent e) { }
	public void keyTyped(KeyEvent e) { }

	public void mousePressed(MouseEvent e) 
	{
		
	}

	public void mouseDragged(MouseEvent e) 
	{	  
		
	}

	public void mouseReleased(MouseEvent e) 
	{

	}

	public void mouseMoved(MouseEvent e)
	{
		
	}

	public void mouseClicked(MouseEvent e){}

	public void mouseExited(MouseEvent e)
	{

		//imagePlace = null;
		repaint();

	}

	public void mouseEntered(MouseEvent e){}
	
	public void componentResized(ComponentEvent e) {}
	public void componentHidden(ComponentEvent e) { }
	
	public void componentMoved(ComponentEvent e)
	{ 
	
		prefs.putInt("SegmentedImage.location.x",getLocation().x);
		prefs.putInt("SegmentedImage.location.y",getLocation().y);

	}
	public void componentShown(ComponentEvent e) { }

}
