/** @file ImageShowArea.java
 *  @brief 
 *
 *  Frame for showing the image
 *
 *	@author editted by: Eric Durback
 *  @bug No known bugs.
 */

import java.awt.image.*;
import java.awt.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import java.awt.geom.*;
import java.util.prefs.Preferences;

public class ImageShowArea extends JFrame implements KeyListener, ComponentListener, FocusListener 
{
	BufferedImage _image;
	String[] imglist;
	byte[] tmap;
	ImageData imageData,YUVimageData;
	int curimg;
	int numImages;
	boolean isRGB;
	boolean sizeFixed;

	Polygon curPoly;
	Area curArea;
	TrainCanvas trainCanvas;
	double resizedX;
	double resizedY;
	double lastX,lastY;
	
	static Preferences prefs = Preferences.userNodeForPackage(ImageShowArea.class);
 
	public void focusGained(FocusEvent e) {} 

	public void focusLost(FocusEvent e) {}

	public static void usageAndExit() 
	{
		System.out.println("usage: java ImageShow (-isRGB|-isYUV) raw_image [raw images]");
		System.exit(1);		
	}

	public ImageShowArea (boolean isRGB, String args[], ImageData imgData) 
	{
	  	curPoly = null;
	  	curArea = null;
	  	resizedX=1.0;
	  	resizedY=1.0;
	  
	  
		numImages = args.length;
		imageData=imgData;
		YUVimageData = new ImageData(1);
		sizeFixed = false;

		this.isRGB=isRGB;
		if(isRGB)
		{
			imageData.loadRGBFileAsRGB(args[0]);
			imageData.loadFullRGBFilesAsRGB(args);
		}
		else
		{
			imageData.loadYUVFileAsRGB(args[0]);
			imageData.loadFullYUVFilesAsRGB(args);
		}

		int[] data=imageData.getPixels();

		setBackground(Color.black);
		setSize(imageData.image_width*2, imageData.image_height*2);
		setTitle((curimg+1) + ": " + args[0]);
		setLocation(prefs.getInt("ImageShowArea.location.x",50),prefs.getInt("ImageShowArea.location.y",50));

		_image=new BufferedImage(imageData.image_width, imageData.image_height,
		    BufferedImage.TYPE_INT_RGB);
		    
		lastX = (double)getWidth();
		lastY = (double)getHeight();

		showImage(data, tmap, imageData.image_width, imageData.image_height);

		imglist=new String[numImages];
		curimg=0;
		imageData.setCurimg(curimg);
		
		//putting images names into the image list
		for (int i=0; i<numImages; i++) 
		{
			imglist[i]=args[i];
		}

		addKeyListener(this);
		addComponentListener(this);
		addFocusListener(this);  

	}

	void showImage(int[] data, byte[] tmap, int width, int height) 
	{
		_image.getRaster().setDataElements(0,0,width,height,data);
		repaint();
	}

	public void setCurArea(Area _curArea)
	{
		curArea = _curArea;
		repaint();
	}

	public void setCurPoly(Polygon _curPoly)
	{
		curPoly = _curPoly;
		repaint();
	}

	public void paint(Graphics graphics) 
	{
		Dimension sz=getSize();
		if (_image!=null)
		  graphics.drawImage(_image, 0, 0, sz.width, sz.height, null);

		graphics.setColor(Color.white);
		if (curArea!=null)
		  ((Graphics2D)graphics).draw(curArea);

		if (curPoly!=null)
		  ((Graphics2D)graphics).draw(curPoly);
	}

	public int getPixel(int x, int y) 
	{
		Dimension sz=getSize();
		x=(x*_image.getWidth())/sz.width;
		y=(y*_image.getHeight())/sz.height;
		return _image.getRGB(x, y);
	}

	public void imageChange(int curimg)
	{
		imageData.setCurimg(curimg);

		if(isRGB)
		{
		  	imageData.loadRGBFileAsRGB(imglist[curimg]);
		  	YUVimageData.loadRGBFileAsYUV(imglist[curimg]);
		}
		else
		{
		  	imageData.loadYUVFileAsRGB(imglist[curimg]);
		  	YUVimageData.loadYUVFileAsYUV(imglist[curimg]);
		}
					
		setTitle(getNewTitle());
		trainCanvas.segImage.setTitle("Segmented " + getNewTitle());
		  
		if(trainCanvas.curImageArea != null)
		 	setCurArea(trainCanvas.curImageArea[curimg]);
		  
		trainCanvas.segImage.setImageData(YUVimageData);
		  			
		int[] data=imageData.getPixels();
		showImage(data, tmap, imageData.image_width, imageData.image_height);	

	}

	public String getNewTitle()
	{
		return (curimg+1) + ": " +imglist[curimg];
	}

	public void keyPressed(KeyEvent e) 
	{
		if (e.getKeyCode()==KeyEvent.VK_LEFT ||
		    e.getKeyCode()==KeyEvent.VK_UP ||
		    e.getKeyCode()==KeyEvent.VK_PAGE_UP ||
		    e.getKeyCode()==KeyEvent.VK_KP_UP ||
		    e.getKeyCode()==KeyEvent.VK_KP_LEFT) 
		{
		  	curimg--;
		  
		  	if (curimg<0) curimg+=imglist.length;
				
		  	imageChange(curimg);
		} 
		else if (e.getKeyCode()==KeyEvent.VK_RIGHT ||
		    e.getKeyCode()==KeyEvent.VK_DOWN ||
		    e.getKeyCode()==KeyEvent.VK_PAGE_DOWN ||
		    e.getKeyCode()==KeyEvent.VK_KP_DOWN ||
		    e.getKeyCode()==KeyEvent.VK_KP_RIGHT) 
		{
		  	curimg++;
		  	
		  	if (curimg>=imglist.length) curimg-=imglist.length;
		  
		  	imageChange(curimg);
		}
	}
	
	public void keyReleased(KeyEvent e) { }
	public void keyTyped(KeyEvent e) { }

	public void componentResized(ComponentEvent e) 
	{            	

		double changeX = (double)getWidth() / lastX;
		double changeY = (double)getHeight() / lastY;


		if(!sizeFixed)
		{
		  	
		  	//maintain aspect ratio at min sizes
		  	if(getWidth()<= 120 || getHeight() <= 98)
		  	{
		  		
		  		changeX = 120 / lastX;
		        changeY = 98 / lastY;
		  		this.setSize(120,98);	  		
		  	}
		  	
		  	else if(   (changeX > changeY && getWidth() > lastX  ) 
		  	   || (changeX < changeY && getWidth() < lastX  )  )
		  	{
		  		changeY = changeX;
		  		this.setSize(getWidth(),(int)(lastY*changeX));
		  	}
		  	else
		  	{
		  		changeX = changeY;
		  		this.setSize((int)(lastX*changeY),getHeight());
		  	}
		  	sizeFixed = true;

		}
		else
		{
		  	lastX = (double)getWidth(); 
		  	lastY = (double)getHeight();
		  	
		  	resizedX = ((double)getWidth()) / (double)(imageData.image_width * 2);
		  	resizedY = ((double)getHeight()) / (double)(imageData.image_height * 2);
		  	  	
		  	trainCanvas.resizeAreas(resizedX,resizedY,changeX,changeY);
		  	
		  	sizeFixed = false;
		}  	
	}

	public void componentHidden(ComponentEvent e) { }
	public void componentMoved(ComponentEvent e)
	{ 
	
		prefs.putInt("ImageShowArea.location.x",getLocation().x);
		prefs.putInt("ImageShowArea.location.y",getLocation().y);

	}
	public void componentShown(ComponentEvent e) { }

	public void setTrainCanvas(TrainCanvas t)
	{
		trainCanvas = t;
	}

	public int getCurImage()
	{
		return curimg;
	}
  
}
