#include "Main.h"
#include "SoundPlay.h"
#include "Motion.h"
#include "Simulator.h"
#include "IPC/RegionRegistry.h"
#include "IPC/MessageReceiver.h"
#include "Motion/Kinematics.h"
#include "Wireless/Wireless.h"
#include "Shared/ProjectInterface.h"
#include "Behaviors/BehaviorBase.h"
#include "Vision/BufferedImageGenerator.h"
#include "Events/DataEvent.h"
#include "Events/EventRouter.h"
#include "Shared/Config.h"

#include "Events/EventBase.h"
#include "Events/LocomotionEvent.h"
#include "Events/TextMsgEvent.h"
#include "Events/VisionObjectEvent.h"

Main::Main()
	: Process(getID(),getClassName()),
	sounds(ipc_setup->registerRegion(SoundPlay::getSoundPlayID(),sizeof(sim::SoundPlayQueue_t))),
	motions(ipc_setup->registerRegion(Motion::getMotionCommandID(),sizeof(sim::MotionCommandQueue_t))),
	events(ipc_setup->registerRegion(getEventsID(),sizeof(sim::EventQueue_t))),
	cameraFrames(ipc_setup->registerRegion(Simulator::getCameraQueueID(),sizeof(sim::CameraQueue_t))),
	sensorFrames(ipc_setup->registerRegion(Simulator::getSensorQueueID(),sizeof(sim::SensorQueue_t))),
	motionmanager(ipc_setup->registerRegion(Motion::getMotionManagerID(),sizeof(MotionManager))),
	soundmanager(ipc_setup->registerRegion(SoundPlay::getSoundManagerID(),sizeof(SoundManager))),
	worldstate(ipc_setup->registerRegion(getWorldStateID(),sizeof(WorldState))),
	visrecv(NULL), sensrecv(NULL), evtrecv(NULL), visionRead(true), sensorRead(true),
	wireless_thread(), curimg(NULL)
{
	new (&(*events)) sim::EventQueue_t;
	new (&(*worldstate)) WorldState;
	state=&(*worldstate);
	motman=&(*motionmanager);
	sndman=&(*soundmanager);

	//Setup wireless
	wireless = new Wireless();
	sout=wireless->socket(SocketNS::SOCK_STREAM,Wireless::WIRELESS_DEF_RECV_SIZE,Wireless::WIRELESS_DEF_SEND_SIZE*12);
	serr=wireless->socket(SocketNS::SOCK_STREAM,Wireless::WIRELESS_DEF_RECV_SIZE,Wireless::WIRELESS_DEF_SEND_SIZE*4);
	wireless->setDaemon(sout);
	wireless->setDaemon(serr);
	serr->setFlushType(SocketNS::FLUSH_BLOCKING);
	sout->setTextForward();
	serr->setForward(sout);

	//Setup Kinematics
	kine=new Kinematics();
	
	//need to register any events which we might be sending or receiving
	EventTranslator::registerPrototype<EventBase>();
	EventTranslator::registerPrototype<LocomotionEvent>();
	EventTranslator::registerPrototype<TextMsgEvent>();
	EventTranslator::registerPrototype<VisionObjectEvent>();

	//EventRouter and Config are set up for all processes by main() before fork
}

Main::~Main() {
}


void Main::DoStart() {
try {

	Process::DoStart();
	//These are constructed by other processes, so need to wait
	//until the construction runlevel is complete
	sndman->InitAccess(*sounds);
	motman->InitAccess(*motions);

	wireless->listen(sout, config->main.console_port);
	wireless->listen(serr, config->main.stderr_port);
	wireless_thread.start();
	evtrecv=new MessageReceiver(*events,gotEvent,false);
	visrecv=new MessageReceiver(*cameraFrames,gotCamera,false);

} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during Main DoStart",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during Main DoStart",NULL))
		throw;
}
}

void Main::run() {
try {

	ProjectInterface::startupBehavior().DoStart();

	evtrecv->start();
	visrecv->start();
	
	Process::run();

	visrecv->finish();
	evtrecv->finish();

	ProjectInterface::startupBehavior().DoStop();

} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during Main 'run' runlevel (startupBehavior initialization and startup)",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during Main 'run' runlevel (startupBehavior initialization and startup)",NULL))
		throw;
}
}

void Main::DoStop() {
try {

	events->close();

	delete visrecv;
	visrecv=NULL;
	delete evtrecv;
	evtrecv=NULL;

	wireless_thread.stop();
	wireless->wakeup();
	wireless_thread.join();
	wireless->setDaemon(sout,false);
	wireless->close(sout);
	sout=NULL;
	wireless->setDaemon(serr,false);
	wireless->close(serr);
	serr=NULL;
	
	motman->RemoveAccess();
	
	if(curimg!=NULL)
		curimg->RemoveReference();

	Process::DoStop();

} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during Main DoStop",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during Main DoStop",NULL))
		throw;
}
}

bool Main::canManuallyAdvance() {
	Main * main=dynamic_cast<Main*>(Process::getCurrent());
	if(main==NULL) {
		cerr << "WARNING: attempt to call Main::advanceVision() from outside Main process!" << endl;
		return false;
	}
	if(globals->timeScale >= 0) //ignore call if we're in realtime mode
		return false;
	if(globals->advanceOnAccess) //ignore call if we're advancing on access
		return false;
	return true;
}

void Main::advanceVision() {
	Main * main=dynamic_cast<Main*>(Process::getCurrent());
	if(main==NULL) {
		cerr << "WARNING: attempt to call Main::advanceVision() from outside Main process!" << endl;
		return;
	}
	if(globals->timeScale >= 0) //ignore call if we're in realtime mode
		return;
	if(globals->advanceOnAccess) //ignore call if we're advancing on access
		return;
	if(main->visionRead) //already marked this frame read
		return;
	main->visionRead=true;
	main->visrecv->markRead();
}

void Main::advanceSensor() {
	Main * main=dynamic_cast<Main*>(Process::getCurrent());
	if(main==NULL) {
		cerr << "WARNING: attempt to call Main::advanceSensor() from outside Main process!" << endl;
		return;
	}
	if(globals->timeScale >= 0) //ignore call if we're in realtime mode
		return;
	if(globals->advanceOnAccess) //ignore call if we're advancing on access
		return;
	if(main->sensorRead) //already marked this frame read
		return;
	main->sensorRead=true;
	main->sensrecv->markRead();
}


bool Main::gotCamera(RCRegion* msg) {
try {
	if(msg==NULL)
		return true;
	msg->AddReference();
	Main * main=dynamic_cast<Main*>(Process::getCurrent());
	if(main->curimg!=NULL)
		main->curimg->RemoveReference();
	main->curimg=msg;
	BufferedImageGenerator::ImageSource img;
	memcpy(&img,msg->Base(),sizeof(unsigned int)*4);
	int l=reinterpret_cast<int*>(msg->Base())[4];
	img.layer = (l<0)?ProjectInterface::defRawCameraGenerator->getNumLayers()+l:l;
	bool isRealtime=(globals->timeScale >= 0);
	main->visionRead=globals->advanceOnAccess; //not actually read until the BufferedImageGenerator is accessed, but it's read as far as advanceVision() is concerned
	if(!isRealtime && main->visionRead)
		img.receiver = main->visrecv;
	img.img = reinterpret_cast<unsigned char*>(msg->Base()+sizeof(unsigned int)*5);
	erouter->postEvent(new DataEvent<BufferedImageGenerator::ImageSource>(img,EventBase::visOFbkEGID,0,EventBase::activateETID));
	erouter->postEvent(new DataEvent<BufferedImageGenerator::ImageSource>(img,EventBase::visOFbkEGID,0,EventBase::statusETID));
	erouter->postEvent(new DataEvent<BufferedImageGenerator::ImageSource>(img,EventBase::visOFbkEGID,0,EventBase::deactivateETID));
	return isRealtime;
} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during camera frame processing",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during camera frame processing",NULL))
		throw;
}
return true;
}

bool Main::gotEvent(RCRegion* msg) {
EventBase* evt=NULL;
try {
	evt=EventTranslator::decodeEvent(msg->Base(),msg->Size());
	if(evt==NULL) {
		cerr << "ERROR: Main::gotEvent() failed to decode message" << endl;
		return true;
	}
	erouter->postEvent(evt);
} catch(const std::exception& ex) {
	std::string emsg("Occurred during inter-process event processing");
	if(evt!=NULL)
		emsg+=": "+evt->getName();
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,emsg.c_str(),&ex))
		throw;
} catch(...) {
	std::string emsg("Occurred during inter-process event processing");
	if(evt!=NULL)
		emsg+=": "+evt->getName();
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,emsg.c_str(),NULL))
		throw;
}
return true;
}

/*! @file
 * @brief 
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.13 $
 * $State: Exp $
 * $Date: 2005/08/16 18:24:43 $
 */

