#include "MMCombo.h"
#include "Shared/WorldState.h"
#include "Shared/Profiler.h"
#include "Shared/debuget.h"
#include "Shared/Config.h"
#include "IPC/SharedObject.h"
#include "IPC/ProcessID.h"
#include "Events/EventRouter.h"
#include "Behaviors/BehaviorBase.h"
#include "Motion/MotionManager.h"
#include "Motion/Kinematics.h"
#include "Sound/SoundManager.h"
#include "Events/DataEvent.h"
#include "Events/TextMsgEvent.h"
#include "Events/FilterBankEvent.h"
#include "Shared/WMclass.h"

#include "Shared/ERS210Info.h"
#include "Shared/ERS220Info.h"
#include "Shared/ERS7Info.h"

#include "Shared/ProjectInterface.h"

#include "Events/EventBase.h"
#include "Events/LocomotionEvent.h"
#include "Events/TextMsgEvent.h"
#include "Events/VisionObjectEvent.h"

#include <OPENR/OSyslog.h>
#include <OPENR/core_macro.h>
#include <OPENR/OFbkImage.h>
#include "aperios/MMCombo/entry.h"

using namespace std;

MMCombo::MMCombo()
	: OObject(), motmanMemRgn(NULL), worldStateMemRgn(NULL),
		soundManagerMemRgn(NULL),
		runLevel(0), num_open(0), etrans(NULL), isStopped(true)
{
try {
	for(unsigned int i=0; i<NumOutputs; i++) {
		primIDs[i]=oprimitiveID_UNDEF;
		open[i]=false;
	}

	//need to register any events which we might be sending or receiving
	EventTranslator::registerPrototype<EventBase>();
	EventTranslator::registerPrototype<LocomotionEvent>();
	EventTranslator::registerPrototype<TextMsgEvent>();
	EventTranslator::registerPrototype<VisionObjectEvent>();
} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during MMCombo construction",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during MMCombo construction",NULL))
		throw;
}
}


OStatus
MMCombo::DoInit(const OSystemEvent&)
{
try {
	cout << objectName << "::DoInit() " << endl;

	isStopped=false;

	NEW_ALL_SUBJECT_AND_OBSERVER;
	REGISTER_ALL_ENTRY;
	SET_ALL_READY_AND_NOTIFY_ENTRY;
		
	// make sure the library doesn't drop data "for" us on this reliable communication channel
	observer[obsReceiveWorldState]->SetBufCtrlParam(0,1,1);
	observer[obsReceiveMotionManager]->SetBufCtrlParam(0,1,1);
	observer[obsReceiveSoundManager]->SetBufCtrlParam(0,1,1);
	observer[obsMotionManagerComm]->SetBufCtrlParam(0,1,MotionManager::MAX_MOTIONS+1);
	//+1 to MAX_MOTIONS so we can get a delete message after we've filled up

	cout << objectName << ": sbjRegisterWorldState==" << sbjRegisterWorldState << " selector==" << subject[sbjRegisterWorldState]->GetID().GetSelector() << '\n'
			 << objectName << ": obsReceiveWorldState==" << obsReceiveWorldState << " selector==" << observer[obsReceiveWorldState]->GetID().GetSelector() << '\n'
			 << objectName << ": sbjRegisterMotionManager==" << sbjRegisterMotionManager << " selector==" << subject[sbjRegisterMotionManager]->GetID().GetSelector() << '\n'
			 << objectName << ": obsReceiveMotionManager==" << obsReceiveMotionManager << " selector==" << observer[obsReceiveMotionManager]->GetID().GetSelector() << '\n'
			 << objectName << ": obsEventTranslatorComm==" << obsEventTranslatorComm << " selector==" << observer[obsEventTranslatorComm]->GetID().GetSelector() << '\n'
			 << objectName << ": sbjEventTranslatorComm==" << sbjEventTranslatorComm << " selector==" << observer[sbjEventTranslatorComm]->GetID().GetSelector() << '\n'
			 << objectName << ": sbjMoveJoint==" << sbjMoveJoint << " selector==" << subject[sbjMoveJoint]->GetID().GetSelector() << '\n'
			 << objectName << ": obsSensorFrame==" << obsSensorFrame << " selector==" << observer[obsSensorFrame]->GetID().GetSelector() << '\n'
			 << objectName << ": obsImage==" << obsImage << " selector==" << observer[obsImage]->GetID().GetSelector() << '\n'
			 << objectName << ": obsMic==" << obsMic << " selector==" << observer[obsMic]->GetID().GetSelector() << '\n'
			 << objectName << ": sbjMotionManagerComm==" << sbjMotionManagerComm << " selector==" << subject[sbjMotionManagerComm]->GetID().GetSelector() << '\n'
			 << objectName << ": obsMotionManagerComm==" << obsMotionManagerComm << " selector==" << observer[obsMotionManagerComm]->GetID().GetSelector() << '\n'
			 << objectName << ": obsReceiveSoundManager==" << obsReceiveSoundManager << " selector==" << observer[obsReceiveSoundManager]->GetID().GetSelector() << '\n'
			 << objectName << ": sbjSoundManagerComm==" << sbjSoundManagerComm << " selector==" << subject[sbjSoundManagerComm]->GetID().GetSelector() << '\n'
			 << flush;

	if(strcmp(objectName,"MainObj")==0)
		ProcessID::setID(ProcessID::MainProcess);
	else if(strcmp(objectName,"MotoObj")==0)
		ProcessID::setID(ProcessID::MotionProcess);
		
	//Read config file
	config=new Config("/ms/config/tekkotsu.cfg");

	erouter = new EventRouter;

	if(strcmp(objectName,"MainObj")==0) {
		bool isSlowOutput[NumOutputs];
		for(unsigned int i=0; i<NumOutputs; i++)
			isSlowOutput[i]=!IsFastOutput[i];

		SetupOutputs(isSlowOutput);

		//Request power status updates
		OPowerStatus observationStatus;
		observationStatus.Set(orsbALL,obsbALL,opsoREMAINING_CAPACITY_NOTIFY_EVERY_CHANGE,opsoTEMPERATURE_NOTIFY_EVERY_CHANGE,opsoTIME_DIF_NOTIFY_EVERY_CHANGE,opsoVOLUME_NOTIFY_EVERY_CHANGE);
		OServiceEntry entry(myOID_, Extra_Entry[entryGotPowerEvent]);
		OStatus result = OPENR::ObservePowerStatus(observationStatus, entry);
		if(result != oSUCCESS) {
			OSYSLOG1((osyslogERROR, "%s : %s %d","MMCombo::DoStart()","OPENR::ObservePowerStatus() FAILED", result));
			return oFAIL;
		}
		
    //Setup wireless
    wireless = new Wireless();
    sout=wireless->socket(SocketNS::SOCK_STREAM,Wireless::WIRELESS_DEF_RECV_SIZE,Wireless::WIRELESS_DEF_SEND_SIZE*12);
    serr=wireless->socket(SocketNS::SOCK_STREAM,Wireless::WIRELESS_DEF_RECV_SIZE,Wireless::WIRELESS_DEF_SEND_SIZE*4);
    wireless->setDaemon(sout);
    wireless->setDaemon(serr);
		serr->setFlushType(SocketNS::FLUSH_BLOCKING);
    sout->setTextForward();
    serr->setForward(sout);

		//worldStateMemRgn -> state setup
		worldStateMemRgn = InitRegion(sizeof(WorldState));
		state=new ((WorldState*)worldStateMemRgn->Base()) WorldState;

		etrans=new NoOpEventTranslator(*erouter);
		MotionManager::setTranslator(etrans);
	}
	if(strcmp(objectName,"MotoObj")==0) {
		SetupOutputs(IsFastOutput);
    OPENR::SetMotorPower(opowerON);
		OPENR::EnableJointGain(oprimitiveID_UNDEF); //oprimitiveID_UNDEF means enable all

    //Setup wireless
    wireless = new Wireless();
    sout=wireless->socket(SocketNS::SOCK_STREAM,Wireless::WIRELESS_DEF_RECV_SIZE,Wireless::WIRELESS_DEF_SEND_SIZE*6);
    serr=wireless->socket(SocketNS::SOCK_STREAM,Wireless::WIRELESS_DEF_RECV_SIZE,Wireless::WIRELESS_DEF_SEND_SIZE*2);
    wireless->setDaemon(sout);
    wireless->setDaemon(serr);
		serr->setFlushType(SocketNS::FLUSH_BLOCKING);
    sout->setTextForward();
    serr->setForward(sout);
		
		//motmanMemRgn -> motman setup
		motmanMemRgn = InitRegion(sizeof(MotionManager));
		motman = new (motmanMemRgn->Base()) MotionManager();
		motman->InitAccess(subject[sbjMotionManagerComm]);

		etrans=new IPCEventTranslator(*subject[sbjEventTranslatorComm]);
		MotionManager::setTranslator(etrans);
		//MotionCommands enqueue directly, so there shouldn't be any riff-raff to catch
		//but just in case, subscribe to everything except erouterEGID
		for(unsigned int i=0; i<EventBase::numEGIDs; i++)
			if(i!=EventBase::erouterEGID)
				erouter->addTrapper(etrans,static_cast<EventBase::EventGeneratorID_t>(i));
	}
	kine = new Kinematics();

	cout << objectName << "::DoInit()-DONE" << endl;
	return oSUCCESS;

} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during MMCombo::DoInit()",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during MMCombo::DoInit()",NULL))
		throw;
}
return oSUCCESS;
}

OStatus
MMCombo::DoStart(const OSystemEvent&)
{
try {
	cout << objectName << "::DoStart() " << endl;

	// initialize the current power status, doesn't always give us
	// a power update right away otherwise
	if(strcmp(objectName,"MainObj")==0) {
		wireless->listen(sout, config->main.console_port);
		wireless->listen(serr, config->main.stderr_port);
		OPowerStatus power;
		OPENR::GetPowerStatus(&power);
		state->read(power,erouter);
	}

	if(strcmp(objectName,"MotoObj")==0) {
		wireless->listen(sout, config->motion.console_port);
		wireless->listen(serr, config->motion.stderr_port);
	}
	
	isStopped=false;

	ENABLE_ALL_SUBJECT;
	ASSERT_READY_TO_ALL_OBSERVER;

	addRunLevel();
	
	cout << objectName << "::DoStart()-DONE" << endl;
	return oSUCCESS;

} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during MMCombo::DoStart()",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during MMCombo::DoStart()",NULL))
		throw;
}
return oSUCCESS;
}

OStatus
MMCombo::DoStop(const OSystemEvent&)
{
try {
	cout << objectName << "::DoStop()..." << endl;
	if(strcmp(objectName,"MainObj")==0) {
		ProjectInterface::startupBehavior().DoStop();
		wireless->close(sout);
		wireless->close(serr);
	}
	DISABLE_ALL_SUBJECT;
	DEASSERT_READY_TO_ALL_OBSERVER;
	isStopped=true;
	cout << objectName << "::DoStop()-DONE" << endl;
	return oSUCCESS;

} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during MMCombo::DoStop()",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during MMCombo::DoStop()",NULL))
		throw;
}
return oSUCCESS;
}

OStatus
MMCombo::DoDestroy(const OSystemEvent&)
{
try {
	cout << objectName << "::DoDestroy()..." << endl;
	delete etrans;
	etrans=NULL;
	MotionManager::setTranslator(NULL);
	if(strcmp(objectName,"MainObj")==0) {
		delete erouter;
		motmanMemRgn->RemoveReference();
	}
	if(strcmp(objectName,"MotoObj")==0) {
		worldStateMemRgn->RemoveReference();
	}
	soundManagerMemRgn->RemoveReference();
	DELETE_ALL_SUBJECT_AND_OBSERVER;
	cout << objectName << "::DoDestroy()-DONE" << endl;
	return oSUCCESS;

} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during MMCombo::DoDestroy()",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during MMCombo::DoDestroy()",NULL))
		throw;
}
return oSUCCESS;
}

/*! Called when MotoObj is initially ready as well as when it has finished
 *  processing the previous message - we only want to do this the first time
 *  otherwise we infinite loop. */
void
MMCombo::ReadyRegisterWorldState(const OReadyEvent&){
	static bool is_init=true;
	if(is_init) {
		is_init=false;
		cout << objectName << " Registering WorldState" << endl;
		if(strcmp(objectName,"MainObj")==0) {
			subject[sbjRegisterWorldState]->SetData(worldStateMemRgn);
			subject[sbjRegisterWorldState]->NotifyObservers();
		}
	}
}

void
MMCombo::GotWorldState(const ONotifyEvent& event){
	cout << objectName << "-GOTWORLDSTATE..." << flush;
	//	PROFSECTION("GotMemRegion()",state->mainProfile);
	if(strcmp(objectName,"MotoObj")==0) {
		ASSERT(event.NumOfData()==1,"Too many WorldStates");
		worldStateMemRgn = event.RCData(0);
		worldStateMemRgn->AddReference();
		state = reinterpret_cast<WorldState*>(worldStateMemRgn->Base());
	}
  observer[obsReceiveWorldState]->AssertReady();
	cout << "done" << endl;
}

		
/*! Called when MainObj is initially ready as well as when it has finished
 *  processing the previous message - we only want to do this the first time
 *  otherwise we infinite loop. */
void
MMCombo::ReadyRegisterMotionManager(const OReadyEvent&){
	static bool is_init=true;
	if(is_init) {
		is_init=false;
		cout << objectName << " Registering MotionManager" << endl;
		if(strcmp(objectName,"MotoObj")==0) {
			subject[sbjRegisterMotionManager]->SetData(motmanMemRgn);
			subject[sbjRegisterMotionManager]->NotifyObservers();
		}
	}
}

void
MMCombo::GotMotionManager(const ONotifyEvent& event){
	cout << objectName << "-GOTMOTIONMANAGER..." << flush;
	//	PROFSECTION("GotMemRegion()",state->mainProfile);
	if(strcmp(objectName,"MainObj")==0) {
		ASSERT(event.NumOfData()==1,"Too many MotionManagers");
		motmanMemRgn = event.RCData(0);
		motmanMemRgn->AddReference();
		motman = reinterpret_cast<MotionManager*>(motmanMemRgn->Base());
		cout << "MAIN INIT MOTMAN..." << flush;
		//			hexout(event.RCData(event_data_id)->Base(),128);
		motman->InitAccess(subject[sbjMotionManagerComm]);
		addRunLevel();
	}
  observer[obsReceiveMotionManager]->AssertReady();
	cout << "done" << endl;
}


void
MMCombo::GotInterProcessEvent(const ONotifyEvent& event){
	EventBase* evt=NULL;
try {
	//cout << objectName << "-GOTInterProcessEvent " << event.NumOfData() << "..." << flush;
	//cout << TimeET() << endl;
	//	PROFSECTION("GotMemRegion()",state->mainProfile);
	if(etrans==NULL)
		return;
	for(int i=0; i<event.NumOfData(); i++) {
		RCRegion * msg = event.RCData(i);
		evt=etrans->decodeEvent(msg->Base(),msg->Size());
		if(evt!=NULL)
			erouter->postEvent(evt);
	}
  observer[obsEventTranslatorComm]->AssertReady();
	//cout << "done" << endl;

} catch(const std::exception& ex) {
  observer[obsEventTranslatorComm]->AssertReady();
	std::string msg("Occurred during inter-process event processing");
	if(evt!=NULL)
		msg+=": "+evt->getName();
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,msg.c_str(),&ex))
		throw;
} catch(...) {
  observer[obsEventTranslatorComm]->AssertReady();
	std::string msg("Occurred during inter-process event processing");
	if(evt!=NULL)
		msg+=": "+evt->getName();
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,msg.c_str(),NULL))
		throw;
}
}

		
void
MMCombo::ReadySendJoints(const OReadyEvent& sysevent) {
try {

	if(isStopped) {
		//cout << "BAH!ReadySendJoints" << endl;
		return;
	}

	static unsigned int id=-1U;
	Profiler::Timer timer;
	if(ProcessID::getID()==ProcessID::MotionProcess) {
		if(state) {
			if(id==-1U)
				id=state->motionProfile.getNewID("ReadySendJoints()");
			timer.setID(id,&state->motionProfile.prof);
		}
	}	else if(ProcessID::getID()==ProcessID::MainProcess) {
		if(id==-1U)
			id=state->mainProfile.getNewID("ReadySendJoints()");
		timer.setID(id,&state->mainProfile.prof);
	}

	if(num_open==0) //If we don't have any joints to open, leave now. (i.e. MainObj on a 220, has no ears)
		return;

	// Find an unused command vector
	RCRegion* rgn=NULL;
	for (unsigned int i = 0; i < NUM_COMMAND_VECTOR; i++) {
		if (region[i]->NumberOfReference() == 1) {
			rgn=region[i];
			/*			if(strcmp(objectName,"MainObj")==0) {
							static unsigned int lasttime=get_time();
							unsigned int thistime=get_time();
							cout << '*' << i << ' ' << thistime << '\t' << (thistime-lasttime) << endl;
							lasttime=thistime;
							}*/
			break;
		}
	}
	ASSERTRET(rgn!=NULL,"Could not find unused command vector");
	ASSERTRET(rgn->Base()!=NULL,"Bad Command Vector");
	OCommandVectorData* cmdVecData = reinterpret_cast<OCommandVectorData*>(rgn->Base());
	
	// Update the outputs (note that Main is doing the ears)
	//I'm using an id compare instead of the slightly more readable strcmp for a tiny bit of speed
	bool isERS7;
	if(state!=NULL)
		isERS7=state->robotDesign&WorldState::ERS7Mask;
	else {
		char robotDesignStr[orobotdesignNAME_MAX + 1];
		memset(robotDesignStr, 0, sizeof(robotDesignStr));
		if (OPENR::GetRobotDesign(robotDesignStr) != oSUCCESS) {
			cout << objectName << "::SetupOutputs - OPENR::GetRobotDesign() failed." << endl;
			return;
		}
		isERS7=(strcmp(robotDesignStr,"ERS-7")==0);
	}		
	if(ProcessID::getID()==ProcessID::MotionProcess) {
		float outputs[NumFrames][NumOutputs];
		if(state!=NULL) {
			motman->getOutputs(outputs);
			motman->updatePIDs(primIDs);
			motman->updateWorldState();
		} else {
			for(unsigned int f=0; f<NumFrames; f++)
				for(unsigned int i=0; i<NumOutputs; i++)
					outputs[f][i]=0;
		}
			
		// Should be a relatively simple matter to copy angles into commands...
		unsigned int used=0; //but only copy open joints (so main does ears on 210, motion does everything else)
		for(unsigned int i=PIDJointOffset; i<PIDJointOffset+NumPIDJoints; i++)
			if(open[i]) {
				float cal=config->motion.calibration[i-PIDJointOffset];
				OJointCommandValue2* jval = reinterpret_cast<OJointCommandValue2*>(cmdVecData->GetData(used)->value);
				for(unsigned int frame=0; frame<NumFrames; frame++)
					jval[frame].value = (slongword)(outputs[frame][i]/cal*1.0e6f);
				used++;
			}
		if(isERS7) {
			// except if it's an ERS-7, we have to use different data structures for some of the leds and the ears
			for(unsigned int i=LEDOffset; i<ERS7Info::FaceLEDPanelOffset; i++)
				if(open[i]) {
					OLEDCommandValue2* jval = reinterpret_cast<OLEDCommandValue2*>(cmdVecData->GetData(used)->value);
					for(unsigned int frame=0; frame<NumFrames; frame++)
						jval[frame].led = calcLEDValue(i-LEDOffset,outputs[frame][i]);
					used++;
				}
			// for instance, this virtual mode thing, which is global to all the affected LEDs
			OLED3Mode curMode[NumFrames];
			for(unsigned int frame=0; frame<NumFrames; frame++)
				curMode[frame]=(calcLEDValue(ERS7Info::LEDABModeOffset-LEDOffset,sqrt(clipRange01(outputs[frame][ERS7Info::LEDABModeOffset])))==oledON?oled3_MODE_B:oled3_MODE_A);
			for(unsigned int i=ERS7Info::FaceLEDPanelOffset; i<LEDOffset+NumLEDs; i++)
				if(open[i]) {
					OLEDCommandValue3* jval = reinterpret_cast<OLEDCommandValue3*>(cmdVecData->GetData(used)->value);
					for(unsigned int frame=0; frame<NumFrames; frame++) {
						jval[frame].intensity = static_cast<sword>(255*clipRange01(outputs[frame][i]));
						jval[frame].mode=curMode[frame];
					}
					used++;
				}
			for(unsigned int i=BinJointOffset; i<BinJointOffset+NumBinJoints; i++)
				if(open[i]) {
					OJointCommandValue4* jval = reinterpret_cast<OJointCommandValue4*>(cmdVecData->GetData(used)->value);
					for(unsigned int frame=0; frame<NumSlowFrames; frame++)
						jval[frame].value = (outputs[frame][i]<.5?ojoint4_STATE0:ojoint4_STATE1);
					used++;
				}
		} else {
			for(unsigned int i=LEDOffset; i<LEDOffset+NumLEDs; i++)
				if(open[i]) {
					OLEDCommandValue2* jval = reinterpret_cast<OLEDCommandValue2*>(cmdVecData->GetData(used)->value);
					for(unsigned int frame=0; frame<NumFrames; frame++)
						jval[frame].led = calcLEDValue(i-LEDOffset,outputs[frame][i]);
					used++;
				}
			for(unsigned int i=BinJointOffset; i<BinJointOffset+NumBinJoints; i++)
				if(open[i]) {
					OJointCommandValue3* jval = reinterpret_cast<OJointCommandValue3*>(cmdVecData->GetData(used)->value);
					for(unsigned int frame=0; frame<NumSlowFrames; frame++)
						jval[frame].value = (outputs[frame][i]<.5?ojoint3_STATE1:ojoint3_STATE0);
					used++;
				}
		}
	}	else if(ProcessID::getID()==ProcessID::MainProcess) {
		// Just copy over the current ear state from WorldState
		unsigned int used=0; //but only copy open joints (so main does ears, motion does everything else)
		if(isERS7) {
			for(unsigned int i=BinJointOffset; i<BinJointOffset+NumBinJoints; i++)
				if(open[i]) {
					OJointCommandValue4* jval = reinterpret_cast<OJointCommandValue4*>(cmdVecData->GetData(used)->value);
					for(unsigned int frame=0; frame<NumSlowFrames; frame++)
						jval[frame].value = (state->outputs[i]<.5?ojoint4_STATE0:ojoint4_STATE1);
					used++;
				}
		} else {
			for(unsigned int i=BinJointOffset; i<BinJointOffset+NumBinJoints; i++)
				if(open[i]) {
					OJointCommandValue3* jval = reinterpret_cast<OJointCommandValue3*>(cmdVecData->GetData(used)->value);
					for(unsigned int frame=0; frame<NumSlowFrames; frame++)
						jval[frame].value = (state->outputs[i]<.5?ojoint3_STATE1:ojoint3_STATE0);
					used++;
				}
		}
	}

	// Send outputs to system
	subject[sbjMoveJoint]->SetData(rgn);

	// The first time this is called, we actually need to send *two* buffers
	// in order to get the double buffering going... (well, actually generalized
	// for NUM_COMMAND_VECTOR level buffering)
	static unsigned int initCount=1;
	if(initCount<NUM_COMMAND_VECTOR) {
		initCount++;
		ReadySendJoints(sysevent);
	} else //recursive base case
		subject[sbjMoveJoint]->NotifyObservers();

} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during joint angle updates",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during joint angle updates",NULL))
		throw;
}
}

void
MMCombo::GotSensorFrame(const ONotifyEvent& event){
try {
	erouter->processTimers();
} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during timer processing",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during timer processing",NULL))
		throw;
}
try {
	//	if(state && state->buttons[RFrPawOffset])
	//	cout << "SENSOR..."<<flush;
	if(isStopped) {
		//cout << "BAH!GotSensorFrame" << endl;
		return;
	}

	PROFSECTION("GotSensorFrame()",state->mainProfile);

  OSensorFrameVectorData* rawsensor = reinterpret_cast<OSensorFrameVectorData*>(event.RCData(0)->Base());
	state->read(rawsensor[0],erouter);
	static unsigned int throwaway=1; //i thought the first few sensor updates might be flakey, but now i think not.  But a good way to delay startup.
	if(throwaway!=0) {
		throwaway--;
		if(throwaway==0) {
			//seed the random number generator with time value and sensor noise
			if(config->main.seed_rng) {
				double t=TimeET().Value(); //current time with nanosecond resolution
				unsigned int * tm=reinterpret_cast<unsigned int*>(&t);
				unsigned int seed=tm[0]+tm[1];
				for(unsigned int i=0; i<NumPIDJoints; i++) { //joint positions
					unsigned int * x=reinterpret_cast<unsigned int*>(&state->outputs[i]);
					seed+=(*x)<<((i%sizeof(unsigned int))*8);
				}
				for(unsigned int i=0; i<NumPIDJoints; i++) { //joint forces
					unsigned int * x=reinterpret_cast<unsigned int*>(&state->pidduties[i]);
					seed+=(*x)<<((i%sizeof(unsigned int))*8);
				}
				for(unsigned int i=0; i<NumSensors; i++) {
					unsigned int * x=reinterpret_cast<unsigned int*>(&state->sensors[i]);
					seed+=(*x)<<((i%sizeof(unsigned int))*8); //sensor values
				}
				cout << "RNG seed=" << seed << endl;;
				srand(seed);
			}
			addRunLevel();
		}
	}
  observer[obsSensorFrame]->AssertReady();
	//	if(state && state->buttons[RFrPawOffset])
	//	cout << "done" << endl;

} catch(const std::exception& ex) {
  observer[obsSensorFrame]->AssertReady();
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during sensor update processing",&ex))
		throw;
} catch(...) {
  observer[obsSensorFrame]->AssertReady();
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during sensor update processing",NULL))
		throw;
}
}

void
MMCombo::GotImage(const ONotifyEvent& event){
	if(isStopped) {
		//cout << "BAH!GotImage" << endl;
		return;
	}

try {
	erouter->processTimers();
} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during timer processing",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during timer processing",NULL))
		throw;
}
try {
	PROFSECTION("GotImage()",state->mainProfile);
  
	WMvari(int, frame_counter, 0);
	++frame_counter;
	
	erouter->postEvent(new DataEvent<const OFbkImageVectorData*>(reinterpret_cast<const OFbkImageVectorData*>(event.Data(0)),EventBase::visOFbkEGID,0,EventBase::activateETID));
	erouter->postEvent(new DataEvent<const OFbkImageVectorData*>(reinterpret_cast<const OFbkImageVectorData*>(event.Data(0)),EventBase::visOFbkEGID,0,EventBase::statusETID));
	erouter->postEvent(new DataEvent<const OFbkImageVectorData*>(reinterpret_cast<const OFbkImageVectorData*>(event.Data(0)),EventBase::visOFbkEGID,0,EventBase::deactivateETID));
	
  observer[obsImage]->AssertReady();

} catch(const std::exception& ex) {
  observer[obsImage]->AssertReady();
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during camera image processing",&ex))
		throw;
} catch(...) {
  observer[obsImage]->AssertReady();
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during camera image processing",NULL))
		throw;
}
try {
	erouter->processTimers();
} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during timer processing",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during timer processing",NULL))
		throw;
}
}

void
MMCombo::GotAudio(const ONotifyEvent& event){
try {
	if(isStopped) {
		//cout << "BAH!GotAudio" << endl;
		return;
	}

	PROFSECTION("GotAudio()",state->mainProfile);

	for (int i = 0; i < event.NumOfData(); i++) {
		erouter->postEvent(new DataEvent<const OSoundVectorData*>(reinterpret_cast<const OSoundVectorData*>(event.Data(i)),EventBase::micOSndEGID,0,EventBase::statusETID));
		try {
			erouter->processTimers();
		} catch(const std::exception& ex) {
			if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during timer processing",&ex))
				throw;
		} catch(...) {
			if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during timer processing",NULL))
				throw;
		}
	}
  
  observer[obsMic]->AssertReady();

} catch(const std::exception& ex) {
  observer[obsMic]->AssertReady();
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during audio processing",&ex))
		throw;
} catch(...) {
  observer[obsMic]->AssertReady();
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during audio processing",NULL))
		throw;
}
}

void
MMCombo::GotPowerEvent(void * msg){
	if(isStopped) {
		//cout << "BAH!GotPowerEvent" << endl;
		return;
	}
try {
	erouter->processTimers();
} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during timer processing",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during timer processing",NULL))
		throw;
}
try {

	//	cout << "POWER..."<<flush;
	PROFSECTION("PowerEvent()",state->mainProfile);

	static bool first=true;
	if(first) {
		addRunLevel();
		first=false;
	}
	const OPowerStatus* result = &static_cast<OPowerStatusMessage*>(msg)->powerStatus;
	state->read(*result,erouter);
	// this part watches to see if the power button is pressed to shutdown the robot
	// i'm leaving this low-level because there's not much else you can do anyway...
	// the hardware kills power to the motors, and as far as we can tell, you can't
	// turn them back on.
	if(state->powerFlags[PowerSourceID::PauseSID]) {
		cout << "%%%%%%%  Pause button was pushed! %%%%%%%" << endl;
		OBootCondition bc(0);
		OPENR::Shutdown(bc);
	}
	//	cout << "done" << endl;

} catch(const std::exception& ex) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during power status update",&ex))
		throw;
} catch(...) {
	if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during power status update",NULL))
		throw;
}
}

void
MMCombo::GotMotionMsg(const ONotifyEvent& event){
	if(isStopped) {
		//cout << "BAH!GotMotionMsg" << endl;
		return;
	}

	//	cout << "RECEIVE..."<<flush;
	if(motman!=NULL)
		motman->receivedMsg(event);
	else
		cout << "*** WARNING Main dropping MotionCommand (motman not ready) " << endl;
	observer[obsMotionManagerComm]->AssertReady();
	//	cout << "done" << endl;
}

void
MMCombo::GotSoundManager(const ONotifyEvent& event) {
	cout << objectName << "-GOTSOUNDMANAGER..." << flush;
	//	PROFSECTION("GotMemRegion()",state->mainProfile);
	ASSERT(event.NumOfData()==1,"Too many SoundManagers");
	soundManagerMemRgn = event.RCData(0);
	soundManagerMemRgn->AddReference();
	sndman = reinterpret_cast<SoundManager*>(soundManagerMemRgn->Base());
  observer[obsReceiveSoundManager]->AssertReady();
	sndman->InitAccess(subject[sbjSoundManagerComm]);
	addRunLevel();
	cout << "done" << endl;
}

void
MMCombo::OpenPrimitives()
{
	for(unsigned int i=0; i<NumOutputs; i++)
		if(open[i]) {
			OStatus result = OPENR::OpenPrimitive(PrimitiveName[i], &primIDs[i]);
			if (result != oSUCCESS)
				OSYSLOG1((osyslogERROR, "%s : %s %d","MMCombo::DoInit()","OPENR::OpenPrimitive() FAILED", result));
		}
}

void
MMCombo::SetupOutputs(const bool to_open[NumOutputs])
{
	char robotDesignStr[orobotdesignNAME_MAX + 1];
	memset(robotDesignStr, 0, sizeof(robotDesignStr));
	if (OPENR::GetRobotDesign(robotDesignStr) != oSUCCESS) {
		cout << objectName << "::SetupOutputs - OPENR::GetRobotDesign() failed." << endl;
		return;
	} else {
		if(strcmp(robotDesignStr,"ERS-210")==0) {
			for(unsigned int j=0; j<NumOutputs; j++)
				open[j]=to_open[j] && ERS210Info::IsRealERS210[j];
		} else if(strcmp(robotDesignStr,"ERS-220")==0) {
			for(unsigned int j=0; j<NumOutputs; j++)
				open[j]=to_open[j] && ERS220Info::IsRealERS220[j];
		} else if(strcmp(robotDesignStr,"ERS-7")==0) {
			for(unsigned int j=0; j<NumOutputs; j++)
				open[j]=to_open[j] && ERS7Info::IsRealERS7[j];
		} else {
			cout << "MMCombo::SetupOutputs - ERROR: Unrecognized model: "<<robotDesignStr<<"\nSorry..."<<endl;
			return;
		}
	}
	
	// count how many we're opening
	for(unsigned int j=0; j<NumOutputs; j++)
		if(open[j])
			num_open++;

	if(num_open==0) //If we don't have any joints to open, leave now. (i.e. MainObj on a 220, has no ears, and on ERS-7, all joints are full speed)
		return;

	OpenPrimitives();

	// request memory regions
	for (unsigned int i = 0; i < NUM_COMMAND_VECTOR; i++) {
		MemoryRegionID      cmdVecDataID;
		OCommandVectorData* cmdVecData;
		OStatus result = OPENR::NewCommandVectorData(num_open,&cmdVecDataID,&cmdVecData);
		if (result != oSUCCESS)
			OSYSLOG1((osyslogERROR, "%s : %s %d","MMCombo::NewCommandVectorData()","OPENR::NewCommandVectorData() FAILED", result));
		region[i] = new RCRegion(cmdVecData->vectorInfo.memRegionID,cmdVecData->vectorInfo.offset,(void*)cmdVecData,cmdVecData->vectorInfo.totalSize);
		cmdVecData->SetNumData(num_open);

		// initialize the outputs we just opened
		unsigned int used=0;
		ASSERT(cmdVecData==reinterpret_cast<OCommandVectorData*>(region[i]->Base())," should be equal!?");
		for(unsigned int j=PIDJointOffset; j<PIDJointOffset+NumPIDJoints; j++)
			if(open[j]) {
				OCommandInfo* info = cmdVecData->GetInfo(used++);
				info->Set(odataJOINT_COMMAND2, primIDs[j], NumFrames);
			}
		if(strcmp(robotDesignStr,"ERS-7")==0) {
			// this part's the same as usual, except stop when we get to face leds
			for(unsigned int j=LEDOffset; j<ERS7Info::FaceLEDPanelOffset; j++)
				if(open[j]) {
					OCommandInfo* info = cmdVecData->GetInfo(used);
					info->Set(odataLED_COMMAND2, primIDs[j], NumFrames);
					OLEDCommandValue2* jval = reinterpret_cast<OLEDCommandValue2*>(cmdVecData->GetData(used)->value);
					for(unsigned int frame=0; frame<NumFrames; frame++)
						jval[frame].period = 1;
					used++;
				}
			//we have to use OLEDCommandValue3 on the face and back LEDs if it's an ERS-7
			for(unsigned int j=ERS7Info::FaceLEDPanelOffset; j<LEDOffset+NumLEDs; j++)
				if(open[j]) {
					OCommandInfo* info = cmdVecData->GetInfo(used);
					info->Set(odataLED_COMMAND3, primIDs[j], NumFrames);
					OLEDCommandValue3* jval = reinterpret_cast<OLEDCommandValue3*>(cmdVecData->GetData(used)->value);
					for(unsigned int frame=0; frame<NumFrames; frame++)
						jval[frame].period = 1;
					used++;
				}
			//also have to use OJointCommandValue4 on the ears now
			for(unsigned int j=BinJointOffset; j<BinJointOffset+NumBinJoints; j++)
				if(open[j]) {
					OCommandInfo* info = cmdVecData->GetInfo(used);
					info->Set(odataJOINT_COMMAND4, primIDs[j], NumSlowFrames);
					OJointCommandValue4* jval = reinterpret_cast<OJointCommandValue4*>(cmdVecData->GetData(used)->value);
					for(unsigned int frame=0; frame<NumFrames; frame++)
						jval[frame].period = 1;
					used++;
				}
		} else {
			for(unsigned int j=LEDOffset; j<LEDOffset+NumLEDs; j++)
				if(open[j]) {
					OCommandInfo* info = cmdVecData->GetInfo(used);
					info->Set(odataLED_COMMAND2, primIDs[j], NumFrames);
					OLEDCommandValue2* jval = reinterpret_cast<OLEDCommandValue2*>(cmdVecData->GetData(used)->value);
					for(unsigned int frame=0; frame<NumFrames; frame++)
						jval[frame].period = 1;
					used++;
				}
			for(unsigned int j=BinJointOffset; j<BinJointOffset+NumBinJoints; j++)
				if(open[j]) {
					OCommandInfo* info = cmdVecData->GetInfo(used);
					info->Set(odataJOINT_COMMAND3, primIDs[j], NumSlowFrames);
					used++;
				}
		}
	}
}

/*! Will round up size to the nearest page */
RCRegion*
MMCombo::InitRegion(unsigned int size) {
	unsigned int pagesize=4096;
	sError err=GetPageSize(&pagesize);
	ASSERT(err==sSUCCESS,"Error "<<err<<" getting page size");
	unsigned int pages=(size+pagesize-1)/pagesize;
	return new RCRegion(pages*pagesize);
}

void
MMCombo::addRunLevel() {
	runLevel++;
	if(runLevel==readyLevel) {
		try {
			cout << "START UP BEHAVIOR..." << flush;
			ProjectInterface::startupBehavior().DoStart();
			cout << "START UP BEHAVIOR-DONE" << endl;
		} catch(const std::exception& ex) {
			if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during StartupBehavior construction and startup",&ex))
				throw;
		} catch(...) {
			if(!ProjectInterface::uncaughtException(__FILE__,__LINE__,"Occurred during StartupBehavior construction and startup",NULL))
				throw;
		}
	}
}


/*! @file
 * @brief Implements MMCombo, the OObject which "forks" (sort of) into Main and Motion processes
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.10 $
 * $State: Exp $
 * $Date: 2005/08/16 18:27:58 $
 */


