#include "BufferedImageGenerator.h"
#include "Events/DataEvent.h"
#include "Events/FilterBankEvent.h"
#include "Wireless/Socket.h"
#include "Events/EventRouter.h"
#include "Shared/debuget.h"
#ifndef PLATFORM_APERIOS
#  include "IPC/MessageReceiver.h"
#endif

using namespace std;

void BufferedImageGenerator::processEvent(const EventBase & event) {
	EventGeneratorBase::processEvent(event);
	if(event.getGeneratorID()!=getListenGeneratorID() || event.getSourceID()!=getListenSourceID())
		return;
	if(event.getTypeID()==EventBase::activateETID) {
		accessed=false;
		const DataEvent<ImageSource>* data=dynamic_cast<const DataEvent<ImageSource>*>(&event);
		if(data==NULL) {
			serr->printf("Error: %s(%s) received event of the wrong type",getClassName().c_str(),getName().c_str());
			return;
		}
		if(imgsrc.layer!=data->getData().layer || imgsrc.channels!=data->getData().channels) {
			//can "quietly" switch the layer of transmission as long as the width and height were scaled properly
			//just need to reset the increments if a different layer is being used.
			for(unsigned int i=0; i<=data->getData().layer; i++) {
				increments[i] = (1<<(data->getData().layer-i))*data->getData().channels;
				strides[i]=(1<<(data->getData().layer-i))*data->getData().width*data->getData().channels;
				skips[i]=strides[i]-data->getData().width*data->getData().channels;
			}
			for(unsigned int i=data->getData().layer+1; i<numLayers; i++) {
				increments[i] = 1;
				strides[i]=data->getData().width*(1<<(i-data->getData().layer));
				skips[i]=0;
			}
		}
		imgsrc=data->getData();
		sysFrameNumber=frameNumber=imgsrc.frameIndex;
		// -- invalidate dependant images: --
		//images in every channel beyond the received channels is invalid
		for(unsigned int i=0; i<numLayers; i++)
			for(unsigned int j=imgsrc.channels; j<numChannels; j++) {
				if(!isAllocated[i][j]) //in case the imgsrc layer changes
					images[i][j]=NULL;
				imageValids[i][j]=false;
			}
		//images in every layer above the received layer is invalid
		for(unsigned int i=imgsrc.layer+1; i<numLayers; i++)
			for(unsigned int j=0; j<imgsrc.channels; j++) { //already covered src.channels to numChannels in previous loop
				if(!isAllocated[i][j]) //in case the imgsrc layer changes
					images[i][j]=NULL;
				imageValids[i][j]=false;
			}
		// -- check if the size has changed: --
		if(numLayers>0) {
			if(imgsrc.width!=getWidth(imgsrc.layer) || imgsrc.height!=getHeight(imgsrc.layer)) {
				freeCaches();
				setDimensions();
				serr->printf("WARNING: the image dimensions don't match values predicted by RobotInfo consts, now %dx%d\n",widths[numLayers-1],heights[numLayers-1]);
			} else if(strides[numLayers-1]==0) {
				// first frame, set it anyway
				setDimensions();
			}
		}
		// -- reassign to the new buffer: --
		for(unsigned int i=imgsrc.layer; i!=-1U; i--) {
			for(unsigned int j=0; j<imgsrc.channels; j++) {
				if(isAllocated[i][j]) { //in case the imgsrc layer changes
					delete [] images[i][j];
					images[i][j]=NULL;
					isAllocated[i][j]=false;
				}
				imageValids[i][j]=true;
			}
			images[i][RawCameraGenerator::CHAN_Y]=imgsrc.img+0;
			images[i][RawCameraGenerator::CHAN_U]=imgsrc.img+2; //note U is *third*
			images[i][RawCameraGenerator::CHAN_V]=imgsrc.img+1; //note V is *second*
		}
		framesProcessed++;
	}
	erouter->postEvent(new FilterBankEvent(this,getGeneratorID(),getSourceID(),event.getTypeID()));
}

unsigned int
BufferedImageGenerator::getBinSize() const {
	unsigned int used=FilterBankGenerator::getBinSize();
	used+=strlen("RawImage")+LoadSave::stringpad;
	used+=widths[selectedSaveLayer]*heights[selectedSaveLayer];
	return used;
}

unsigned int
BufferedImageGenerator::LoadBuffer(const char buf[], unsigned int len) {
	unsigned int origlen=len;
	unsigned int used;
	std::string tmp;
	if(0==(used=FilterBankGenerator::LoadBuffer(buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=decode(tmp,buf,len))) return 0;
	len-=used; buf+=used;
	if(tmp!="RawImage") {
		serr->printf("Unhandled image type for BufferedImageGenerator: %s",tmp.c_str());
		return 0;
	} else if(selectedSaveLayer!=numLayers-1) {
		serr->printf("Can't load into BufferedImageGenerator layer %d!=%d",selectedSaveLayer,numLayers-1);
		return 0;
	} else {
		if(images[selectedSaveLayer][selectedSaveChannel]==NULL)
			images[selectedSaveLayer][selectedSaveChannel]=createImageCache(selectedSaveLayer,selectedSaveChannel);
		unsigned char* img=images[selectedSaveLayer][selectedSaveChannel];
		used=widths[selectedSaveLayer]*heights[selectedSaveLayer];
		ASSERTRETVAL(used<=len,"buffer too small",0);
		memcpy(img,buf,used);
		len-=used; buf+=used;
		imageValids[selectedSaveLayer][selectedSaveChannel]=true;
		return origlen-len;	
	}
}

unsigned int
BufferedImageGenerator::SaveBuffer(char buf[], unsigned int len) const {
	unsigned int origlen=len;
	unsigned int used;
	if(0==(used=FilterBankGenerator::SaveBuffer(buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=encode("RawImage",buf,len))) return 0;
	len-=used; buf+=used;
	
	if(images[selectedSaveLayer][selectedSaveChannel]==NULL) {
		serr->printf("BufferedImageGenerator::SaveBuffer() failed because selected image is NULL -- call selectSaveImage first to make sure it's up to date\n");
		return 0;
	}
	if(!imageValids[selectedSaveLayer][selectedSaveChannel]) {
		serr->printf("BufferedImageGenerator::SaveBuffer() failed because selected image is invalid -- call selectSaveImage first to make sure it's up to date\n");
		return 0;
	}
	unsigned char* img=images[selectedSaveLayer][selectedSaveChannel];
	used=widths[selectedSaveLayer]*heights[selectedSaveLayer];
	ASSERTRETVAL(used<=len,"buffer too small",0);
	unsigned int inc=getIncrement(selectedSaveLayer);
	if(inc==1) {
		//special case, straight copy
		for(unsigned int y=0; y<heights[selectedSaveLayer]; y++) {
			memcpy(buf,img,widths[selectedSaveLayer]);
			buf+=widths[selectedSaveLayer];
			img+=getStride(selectedSaveLayer);
		}
	} else {
		//otherwise, interleaved or subsampling
		for(unsigned int y=0; y<heights[selectedSaveLayer]; y++) {
			unsigned char* const rowend=img+widths[selectedSaveLayer]*inc;
			while(img!=rowend) {
				*buf++=*img;
				img+=inc;
			}
			img+=getSkip(selectedSaveLayer);
		}
	}
	len-=used;
	
	return origlen-len;
}

unsigned int
BufferedImageGenerator::SaveFileStream(FILE * f) const {
	unsigned int totalused=0;
	unsigned int used;
	{ //sigh, inheritance has failed me (I wouldn't want FilterBankGenerator::SaveFileStream() to call the virtuals...)
		unsigned int sz=FilterBankGenerator::getBinSize();
		char * buf = new char[sz];
		memset(buf,0xF0,sz);
		if(buf==NULL) {
			std::cout << "*** WARNING could not allocate " << sz << " bytes for LoadFile";
			return 0;
		}
		unsigned int resp=FilterBankGenerator::SaveBuffer(buf,sz);
		if(resp==0) {
			std::cout << "*** WARNING SaveBuffer didn't write any data (possibly due to overflow or other error)" << std::endl;
			fwrite(buf,1,sz,f);
		}	else {
			unsigned int wrote=fwrite(buf,1,resp,f);
			if(wrote!=resp)
				std::cout << "*** WARNING short write (wrote " << wrote << ", expected " << resp << ")" << std::endl;
		}
		delete [] buf;
		used=resp;
	}
	if(0==used) return 0;
	totalused+=used;
	if(0==(used=encode("RawImage",f))) return 0;
	totalused+=used;
	
	if(images[selectedSaveLayer][selectedSaveChannel]==NULL) {
		serr->printf("BufferedImageGenerator::SaveBuffer() failed because selected image is NULL -- call selectSaveImage first to make sure it's up to date\n");
		return 0;
	}
	if(!imageValids[selectedSaveLayer][selectedSaveChannel]) {
		serr->printf("BufferedImageGenerator::SaveBuffer() failed because selected image is invalid -- call selectSaveImage first to make sure it's up to date\n");
		return 0;
	}
	unsigned char* img=images[selectedSaveLayer][selectedSaveChannel];
	used=widths[selectedSaveLayer]*heights[selectedSaveLayer];
	unsigned int inc=getIncrement(selectedSaveLayer);
	if(inc==1) {
		//special case, straight copy
		sout->printf("Saving %d by %d\n",widths[selectedSaveLayer],heights[selectedSaveLayer]);
		for(unsigned int y=0; y<heights[selectedSaveLayer]; y++) {
			if(fwrite(img,widths[selectedSaveLayer],1,f)==0) {
				serr->printf("short write on image data - ran out of space?\n");
				return 0;
			}
			img+=getStride(selectedSaveLayer);
		}
	} else {
		//otherwise, interleaved or subsampling
		for(unsigned int y=0; y<heights[selectedSaveLayer]; y++) {
			unsigned char* const rowend=img+widths[selectedSaveLayer]*inc;
			while(img!=rowend) {
				if(fputc(*img,f)==EOF) {
					serr->printf("short write on image data - ran out of space?\n");
					return 0;
				}
				img+=inc;
			}
			img+=getSkip(selectedSaveLayer);
		}
	}
	totalused+=used;
	
	return totalused;
}

unsigned char * BufferedImageGenerator::getImage(unsigned int layer, unsigned int channel) {
	if(!accessed && imgsrc.receiver!=NULL) {
		accessed=true;
#ifndef PLATFORM_APERIOS
		imgsrc.receiver->markRead();
#endif
	}
	return FilterBankGenerator::getImage(layer,channel);
}

void BufferedImageGenerator::freeCaches() {
	FilterBankGenerator::freeCaches();
	for(unsigned int i=0; i<numLayers; i++)
		for(unsigned int j=0; j<numChannels; j++)
			isAllocated[i][j]=false;
}

void BufferedImageGenerator::invalidateCaches() {
	for(unsigned int i=0; i<numLayers; i++)
		for(unsigned int j=0; j<numChannels; j++) {
			if(!isAllocated[i][j])
				images[i][j]=NULL;
			imageValids[i][j]=false;
		}
}

unsigned char * BufferedImageGenerator::createImageCache(unsigned int layer, unsigned int channel) const {
	isAllocated[layer][channel]=true;
	return new unsigned char[widths[layer]*heights[layer]];
}
void BufferedImageGenerator::calcImage(unsigned int layer, unsigned int channel) {
	//cout << "BufferedImageGenerator::calcImage(" << layer << ',' << channel << ')' << endl;
	switch(channel) {
		case RawCameraGenerator::CHAN_Y:
			upsampleImage(imgsrc.layer,channel,layer); break;
		case RawCameraGenerator::CHAN_U:
		case RawCameraGenerator::CHAN_V:
			if(imgsrc.channels>1)
				upsampleImage(imgsrc.layer,channel,layer);
			else
				memset(images[layer][channel],128,widths[layer]*heights[layer]);
			break;
		case RawCameraGenerator::CHAN_Y_DY:
		case RawCameraGenerator::CHAN_Y_DX:
		case RawCameraGenerator::CHAN_Y_DXDY:
			if(layer<=imgsrc.layer) {
				//todo
				memset(images[layer][channel],128,widths[layer]*heights[layer]);
			} else if(layer>imgsrc.layer) {
				getImage(imgsrc.layer,channel);
				upsampleImage(imgsrc.layer,channel,layer);
			}
			break;
		default:
			cerr << "Bad layer selection!" << endl;
	}
}
void BufferedImageGenerator::setDimensions() {
	if(imgsrc.img==NULL) //don't have an image to set from
		return;
	// set dimensions of layers below and including the input layer
	for(unsigned int i=0; i<=imgsrc.layer; i++) {
		//s is the scaling factor -- 2 means *half* size
		unsigned int s=(1<<(imgsrc.layer-i));
		//width and height are scaled down (divide by s)
		widths[i]=imgsrc.width/s;
		heights[i]=imgsrc.height/s;
		//need to skip rows to make good on promised height (multiply by s)
		strides[i]=imgsrc.width*imgsrc.channels*s;
		skips[i]=strides[i]-imgsrc.width*imgsrc.channels;
		cout << "setDimensions() " << widths[i] << ' ' << skips[i] << ' ' << strides[i] << endl;
	}
	//set dimensions for layers above the input layer, these are not interleaved
	for(unsigned int i=imgsrc.layer+1; i<numLayers; i++) {
		//s is the scaling factor -- 2 means *double* size
		unsigned int s=(1<<(i-imgsrc.layer));
		//multiply by s
		widths[i]=strides[i]=imgsrc.width*s;
		heights[i]=imgsrc.height*s;
		//stride is same as width (set above) -- we allocate these layers, don't skip rows
		skips[i]=0;
	}
}
void BufferedImageGenerator::destruct() {
	FilterBankGenerator::destruct();
	for(unsigned int i=0; i<numLayers; i++)
		delete [] isAllocated[i];
	delete [] isAllocated;
	isAllocated=NULL;
}
void BufferedImageGenerator::setNumImages(unsigned int nLayers, unsigned int nChannels) {
	if(nLayers==numLayers && nChannels==numChannels)
		return;
	FilterBankGenerator::setNumImages(nLayers,nChannels);
	isAllocated=new bool*[numLayers];
	for(unsigned int i=0; i<numLayers; i++)
		isAllocated[i]=new bool[numChannels];
	setDimensions();
}

void BufferedImageGenerator::upsampleImage(unsigned int srcLayer, unsigned int chan, unsigned int destLayer) {
	unsigned char * cur=images[destLayer][chan];
	ASSERTRET(cur!=NULL,"destination layer is NULL");
	unsigned char * orig=getImage(srcLayer,chan);
	ASSERTRET(orig!=NULL,"source layer is NULL");
	unsigned int width=widths[destLayer];
	unsigned int height=heights[destLayer];
	unsigned int inc=getIncrement(srcLayer);
	int power=destLayer-srcLayer;
	ASSERTRET(power>0,"upsampleImage attempting to downsample")
	
	unsigned char * const imgend=cur+width*height;
	while(cur!=imgend) {
		unsigned char * const row=cur;
		unsigned char * const rowend=cur+width;
		//upsample pixels within one row
		while(cur!=rowend) {
			for(int p=power; p>=0; p--)
				*cur++=*orig;
			orig+=inc;
		}
		//now replicate that row 1<<power times, doubling each time
		for(int p=0; p<power; p++) {
			unsigned int avail=width*(1<<p);
			memcpy(cur,row,avail);
			cur+=avail;
		}
		orig+=getSkip(srcLayer);
	}
}


/*! @file
 * @brief 
 * @author Ethan Tira-Thompson (ejt) (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.12 $
 * $State: Exp $
 * $Date: 2005/08/15 23:19:01 $
 */
