//-*-c++-*-
#ifndef INCLUDED_PListSupport_h_
#define INCLUDED_PListSupport_h_

/*
 From: <!DOCTYPE plist PUBLIC "-//Apple Computer//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
 
<!ENTITY % plistObject "(array | data | date | dict | real | integer | string | true | false )" >
<!ELEMENT plist %plistObject;>
<!ATTLIST plist version CDATA "1.0" >

<!-- Collections -->
<!ELEMENT array (%plistObject;)*>
<!ELEMENT dict (key, %plistObject;)*>
<!ELEMENT key (#PCDATA)>

<!--- Primitive types -->
<!ELEMENT string (#PCDATA)>
<!ELEMENT data (#PCDATA)> <!-- Contents interpreted as Base-64 encoded -->
<!ELEMENT date (#PCDATA)> <!-- Contents should conform to a subset of ISO 8601 (in particular, YYYY '-' MM '-' DD 'T' HH ':' MM ':' SS 'Z'.  Smaller units may be omitted with a loss of precision) -->

<!-- Numerical primitives -->
<!ELEMENT true EMPTY>  <!-- Boolean constant true -->
<!ELEMENT false EMPTY> <!-- Boolean constant false -->
<!ELEMENT real (#PCDATA)> <!-- Contents should represent a floating point number matching ("+" | "-")? d+ ("."d*)? ("E" ("+" | "-") d+)? where d is a digit 0-9.  -->
<!ELEMENT integer (#PCDATA)> <!-- Contents should represent a (possibly signed) integer number in base 10 -->
*/

#include "XMLLoadSave.h"
#include <exception>
#include <string>
#include <iostream>
#include <iomanip>
#include <sstream>
#include <map>
#include <list>

//! A collection of classes to implement the Propery List data storage format, a XML standard used by Apple and others
namespace plist {
	
	//! Base class for the plist listener callbacks
	class Listener {
	public:
		//! destructor
		virtual ~Listener() {}
	};
	
	class PrimitiveBase;
	//! If you wish to be notified any time a particular plist primitive's value has been changed, inherit from this and implement the callback, then register it with the plist object through Primitive::addPrimitiveListener()
	class PrimitiveListener : public Listener {
	public:
		//! This will be called whenever a plist you have registered with is changed
		/*! @a pl is const to help you avoid infinite recursion from an
		 *  accidental modification of @a pl's value -- use a const cast
		 *  if you're sure you know what you're doing */
		virtual void plistValueChanged(const PrimitiveBase& pl)=0;
	};
	
	class ObjectBase;
	class Dictionary;
	//! If you wish to be notified any time an entry is added or removed from a Dictionary, inherit from this and implement one or both of the functions, then register it with the dictionary's Dictionary::addDictionaryListener()
	class DictionaryListener : public Listener {
	public:
		//! This will be called whenever an entry is added to the dictionary
		virtual void plistDictionaryEntryAdded(Dictionary& /*dict*/, ObjectBase& /*primitive*/) {}
		//! This will be called whenever an entry is added to the dictionary
		virtual void plistDictionaryEntryRemoved(Dictionary& /*dict*/, ObjectBase& /*primitive*/) {}
	};
	
	//! This base class provides the root functionality for all plist entities -- Dictionary and the various templated subclasses of PrimitiveBase
	/*! The subclasses may throw std::bad_format if the
	 *  document is poorly structured or bad values are found. */
	class ObjectBase : public XMLLoadSave {
	public:
		ObjectBase(); //!< constructor
		virtual ~ObjectBase()=0; //!< destructor
		
	protected:
		//!@name Inherited from XMLLoadSave
		virtual void setParseTree(xmlDoc * doc) const;
		virtual xmlNode* FindRootXMLElement(xmlDoc* doc) const;
		//@}
		
		//!Provides indirect access to the libxml function of the same name
		//!@name libxml Forwards
		typedef unsigned char xChar;
		static int xStrEqual(const xChar* a, const xChar* b);
		static int xStrCaseEqual(const xChar* a, const xChar* b);
		static xChar* xNodeGetContent(xmlNode* node);
		static void xNodeSetContent(xmlNode* node, const xChar* content);
		static const xChar* xNodeGetName(xmlNode* node);
		static bool xNodeHasName(xmlNode* node, const char* name);
		static void xNodeSetName(xmlNode* node, const xChar* name);
		static xmlAttr* xHasProperty(xmlNode* node, const xChar* name);
		static xChar* xGetProperty(xmlNode* node, const xChar* name);
		static long xGetLineNo(xmlNode* node);
		static void xFree(void* ptr);
		//@}
	};
	
	//! Maintains a set of (key,value) pairs, where a value can be any subclass of ObjectBase
	/*! This class supports callbacks upon modification through the use of the
	 *  DictionaryListener interface.  Note that we only store a pointer to the
	 *  listener list, which is typically unallocated when no listeners are
	 *  active.  This should ensure minimal memory usage per object, as well as
	 *  support safe storage of plist objects in inter-process shared memory
	 *  regions.
	 *
	 *  If you are using these in a shared memory region, just be sure that only
	 *  the process with listeners does any and all modifications, and that it
	 *  unsubscribes before detaching from the region (or else destroys the region
	 *  itself)
	 *
	 *  There isn't a callback if entries themselves are modified, only when new
	 *  entries are added, or old ones removed.  If you want to know any time any
	 *  aspect of any entry is modified, listen for the add and remove callbacks,
	 *  and then add yourself as a listener to each entry individually.  */
	class Dictionary : public ObjectBase {
		friend std::ostream& operator<<(std::ostream& os, const Dictionary& d);
	public:
		//! constructor
		Dictionary() : ObjectBase(), dictionaryListeners(), warnUnused(true), dict(), comments() {autoFormat=false;}
		//! copy constructor (don't assign listeners)
		Dictionary(const Dictionary& d) : ObjectBase(d), dictionaryListeners(), warnUnused(d.warnUnused), dict(d.dict), comments(d.comments) {}
		//! assignment (don't assign listeners)
		Dictionary& operator=(const Dictionary& d) { if(&d==this) return *this; ObjectBase::operator=(d); warnUnused=d.warnUnused; dict=d.dict; comments=d.comments; return *this; }
		//! destructor
		virtual ~Dictionary();
		
		//! insert a new entry to the dictionary, with key @a name and value @a val (replaces any previous entry by same name, but gives a warning)
		void addEntry(const std::string& name, ObjectBase& val);
		//! insert a new entry to the dictionary, with key @a name and value @a val (replaces any previous entry by same name, but gives a warning)
		void addEntry(const std::string& name, ObjectBase& val, const std::string& comment);
		//! replaces previous entry of the same name, or adds it if it doesn't exist; use addEntry if you want to assert that the name does not already exist
		void setEntry(const std::string& name, ObjectBase& val) { dict[name]=&val; }
		//! replaces previous comment for @a name, or adds it if it doesn't already exist (can preceed actual entry!)
		void setComment(const std::string& name, const std::string& comment) { comments[name]=comment; }
		//! returns comment retrieved from loaded file, or any subsequent call to setComment
		const std::string& getComment(const std::string& name) const;
		//! remove the entry with key @a name
		void removeEntry(const std::string& name) { dict.erase(name); }
		//! return the value of the key @a name, or NULL if it doesn't exist
		ObjectBase* findEntry(const std::string& name) const;
		
		void LoadXML(xmlNode* node);
		void SaveXML(xmlNode* node) const;
		
		void setUnusedWarning(bool b) { warnUnused=b; } //!< set #warnUnused
		bool getUnusedWarning() { return warnUnused; } //!< returns #warnUnused

		//! get notified of changes; be sure to call removeValueListener before destructing @a vl!
		virtual void addDictionaryListener(DictionaryListener* vl);
		//! no longer take notification of changes to this object's value
		virtual void removeDictionaryListener(DictionaryListener* vl);
		
	protected:
		//! run through #dictionaryListeners, calling DictionaryListener::plistDictionaryEntryAdded(*this,val)
		virtual void fireEntryAdded(ObjectBase& val);
		//! run through #dictionaryListeners, calling DictionaryListener::plistDictionaryEntryRemoved(*this,val)
		virtual void fireEntryRemoved(ObjectBase& val);
		//! stores a list of the current listeners
		std::list<DictionaryListener*>* dictionaryListeners;

		//! return the length of the longest key for formatting purposes
		unsigned int getLongestKeyLen() const;
		//! if true (the default) LoadXML will give warnings if there are unused entries in the node it is passed
		bool warnUnused;

		//! shorthand for the type of #dict
		typedef std::map<std::string,ObjectBase*> dict_t;
		//! storage of entries -- mapping from keys to value pointers
		dict_t dict;

		//! shorthand for the type of #comments
		typedef std::map<std::string,std::string> comments_t;
		//! storage of entry comments -- mapping from keys to help text comments for manual editing or user feedback
		/*! not every key necessarily has a comment! */
		comments_t comments;
	};
	//! provides textual output
	std::ostream& operator<<(std::ostream& os, const Dictionary& d);


	//! Provides common functionality to all primitive value classes (implemented in a templated subclass Primitive)
	/*! This class supports callbacks upon modification through the use of the
	 *  PrimitiveListener interface.  Note that we only store a pointer to the
	 *  listener list, which is typically unallocated when no listeners are
	 *  active.  This should ensure minimal memory usage per object, as well as
	 *  support safe storage of plist objects in inter-process shared memory
	 *  regions.
	 *
	 *  If you are using these in a shared memory region, just be sure that only
	 *  the process with listeners does any and all modifications, and that it
	 *  unsubscribes before detaching from the region (or else destroys the region
	 *  itself) */
	class PrimitiveBase : public ObjectBase {
	public:
		//! constructor
		PrimitiveBase() : ObjectBase(), primitiveListeners() {}
		//! copy constructor (don't copy listeners)
		PrimitiveBase(const PrimitiveBase& pb) : ObjectBase(pb), primitiveListeners() {}
		//! assignment (don't assign listeners)
		PrimitiveBase& operator=(const PrimitiveBase& pb) { ObjectBase::operator=(pb); return *this; }
		//! destructor
		~PrimitiveBase();
		
		//! assign a new value
		virtual void set(const std::string& str)=0;
		//! return current value as a string
		virtual std::string get() const=0;

		//! get notified of changes; be sure to call removeValueListener before destructing @a vl!
		virtual void addPrimitiveListener(PrimitiveListener* vl);
		//! no longer take notification of changes to this object's value
		virtual void removePrimitiveListener(PrimitiveListener* vl);

	protected:
		//! run through #primitiveListeners, calling PrimitiveListener::plistValueChanged(*this)
		virtual void fireValueChanged() const;
		//! stores a list of the current listeners
		std::list<PrimitiveListener*>* primitiveListeners;
	};
	//! output stringified value (from PrimitiveBase::get()) to stream
	inline std::ostream& operator<<(std::ostream& os, const PrimitiveBase& pb) {
		return os << pb.get();
	}
	//! input value from next word in @a is, via PrimitiveBase::set()
	inline std::istream& operator>>(std::istream& is, PrimitiveBase& pb) {
		std::string s;
		is >> s;
		pb.set(s);
		return is;
	}
	

	//! Implements type-specific functionality through template specialization, mainly involving value conversion and stringification formatting
	/*! Provides smart-pointer style functionality for transparent
	 *  access to the value storage, as well as automatic casting
	 *  to and from the value type so you can almost always treat
	 *  the Primitive as if it was the value itself. */
	template<typename T>
	class Primitive : public PrimitiveBase {
	public:
		Primitive() : ObjectBase(), val() {} //!< constructor
		Primitive(const T& v) : ObjectBase(), val(v) {} //!< constructor, automatic casting from the value type
		Primitive& operator=(const T& v) { val=v; fireValueChanged(); return *this; } //!< assignment
		//T& operator*() { return val; } //!< access value -- if #val is modified, this won't know, so PrimitiveListeners are going to be clueless
		const T& operator*() const { return val; } //!< smart pointer access to value
		//T* operator->() { return &val; } //!< smart pointer access to value -- if #val is modified, this won't know, so PrimitiveListeners are going to be clueless
		const T* operator->() const { return &val; } //!< smart pointer access to value
		operator T() const { return val; } //!< automatic casting to the value type
/*		void LoadXML(xmlNode* node) {
			if(node==NULL)
				return;
			//decode base64 from xml node
			//val->LoadBuffer(buf,bufsize);
			fireValueChanged(); 
		}
		void SaveXML(xmlNode* node) const {
			if(node==NULL)
				return;
			unsigned int bufsize=val->getBinSize();
			char * buf = new char[bufsize];
			val->SaveBuffer(buf,bufsize);
			//base64 encode into xml node
			delete [] buf;
		}*/
	protected:
		T val; //!< value storage
	};

	//! provides a @c bool specialization of Primitive<T>
	/*! A @c bool can be treated as either a string or an integer\n
	 *  When saving, "true" or "false" will be used, but users could
	 *  also specify "yes"/"no" or "on"/"off".  If an integer is used,
	 *  it will be interpreted as usual: 0==false, otherwise true. */
	template<>
	class Primitive<bool> : public PrimitiveBase {
	public:
		Primitive() : PrimitiveBase(), val() {} //!< constructor
		Primitive(const bool& v) : PrimitiveBase(), val(v) {} //!< casting constructor
		Primitive& operator=(const bool& v) { val=v; fireValueChanged(); return *this; } //!< assignment constructor
		//bool& operator*() { return val; }
		const bool& operator*() const { return val; } //!< dereference will return data storage
		//bool* operator->() { return &val; }
		const bool* operator->() const { return &val; } //!< can use -> to access members of data storage
		operator bool() const { return val; } //!< casting operator
		void LoadXML(xmlNode* node) {
			if(node==NULL)
				return;
			if(xNodeHasName(node,"true")) {
				val=true;
				fireValueChanged(); 
			} else if(xNodeHasName(node,"false")) {
				val=false;
				fireValueChanged(); 
			} else if(xNodeHasName(node,"integer") || xNodeHasName(node,"real")) {
				xChar * cont=xNodeGetContent(node);
				std::stringstream str((char*)cont);
				str >> val;
				xFree(cont);
				fireValueChanged(); 
			} else if(xNodeHasName(node,"string")) {
				xChar * cont=xNodeGetContent(node);
				try {
					set((char*)cont);
					std::cerr << "Warning: plist boolean expects value of '<true/>', '<false/>', or numeric.  String value of '" << (char*)cont << "' is not recommended." << std::endl;
				} catch(const bad_format& err) {
					xFree(cont);
					throw bad_format(node,err.what());
				} catch(...) {
					xFree(cont);
					throw;
				}
				xFree(cont);
			} else
				throw bad_format(node,"Error: plist boolean must be 'true', 'false', or numeric type");
		}
		void SaveXML(xmlNode* node) const {
			if(node==NULL)
				return;
			xNodeSetName(node,(const xChar*)(val?"true":"false"));
			xNodeSetContent(node,NULL);
		}
		void set(const std::string& str) {
			if(str=="true" || str=="yes" || str=="on")
				val=true;
			else if(str=="false" || str=="no" || str=="off")
				val=false;
			else {
				float t;
				if(sscanf(str.c_str(),"%g",&t)==0)
					throw bad_format(NULL,"Error: plist boolean must be 'true', 'false', or numeric type");
				val=t;
			}
			fireValueChanged(); 
		}
		std::string get() const {
			return val?"true":"false";
		}
	protected:
		bool val; //!< the actual data storage
	};

	//! provides a @c char specialization of Primitive<T>
	/*! A @c char can be treated as either a string or an integer, you can use
	 *  the setNumeric(bool) function to indicate which style to use when saving */
	template<>
	class Primitive<char> : public PrimitiveBase {
	public:
		Primitive() : PrimitiveBase(), val(), numeric(false) {} //!< constructor
		Primitive(const char& v, bool isNum=false) : PrimitiveBase(), val(v), numeric(isNum) {} //!< casting constructor
		Primitive& operator=(const char& v) { val=v; fireValueChanged(); return *this; } //!< assignment
		Primitive& operator+=(const char& v) { val+=v; fireValueChanged(); return *this; } //!< add-assign
		Primitive& operator-=(const char& v) { val-=v; fireValueChanged(); return *this; } //!< subtract-assign
		Primitive& operator*=(const char& v) { val*=v; fireValueChanged(); return *this; } //!< multiply-assign
		Primitive& operator/=(const char& v) { val/=v; fireValueChanged(); return *this; } //!< divide-assign
		//char& operator*() { return val; }
		const char& operator*() const { return val; } //!< dereference will return data storage
		//char* operator->() { return &val; }
		const char* operator->() const { return &val; } //!< can use -> to access members of data storage
		operator char() const { return val; } //!< casting operator
		void setNumeric(bool isNum) { numeric=isNum; } //!< sets #numeric
		bool getNumeric() const { return numeric; } //!< returns #numeric
		void LoadXML(xmlNode* node) {
			if(node==NULL)
				return;
			xChar* cont=xNodeGetContent(node);
			try {
				if(xNodeHasName(node,"string")) {
					set((char*)cont);
				} else if(xNodeHasName(node,"integer")) {
					val=strtol((const char*)cont,NULL,0);
					fireValueChanged(); 
				} else if(xNodeHasName(node,"real")) {
					val=(char)strtod((const char*)cont,NULL);
					fireValueChanged(); 
				} else if(xNodeHasName(node,"true")) {
					val=true;
					fireValueChanged(); 
				} else if(xNodeHasName(node,"false")) {
					val=false;
					fireValueChanged(); 
				} else {
					throw bad_format(node,"Error: plist char must be either a string or integer");
				} 
			} catch(const bad_format& err) {
				xFree(cont);
				throw bad_format(node,err.what());
			} catch(...) {
				xFree(cont);
				throw;
			}
			xFree(cont);
		}
		void SaveXML(xmlNode* node) const {
			if(node==NULL)
				return;
			if(numeric) {
				xNodeSetName(node,(const xChar*)"integer");
				std::stringstream str;
				str << (int)val;
				xNodeSetContent(node,(const xChar*)str.str().c_str());
			} else {
				xNodeSetName(node,(const xChar*)"string");
				xChar str[2] = {val,'\0'};
				xNodeSetContent(node,(const xChar*)str);
			}
		}
		void set(const std::string& str) {
			if(str.size()==0)
				throw bad_format(NULL,"Error: plist char must have non-empty content");
			val=str[0];
			if(str.size()!=1)
				std::cerr << "Warning: plist expected single char, found multi-char string '" << str << "'" << std::endl;
			fireValueChanged(); 
		}
		std::string get() const {
			return std::string(1,val);
		}
	protected:
		char val; //!< data storage
		bool numeric; //!< if true, requests that saves store the numeric value instead of corresponding character
	};
	
	//! provides an @c unsigned @c char specialization of Primitive<T>
	/*! A @c char can be treated as either a string or an integer, you can use
	 *  the setNumeric(bool) function to indicate which style to use when saving */
	template<>
	class Primitive<unsigned char> : public PrimitiveBase {
	public:
		Primitive() : PrimitiveBase(), val(), numeric(false) {} //!< constructor
		Primitive(const unsigned char& v, bool isNum=false) : PrimitiveBase(), val(v), numeric(isNum) {} //!< casting constructor
		Primitive& operator=(const unsigned char& v) { val=v; fireValueChanged(); return *this; } //!< assignment
		Primitive& operator+=(const unsigned char& v) { val+=v; fireValueChanged(); return *this; } //!< add-assign
		Primitive& operator-=(const unsigned char& v) { val-=v; fireValueChanged(); return *this; } //!< subtract-assign
		Primitive& operator*=(const unsigned char& v) { val*=v; fireValueChanged(); return *this; } //!< multiple-assign
		Primitive& operator/=(const unsigned char& v) { val/=v; fireValueChanged(); return *this; } //!< divide-assign
		//unsigned char& operator*() { return val; }
		const unsigned char& operator*() const { return val; } //!< dereference will return data storage
		//unsigned char* operator->() { return &val; }
		const unsigned char* operator->() const { return &val; } //!< can use -> to access members of data storage
		operator unsigned char() const { return val; } //!< casting operator
		void setNumeric(bool isNum) { numeric=isNum; } //!< sets #numeric
		bool getNumeric() const { return numeric; } //!< returns #numeric
		void LoadXML(xmlNode* node) {
			if(node==NULL)
				return;
			xChar* cont=xNodeGetContent(node);
			try {
				if(xNodeHasName(node,"string")) {
					set((char*)cont);
				} else if(xNodeHasName(node,"integer")) {
					val=strtol((const char*)cont,NULL,0);
					fireValueChanged(); 
				} else if(xNodeHasName(node,"real")) {
					val=(char)strtod((const char*)cont,NULL);
					fireValueChanged(); 
				} else if(xNodeHasName(node,"true")) {
					val=true;
					fireValueChanged(); 
				} else if(xNodeHasName(node,"false")) {
					val=false;
					fireValueChanged(); 
				} else {
					throw bad_format(node,"Error: plist unsigned char must be either a string or integer");
				} 
			} catch(const bad_format& err) {
				xFree(cont);
				throw bad_format(node,err.what());
			} catch(...) {
				xFree(cont);
				throw;
			}
			xFree(cont);
		}
		void SaveXML(xmlNode* node) const {
			if(node==NULL)
				return;
			if(numeric) {
				xNodeSetName(node,(const xChar*)"integer");
				std::stringstream str;
				str << (int)val;
				xNodeSetContent(node,(const xChar*)str.str().c_str());
			} else {
				xNodeSetName(node,(const xChar*)"string");
				xChar str[2] = {val,'\0'};
				xNodeSetContent(node,(const xChar*)str);
			}
		}
		void set(const std::string& str) {
			if(str.size()==0)
				throw bad_format(NULL,"Error: plist char must have non-empty content");
			val=str[0];
			if(str.size()!=1)
				std::cerr << "Warning: plist expected single char, found multi-char string '" << str << "'" << std::endl;
		}
		std::string get() const {
			return std::string(1,val);
		}
	protected:
		unsigned char val; //!< data storage
		bool numeric; //!< if true, requests that saves store the numeric value instead of corresponding character
	};

	
//! a macro to provide template specializations for the numeric primitives, which only need to vary their string name
#define PLIST_OBJECT_SPECIALIZATION(T,PRIM) \
	template<> \
	class Primitive<T> : public PrimitiveBase { \
	public: \
		Primitive() : PrimitiveBase(), val() {} \
		Primitive(const T& v) : PrimitiveBase(), val(v) {} \
		Primitive& operator=(const T& v) { val=v; fireValueChanged(); return *this; } \
		Primitive& operator+=(const T& v) { val+=v; fireValueChanged(); return *this; } \
		Primitive& operator-=(const T& v) { val-=v; fireValueChanged(); return *this; } \
		Primitive& operator*=(const T& v) { val*=v; fireValueChanged(); return *this; } \
		Primitive& operator/=(const T& v) { val/=v; fireValueChanged(); return *this; } \
		/*T& operator*() { return val; }*/ \
		const T& operator*() const { return val; } \
		/*T* operator->() { return &val; }*/ \
		const T* operator->() const { return &val; } \
		operator T() const { return val; } \
		void LoadXML(xmlNode* node) { \
			if(node==NULL) \
				return; \
			bool bt=xNodeHasName(node,"true"); \
			bool bf=xNodeHasName(node,"false"); \
			if(!bt && !bf && !xNodeHasName(node,"integer") && !xNodeHasName(node,"real") && !xNodeHasName(node,"string")) \
				throw bad_format(node,"Error: plist "#T" expects "PRIM" type"); \
			if(!xNodeHasName(node,PRIM)) \
				std::cerr << "Warning: plist expected "PRIM" got " << (const char*)xNodeGetName(node) << ", trying to convert. (line " << xGetLineNo(node) << ")" << std::endl; \
			if(bt) \
				val = true; \
			else if(bf) \
				val = false; \
			else { \
				xChar * cont=xNodeGetContent(node); \
				std::stringstream str((const char*)cont); \
				str >> val; \
				xFree(cont); \
			} \
			fireValueChanged(); \
		} \
		void SaveXML(xmlNode* node) const { \
			if(node==NULL) \
				return; \
			xNodeSetName(node,(const xChar*)PRIM); \
			std::stringstream str; \
			str <<std::setprecision(128)<< val; \
			xNodeSetContent(node,(const xChar*)str.str().c_str()); \
		} \
		void set(const std::string& str) { \
			std::stringstream sstr(str); \
			sstr >> val; \
			fireValueChanged(); \
		} \
		std::string get() const { \
			std::stringstream sstr; \
			sstr <<std::setprecision(128)<< val; \
			return sstr.str(); \
		} \
	protected: \
		T val; \
	}
	
	PLIST_OBJECT_SPECIALIZATION(short,"integer"); //!< short instance of PrimitiveBase ("integer")
	PLIST_OBJECT_SPECIALIZATION(unsigned short,"integer"); //!< unsigned short instance of PrimitiveBase ("integer")
	PLIST_OBJECT_SPECIALIZATION(int,"integer"); //!< int instance of PrimitiveBase ("integer")
	PLIST_OBJECT_SPECIALIZATION(unsigned int,"integer"); //!< unsigned int instance of PrimitiveBase ("integer")
	PLIST_OBJECT_SPECIALIZATION(long,"integer"); //!< long instance of PrimitiveBase ("integer")
	PLIST_OBJECT_SPECIALIZATION(unsigned long,"integer"); //!< unsigned long instance of PrimitiveBase ("integer")
	PLIST_OBJECT_SPECIALIZATION(long long,"integer"); //!< long long instance of PrimitiveBase ("integer")
	PLIST_OBJECT_SPECIALIZATION(unsigned long long,"integer"); //!< unsigned long long instance of PrimitiveBase ("integer")
	PLIST_OBJECT_SPECIALIZATION(float,"real"); //!< float instance of PrimitiveBase ("real")
	PLIST_OBJECT_SPECIALIZATION(double,"real"); //!< double instance of PrimitiveBase ("real")

	//! Provides a @c std::string specialization of Primitive<T>
	/*! Doesn't need to provide a operator cast because we subclass std::string itself! */
	template<>
	class Primitive<std::string> : public PrimitiveBase, public std::string {
	public:
		Primitive() : PrimitiveBase(), std::string() {} //!< constructor
		Primitive(const std::string& v) : PrimitiveBase(), std::string(v) {} //!< casting constructor
		Primitive& operator=(const std::string& v) { std::string::operator=(v); fireValueChanged(); return *this; } //!< assignment
		//std::string& operator*() { return *this; }
		const std::string& operator*() const { return *this; } //!< dereference will return data storage as a string (for uniformity with the other Primitives, although unnecessary with this instantiation)
		//std::string* operator->() { return this; }
		const std::string* operator->() const { return this; } //!< returns a pointer to this (for uniformity with the other Primitives, although unnecessary with this instantiation)
		//no casting operator because we subclass string
		void LoadXML(xmlNode* node) {
			if(node==NULL)
				return;
			if(xNodeHasName(node,"string")) {
				xChar * cont=xNodeGetContent(node);
				*this=(char*)cont;
				xFree(cont);
			} else {
				if(xNodeHasName(node,"integer") || xNodeHasName(node,"real")) {
					xChar * cont=xNodeGetContent(node);
					*this=(char*)cont;
					xFree(cont);
				} else if(xNodeHasName(node,"true"))
					*this="true";
				else if(xNodeHasName(node,"false"))
					*this="false";
				else
					throw bad_format(node,"Error: plist string must be 'true', 'false', or numeric type");
				std::cerr << "Warning: plist string expected, found " << (const char*)xNodeGetName(node) << " on line " << xGetLineNo(node) << std::endl;
				fireValueChanged(); 
			}
		}
		void SaveXML(xmlNode* node) const {
			if(node==NULL)
				return;
			xNodeSetName(node,(const xChar*)"string");
			xNodeSetContent(node,(const xChar*)c_str());
		}
		void set(const std::string& str) {
			*this=str;
			fireValueChanged(); 
		}
		std::string get() const {
			return *this;
		}
	};
	
	//! Provides an interface for the use of enumerations in a plist -- you can specify values by either the string name or the corresponding integer value
	/*! Where an array of names is required, you must order the array such that
	 *  the enumeration value can be used as an index into the array.
	 *  The 'maxval' parameters should be one above the maximum enumeration -- 
	 *  if your enumeration runs sequentially from 0 to n, #max should be the
	 *  number of enumerations: n+1 */
	template<typename T> 
	class NamedEnumeration : public PrimitiveBase {
	public:
		NamedEnumeration() : PrimitiveBase(), val(), names(NULL), max(0) {} //!< constructor
		NamedEnumeration(const NamedEnumeration& ne) : PrimitiveBase(ne), val(ne.val), names(ne.names), max(ne.max) {} //!< copy constructor
		NamedEnumeration(const T& v, const char * const* enumnames, unsigned int maxval) : PrimitiveBase(), val(v), names(enumnames), max(maxval) {} //!< constructor, pass initial value, array of strings (the names), and the one-plus-maximum enum value (i.e. the number of enumeration values if they run sequentially from 0)
		NamedEnumeration(const T& v) : PrimitiveBase(), val(v), names(NULL), max(0) {} //!< automatic casting from the enumeration
		NamedEnumeration& operator=(const T& v) { val=v; fireValueChanged(); return *this; } //!< assignment from enumeration value (numeric)
		NamedEnumeration& operator=(const std::string& v) { set(v); return *this; } //!< assignment from string
		NamedEnumeration& operator=(const NamedEnumeration& ne) { PrimitiveBase::operator=(ne); val=ne.val; return *this; } //!< assignment
		//T& operator*() { return val; }
		const T& operator*() const { return val; } //!< value access
		operator T() const { return val; } //!< automatic casting to the enumeration value
		void setNames(const char *  const* enumnames, unsigned int maxval) { names=enumnames; max=maxval; } //!< (re)set the array of names and one-plus-maximum enum value (i.e. the number of enumeration values if they run sequentially from 0)
		const char*  const* getNames() const { return names; } //!< returns the array of names previously provided from constructor or setNames()
		const char* getName(unsigned int i) const { return names[i]; } //!< returns the name for a particular index
		unsigned int getMax() const { return max; } //!< returns the one-past-maximum of enumeration values previously provided to constructor or setNames()

		void LoadXML(xmlNode* node) {
			if(node==NULL)
				return;
			if(xNodeHasName(node,"true") || xNodeHasName(node,"false")) {
				std::string name=(const char*)xNodeGetName(node);
				unsigned int i=findName(name.c_str());
				if(i==-1U)
					throw bad_format(node,("Error: plist NamedEnumeration cannot be '"+name+"'").c_str());
				val=static_cast<T>(i);
				std::cerr << "Warning: plist NamedEnumeration should use <string>" << name << "</string>, not <" << name << "/>" << std::endl;
			} else if(xNodeHasName(node,"integer") || xNodeHasName(node,"real") || xNodeHasName(node,"string")) {
				xChar * cont=xNodeGetContent(node);
				try {
					set((char*)cont);
				} catch(const bad_format& err) {
					xFree(cont);
					throw bad_format(node,err.what());
				} catch(...) {
					xFree(cont);
					throw;
				}
				xFree(cont);
			} else
				throw bad_format(node,"Error: plist NamedEnumeration must be numeric or valid string");
		}
		void SaveXML(xmlNode* node) const {
			if(node==NULL)
				return;
			if(names!=NULL && names[val]!=NULL && val>0 && (unsigned int)val<max) {
				xNodeSetName(node,(const xChar*)"string");
			} else {
				xNodeSetName(node,(const xChar*)"integer");
			}
			xNodeSetContent(node,(const xChar*)get().c_str());
		}
		void set(const std::string& str) {
			unsigned int i=findName(str.c_str());
			if(i==-1U) {
				if(sscanf(str.c_str(),"%d",&i)==0)
					throw bad_format(NULL,"Error: plist NamedEnumeration must be numeric or valid string");
			}
			val=static_cast<T>(i);
			fireValueChanged(); 
		}
		std::string get() const {
			if(names!=NULL && names[val]!=NULL && val>=0 && (unsigned int)val<max)
				return names[val];
			std::stringstream str;
			str << val;
			return str.str();
		}
	protected:
		//! returns the enum corresponding to @a name
		unsigned int findName(const char* name) {
			if(name==NULL || names==NULL)
				return -1U;
			for(unsigned int i=0; i<max; i++)
				if(names[i] && strcmp(name,names[i])==0)
					return i;
			return -1U;
		}
		T val; //!< storage of enum value
		const char *  const* names; //!< pointer to array of names -- enum value must be able to serve as index for lookup
		unsigned int max; //!< one-plus-maximum enum value, i.e. the number of enum entries if they are ordered sequentially from 0
	};
	
	//! output of an ObjectBase to a stream
	inline std::ostream& operator<<(std::ostream& os, const ObjectBase& pb) {
		if(const PrimitiveBase * pbp=dynamic_cast<const PrimitiveBase*>(&pb))
			os << *pbp;
		else if(const Dictionary * dp=dynamic_cast<const Dictionary*>(&pb))
			os << *dp;
		else
			os << "[Unknown ObjectBase subclass, edit " << __FILE__ << ":" << __LINE__ << "]";
		return os;
	}
	
} //namespace plist

/*! @file
 * @brief Provides the plist namespace, a collection of classes to implement the Propery List data storage format, a XML standard used by Apple and others
 * @author Ethan Tira-Thompson (ejt) (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.9 $
 * $State: Exp $
 * $Date: 2005/08/07 04:11:04 $
 */

#endif
