#include "plist.h"
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <iomanip>

using namespace std;

namespace plist {

	ObjectBase::ObjectBase()
	  : XMLLoadSave()
	{}
	
	ObjectBase::~ObjectBase() {}
	
	void ObjectBase::setParseTree(xmlDoc * doc) const {
		XMLLoadSave::setParseTree(doc);
		if(xmldocument==NULL)
			return;
		xmlNodePtr cur = xmlNewNode(NULL,(const xmlChar*)"plist");
		xmlNewProp(cur,(const xmlChar*)"version",(const xmlChar*)"1.0");
		xmlDocSetRootElement(xmldocument,cur);
		xmlCreateIntSubset(xmldocument,(const xmlChar*)"plist",(const xmlChar*)"-//Apple Computer//DTD PLIST 1.0//EN",(const xmlChar*)"http://www.apple.com/DTDs/PropertyList-1.0.dtd");
	}
	
	xmlNode* ObjectBase::FindRootXMLElement(xmlDoc* doc) const {
		if(doc==NULL)
			return NULL;
		xmlNode* root=XMLLoadSave::FindRootXMLElement(doc);
		string filename;
		if(doc->name!=NULL && doc->name[0]!='\0') {
			filename="document '";
			filename+=doc->name;
			filename+="' ";
		}
		if (root == NULL)
			throw bad_format(root,"Error: plist read empty document");
		if (xmlStrcmp(root->name, (const xmlChar *)"plist"))
			throw bad_format(root,"Error: plist read document of the wrong type, root node != plist");
		if (!xmlHasProp(root,(const xmlChar*)"version"))
			cerr << "Warning: plist " << filename << "does not carry version number, assuming 1.0" << endl;
		else {
			xmlChar* strv=xmlGetProp(root,(const xmlChar*)"version");
			double version=strtod((const char*)strv,NULL);
			if(version>1.0)
				cerr << "WARNING: plist " << filename << "is version " << strv << ", this software only knows 1.0.  Trying anyway..." << endl;
			if(version==0)
				cerr << "WARNING: plist " << filename << "seems to have invalid version '" << strv << "', this software only knows 1.0.  Trying anyway..." << endl;
			xmlFree(strv);
		}
		
		// find first element node within the plist
		xmlNode* cur=root->children;
		while(cur!=NULL && cur->type!=XML_ELEMENT_NODE)
			cur=cur->next;
		if(cur==NULL) //empty plist
			cur = xmlNewChild(root,NULL,(const xmlChar*)"",NULL);
		return cur;
	}
	
	int ObjectBase::xStrEqual(const xChar* a, const xChar* b) {
		return xmlStrEqual(a,b);
	}
	int ObjectBase::xStrCaseEqual(const xChar* a, const xChar* b) {
		return !xmlStrcasecmp(a,b);
	}
	ObjectBase::xChar* ObjectBase::xNodeGetContent(xmlNode* node) {
		return xmlNodeGetContent(node);
	}
	void ObjectBase::xNodeSetContent(xmlNode* node, const xChar* content) {
		xmlNodeSetContent(node,content);
	}
	const ObjectBase::xChar* ObjectBase::xNodeGetName(xmlNode* node) {
		return node->name;
	}
	bool ObjectBase::xNodeHasName(xmlNode* node, const char* name) {
		return xmlStrEqual(node->name,(const xmlChar*)name);
	}
	void ObjectBase::xNodeSetName(xmlNode* node, const xChar* name) {
		xmlNodeSetName(node,name);
	}
	xmlAttr* ObjectBase::xHasProperty(xmlNode* node, const xChar* name) {
		return xmlHasProp(node,name);
	}
	ObjectBase::xChar* ObjectBase::xGetProperty(xmlNode* node, const xChar* name) {
		return xmlGetProp(node,name);
	}
	long ObjectBase::xGetLineNo(xmlNode* node) {
		return xmlGetLineNo(node);
	}
	void ObjectBase::xFree(void* ptr) {
		xmlFree(ptr);
	}

	Dictionary::~Dictionary() {
		delete dictionaryListeners;
		dictionaryListeners=NULL;
	}

	void Dictionary::addEntry(const std::string& name, ObjectBase& val) {
		dict_t::iterator it=dict.find(name);
		if(it!=dict.end()) {
			cerr << "Warning: addEntry("<<name<<","<<val<<") conflicts with previous addEntry("<<name<<","<<val<<")"<<endl;
			cerr << "         (use setEntry() if you expect you might need to overwrite)" << endl;
			it->second=&val;
			return;
		}
		dict[name]=&val;
	}

	void Dictionary::addEntry(const std::string& name, ObjectBase& val, const std::string& comment) {
		comments[name]=comment;
		dict_t::iterator it=dict.find(name);
		if(it!=dict.end()) {
			cerr << "Warning: addEntry("<<name<<","<<val<<") conflicts with previous addEntry("<<name<<","<<val<<")"<<endl;
			cerr << "         (use setEntry() if you expect you might need to overwrite)" << endl;
			it->second=&val;
			return;
		}
		dict[name]=&val;
	}
	
	const std::string& Dictionary::getComment(const std::string& name) const {
		comments_t::const_iterator it=comments.find(name);
		static const std::string empty;
		return (it!=comments.end()) ? it->second : empty;
	}

	ObjectBase* Dictionary::findEntry(const std::string& name) const {
		//do we have a key with this name?
		dict_t::const_iterator it=dict.find(name);
		if(it!=dict.end())
			return it->second; //yes, return it
		
		//perhaps there's a sub-dictionary, separated by '.'
		size_t p=name.find(".");
		if(p==string::npos)
			return NULL; //no, '.'s found -- go away
		it=dict.find(name.substr(0,p));
		if(it==dict.end())
			return NULL; //no entry matching prefix -- go away
		const Dictionary* d=dynamic_cast<const Dictionary*>(it->second);
		if(d==NULL)
			return NULL; //matching prefix is not a dictionary -- go away
		
		//found a matching sub-dictionary, have it find the rest recursively
		return d->findEntry(name.substr(p+1));
	}

	void Dictionary::LoadXML(xmlNode* node) {
		//check if our node has been set to NULL (invalid or not found)
		if(node==NULL)
			return;
		
		std::string comment;
		
		//process children nodes
		for(xmlNode* cur = skipToElement(node->children,comment); cur!=NULL; cur = skipToElement(cur->next,comment)) {
						
			//find the next key node
			xmlNode * k=cur;
			if(xmlStrcmp(k->name, (const xmlChar *)"key"))
				throw bad_format(k,"Dictionary format error: expect data in pairs of key and value (two values found in a row)");
			cur=skipToElement(cur->next);
			
			//find the following value (non-key) node
			xmlNode * v=cur;
			if(v==NULL)
				throw bad_format(cur,"Dictionary format error: expect data in pairs of key and value (dictionary ended with hanging key)");
			if(!xmlStrcmp(v->name, (const xmlChar *)"key"))
				throw bad_format(v,"Dictionary format error: expect data in pairs of key and value (two keys found in a row)");
			
			//find corresponding entry
			xmlChar* cont=xmlNodeGetContent(k);
			string key=(const char*)cont;
			xmlFree(cont);
			dict_t::const_iterator it=dict.find(key);
			if(it==dict.end()) {
				if(warnUnused)
					cerr << "Warning: reading plist dictionary, key '" << key << "' does not match a registered variable.  Ignoring..." << endl;
				continue;
			}
			if(comment.size()!=0)
				setComment(key,comment);
			it->second->LoadXML(v);
		}
	}
	
	void Dictionary::SaveXML(xmlNode* node) const {
		//check if our node has been set to NULL (invalid or not found)
		if(node==NULL)
			return;
		
		//set the type of the current node
		xmlNodeSetName(node,(const xmlChar*)"dict");
		
		// we'll use this to keep track of which nodes were already present, so we'll
		// know which ones were missing for which we need to make new nodes
		dict_t seen;

		//find the depth of the target node in the xml tree to maintain proper indentation
		std::string perIndent("    ");
		std::string indentStr;
		for(xmlNode* cur=node->parent; cur!=NULL; cur=cur->parent) {
			if((void*)cur==(void*)node->doc) { //if we hit the document node, discount it and we're done
				if(indentStr.size()>0)
					indentStr=indentStr.substr(0,indentStr.size()-perIndent.size());
				break;
			}
			indentStr+=perIndent;
		}
		
		//This will hold any comments found between elements -- if no comment is found, a new one may be added
		std::string comment;

		//process children nodes
		for(xmlNode* cur = skipToElement(node->children,comment); cur!=NULL; cur = skipToElement(cur->next,comment)) {
			
			//find the next key node
			xmlNode * k=cur;
			if(xmlStrcmp(k->name, (const xmlChar *)"key"))
				throw bad_format(k,"Dictionary format error: expect data in pairs of key and value (two values found in a row)");
			cur=skipToElement(cur->next);
			
			//find the following value (non-key) node
			xmlNode * v=cur;
			if(v==NULL)
				throw bad_format(cur,"Dictionary format error: expect data in pairs of key and value (dictionary ended with hanging key)");
			if(!xmlStrcmp(v->name, (const xmlChar *)"key"))
				throw bad_format(v,"Dictionary format error: expect data in pairs of key and value (two keys found in a row)");
			
			//find corresponding entry
			xmlChar* cont=xmlNodeGetContent(k);
			string key=(const char*)cont;
			xmlFree(cont);
			dict_t::const_iterator it=dict.find(key);
			if(it==dict.end()) {
				if(warnUnused)
					cerr << "Warning: reading plist dictionary, key '" << key << "' does not match a registered variable.  Ignoring..." << endl;
				continue;
			}
			if(comment.size()==0) {
				bool isSub=dynamic_cast<Dictionary*>(it->second);
				if(isSub) {
					xmlAddPrevSibling(k,xmlNewText((const xmlChar*)"\n"));
					xmlAddPrevSibling(k,xmlNewComment((const xmlChar*)("======== "+it->first+" ========").c_str()));
				}
				comments_t::const_iterator cit=comments.find(key);
				if(cit!=comments.end()) {
					if(isSub || cit->second.substr(0,key.size())==key)
						comment=cit->second;
					else //if not a sub-dictionary, and comment doesn't already start with entry name, prepend entry name
						comment=key+": "+cit->second;
					xmlAddPrevSibling(k,xmlNewText((const xmlChar*)"\n"));
					xmlAddPrevSibling(k,xmlNewComment((const xmlChar*)comment.c_str()));
					xmlAddPrevSibling(k,xmlNewText((const xmlChar*)("\n"+indentStr).c_str()));
				}
			}
			it->second->SaveXML(v);
			seen.insert(*it);
		}

		if(seen.size()!=dict.size()) {
			// the main dictionary has entries that weren't seen... find which ones
			// if needed, this could be made faster (O(n) vs. current O(n lg n)) by assuming the maps
			// are sorted and moving two iterators through together instead of repeated find()'s
			for(dict_t::const_iterator it=dict.begin(); it!=dict.end(); ++it) {
				if(seen.find(it->first)==seen.end()) {
					//we didn't see this node in the existing xml tree, have to add a new node pair for it
					bool isSub=dynamic_cast<Dictionary*>(it->second);
					if(isSub) {
						xmlAddChild(node,xmlNewText((const xmlChar*)"\n"));
						xmlAddChild(node,xmlNewComment((const xmlChar*)("======== "+it->first+" ========").c_str()));
					}
					comments_t::const_iterator cit=comments.find(it->first);
					if(cit!=comments.end()) {
						if(isSub || cit->second.substr(0,it->first.size())==it->first)
							comment=cit->second;
						else
							comment=it->first+": "+cit->second;
						xmlAddChild(node,xmlNewText((const xmlChar*)"\n"));
						xmlAddChild(node,xmlNewComment((const xmlChar*)comment.c_str()));
					}
					xmlAddChild(node,xmlNewText((const xmlChar*)("\n"+indentStr).c_str()));
					xmlNode* k=xmlNewChild(node,NULL,(const xmlChar*)"key",(const xmlChar*)it->first.c_str());
					if(k==NULL)
						throw bad_format(node,"Error: plist Dictionary xml error on saving key");
					xmlAddChild(node,xmlNewText((const xmlChar*)" "));
					xmlNode* v=xmlNewChild(node,NULL,(const xmlChar*)"",NULL);
					if(v==NULL)
						throw bad_format(node,"Error: plist Dictionary xml error on saving value");
					if(indentStr.size()>=perIndent.size())
						xmlAddChild(node,xmlNewText((const xmlChar*)("\n"+indentStr.substr(perIndent.size())).c_str()));
					else
						xmlAddChild(node,xmlNewText((const xmlChar*)("\n")));
					it->second->SaveXML(v);
				}
			}
		}
	}

	unsigned int Dictionary::getLongestKeyLen() const {
		size_t longest=0;
		for(Dictionary::dict_t::const_iterator it=dict.begin(); it!=dict.end(); ++it) {
			size_t cur=it->first.size();
			if(Dictionary* dp=dynamic_cast<Dictionary*>(it->second))
				cur+=dp->getLongestKeyLen()+1;
			longest=std::max(longest,cur);
		}
		return longest;
	}
	
	void Dictionary::addDictionaryListener(DictionaryListener* vl) {
		if(vl!=NULL) {
			if(dictionaryListeners==NULL)
				dictionaryListeners=new std::list<DictionaryListener*>;
			dictionaryListeners->push_back(vl);
		}
	}
		
	void Dictionary::removeDictionaryListener(DictionaryListener* vl) {
		if(dictionaryListeners==NULL)
			return;
		std::list<DictionaryListener*>::iterator it=find(dictionaryListeners->begin(),dictionaryListeners->end(),vl);
		if(it!=dictionaryListeners->end()) {
			dictionaryListeners->erase(it);
			if(dictionaryListeners->empty()) {
				delete dictionaryListeners;
				dictionaryListeners=NULL;
			}
		}
	}
	
	void Dictionary::fireEntryAdded(ObjectBase& val) {
		if(dictionaryListeners==NULL)
			return;
		std::list<DictionaryListener*>::iterator it=dictionaryListeners->begin();
		while(it!=dictionaryListeners->end()) {
			std::list<DictionaryListener*>::iterator cur=it++; //increment early in case the listener changes subscription
			(*cur)->plistDictionaryEntryAdded(*this,val);
		}
	}

	void Dictionary::fireEntryRemoved(ObjectBase& val) {
		if(dictionaryListeners==NULL)
			return;
		std::list<DictionaryListener*>::iterator it=dictionaryListeners->begin();
		while(it!=dictionaryListeners->end()) {
			std::list<DictionaryListener*>::iterator cur=it++; //increment early in case the listener changes subscription
			(*cur)->plistDictionaryEntryRemoved(*this,val);
		}
	}

	std::ostream& operator<<(std::ostream& os, const Dictionary& d) {
		unsigned int longest=std::max(d.getLongestKeyLen(),static_cast<unsigned int>(os.width()));
		for(Dictionary::dict_t::const_iterator it=d.dict.begin(); it!=d.dict.end(); ++it)
			if(Dictionary* dp=dynamic_cast<Dictionary*>(it->second)) {
				stringstream ss;
				ss << setw(longest-it->first.size()-1) << *dp;
				string line;
				for(getline(ss,line); ss; getline(ss,line))
					os << it->first << "." << line << endl;
			} else {
				os << left << setw(longest) << it->first << " = " << *it->second << endl;
			}
		return os;
	}
	
	PrimitiveBase::~PrimitiveBase() {
		delete primitiveListeners;
		primitiveListeners=NULL;
	}

	void PrimitiveBase::addPrimitiveListener(PrimitiveListener* vl) {
		if(vl!=NULL) {
			if(primitiveListeners==NULL)
				primitiveListeners=new std::list<PrimitiveListener*>;
			primitiveListeners->push_back(vl);
		}
	}
	void PrimitiveBase::removePrimitiveListener(PrimitiveListener* vl) {
		if(primitiveListeners==NULL)
			return;
		std::list<PrimitiveListener*>::iterator it=find(primitiveListeners->begin(),primitiveListeners->end(),vl);
		if(it!=primitiveListeners->end()) {
			primitiveListeners->erase(it);
			if(primitiveListeners->empty()) {
				delete primitiveListeners;
				primitiveListeners=NULL;
			}
		}
	}
	void PrimitiveBase::fireValueChanged() const {
		if(primitiveListeners==NULL)
			return;
		std::list<PrimitiveListener*>::const_iterator it=primitiveListeners->begin();
		while(it!=primitiveListeners->end()) {
			std::list<PrimitiveListener*>::const_iterator cur=it++; //increment early in case the listener changes subscription
			(*cur)->plistValueChanged(*this);
		}
	}
	
} //namespace plist

/*! @file
 * @brief 
 * @author Ethan Tira-Thompson (ejt) (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.6 $
 * $State: Exp $
 * $Date: 2005/07/27 05:58:24 $
 */
