//-*-c++-*-
#ifndef INCLUDED_LoadSave_h
#define INCLUDED_LoadSave_h

#include <string>
#include <sys/param.h>

#ifndef LOADSAVE_SWAPBYTES

#ifdef BYTE_ORDER
#  if BYTE_ORDER == BIG_ENDIAN
#    define LOADSAVE_SWAPBYTES 1
#  elif BYTE_ORDER == LITTLE_ENDIAN
#    define LOADSAVE_SWAPBYTES 0
#  else
#    warning unknown byte ordering for current platform, assuming no swap
#    define LOADSAVE_SWAPBYTES 0
#  endif
#else
//aperios distribution doesn't actually define BYTE_ORDER :(
//just as well, since we're using that byte order anyway
#  ifdef PLATFORM_APERIOS
//!On PLATFORM_APERIOS, we avoid swapping bytes, since it's likely to be the most CPU constrained platform
#    define LOADSAVE_SWAPBYTES 0
#  else
#    warning unknown byte ordering for current platform, assuming no swap
#    define LOADSAVE_SWAPBYTES 0
#  endif
#endif

#endif

//! Intended as an interface to allow easy and uniform file operations
/*!
	
	Generally, for usage, all you need to know is to call
	<tt>SaveFile("/path/to/file.ext")</tt> or
	<tt>SaveBuffer(membuffer)</tt> in order to have the class serialize
	itself, and the LoadFile() / LoadBuffer() in order to reload the
	data.
	
	So, when SaveFile() is called, it checks that it
	can open the specified file, and then calls SaveFileStream()
	with the open file.  This will then check getBinSize(), create a
	buffer of that size, and call SaveBuffer() to do the actual work of
	serialization into that buffer.  If SaveBuffer is successful (make
	sure your getBinSize() doesn't underestimate!) SaveFileStream()
	copies the buffer out to the file, and then finally, SaveFile() will
	close the file.
	
  This has the nice side effect that if you want to send the data over
  the network instead of a file, you can call SaveBuffer() directly
  and reuse the code.  However, if you have a lot of data to save, you
  can override the SaveFileStream() function as well, and save into
  the file directly.  So far I've only bothered to do that with
  only a few classes.
	
	The recommended style for using LoadSave in classes with inheritance
	is to have each subclass first call the superclass's implementation,
	and then save their own local data.  This compartmentalizes the data
	access and makes it easy to maintain - the code that serializes is
	right in with the code that defines the structure.  If you change
	one, it's easy to see where to change the other.  And protection
	between levels of inheritance is retained.  (This is why I say it's
	highly flexible/maintainable, but poor readability since the
	serialization is all broken up.)
	
	I also recommend putting a little string header at the beginning of
	each class's info.  This will allow polymorphism when loading files
	(you can look at the string and create the appropriate type) but
	also is handy for checking field alignment... it's a lot easier to
	tell how much you're offset within a string than to do the same with
	a stream of binary values.  Further, you can use the string as
	version information if you want to be backward compatible in future
	versions of your code.
	
	LoadSave provides a series of encode and decode functions for all
	the primitive types.  This will handle copying the value into the
	buffer or file, and can provide platform independence through byte
	swapping if needed (there's a compiler flag you can set for
	platforms that have the opposite byte order).  Most of these are
	pretty straightfoward - an int is just 4 bytes and so on.
	
	However, there's one caveat that I want to make sure to point out if
	you have to write parsing code in say, Java.  Strings are encoded by
	first storing an int to hold the string's length, then the string
	itself, and then a null character.  This adds 5 bytes to the length
	of any string, but makes loading the files much easier/faster - you
	can call string library functions directly on the buffer if it's
	already in memory since the string is null terminated, or can
	allocate memory to hold the string during loading from a file if
	needed because you'll know the size of the string before you get to
	it.
	
	Of course, the string stuff is transparent if you just use
	LoadSave's encode/decode functions to parse it.  But if that's not
	available (for instance if your receiver is in Java, there's a
	readLoadSaveString() in VisionListener.java if that will help:
	http://cvs.tekkotsu.org/cgi-bin/viewcvs.cgi/Tekkotsu/tools/mon/org/tekkotsu/mon/VisionListener.java?rev=1.6&content-type=text/vnd.viewcvs-markup
	@code
	public boolean _isConnected;

	public String readLoadSaveString(InputStream in) throws java.io.IOException {
		int creatorLen=readInt(in);
		if(!_isConnected) return ""; 
		String creator=new String(readBytes(in,creatorLen));
		if(!_isConnected) return "";
		if(readChar(in)!='\0')
			System.err.println("Misread LoadSave string? "+creator);
		return creator;
	}

	public int readInt(InputStream in) throws IOException {
		int read=0;
		int last=0;
		byte[] buf=new byte[4];
		while (read<4 && last>=0) { last=in.read(buf,read,4-read); read+=last; }
		if(last<0)
			_isConnected=false;
		return (b2i(buf[3])<<24) | (b2i(buf[2])<<16) |
					 (b2i(buf[1])<< 8) | b2i(buf[0]);
	}
  public byte[] readBytes(InputStream in, int bytes) throws IOException {
    byte[] ret=new byte[bytes];
    readBytes(ret, in, bytes);
    return ret;
  }
	public char readChar(InputStream in) throws IOException {
		return (char)in.read();
	}
	@endcode
*/
class LoadSave {
 public:
	//! This is the amount of extra space needed to store a string (int for len of string plus 1 for null term
	static const unsigned int stringpad=sizeof(unsigned int)+1;

	//!@name Constructors/Destructors
	/*! @brief constructor */
	LoadSave() {}
	virtual ~LoadSave(); //!< destructor
	//@}

	//! These are useful for sending the data across a network as well as to a file.\n
	//! These are the only ones that MUST be overridden, as the file
	//! ops can be based on calling these, tho feel free to override
	//! the file ops as well if speed or temp. memory is tight.
	//!@name Buffer Operations
	/*! @brief calculates space needed to save - if you can't precisely add up the size, overestimate and things will still work.
	 *  @return number of bytes read/written, 0 if error (or empty) */
	virtual unsigned int getBinSize() const =0;
	//! Load from a saved buffer
	/*! @param buf pointer to the memory where you should begin loading
	 *  @param len length of @a buf available (this isn't all yours, might be more stuff saved after yours)
	 *  @return the number of bytes actually used */
	virtual unsigned int LoadBuffer(const char buf[], unsigned int len)=0;
	//! Save to a given buffer
	/*! @param buf pointer to the memory where you should begin writing
	 *  @param len length of @a buf available.  (this isn't all yours, constrain yourself to what you returned in getBinSize() )
	 *  @return the number of bytes actually used */
	virtual unsigned int SaveBuffer(char buf[], unsigned int len) const =0;
	//@}

	//!These are called to load and save to files
	//!@name File Operations
	/*!@brief initiate opening of the specified file and loading/saving of all appropriate information.
	 * @param filename the file to load/save @return number of bytes read/written, 0 if error (or empty)*/
	virtual	unsigned int LoadFile(const char* filename);
	virtual unsigned int SaveFile(const char* filename) const;

	//!Used recursively on member objects once a file is already open - DON'T CLOSE the file in your overridden functions
	/*! @param f a pointer to the file to load
	 *  @warning could potentially be very inefficient if root-level objects
	 *  override LoadFile but leaf-level ones use this implementation, but
	 *  leaf-level ones won't even get this call unless you override the ones
	 *  above them - hence, this is all or nothing
	 *	@return number of bytes read, 0 if error (or empty) */
	virtual unsigned int LoadFileStream(FILE* f);
	//!Used recursively on member objects once a file is already open - DON'T CLOSE the file in your overridden functions
	/*! @param f a pointer to the file to save
	 *  @warning could potentially be very inefficient if root-level objects
	 *  override SaveFile but leaf-level ones use this implementation, but
	 *  leaf-level ones won't even get this call unless you override the ones
	 *  above them - hence, this is all or nothing
	 *	@return number of bytes written, 0 if error (or empty) */
	virtual unsigned int SaveFileStream(FILE* f) const;
	//@}

	//! Handy for checking results from functions which manipulate the buffer
	/*! @param res number of bytes used
	 *  @param buf pointer to address of current place in buffer, will be incremented by @a res bytes
	 *  @param len number of bytes remain between current place and end of buffer, will be decremented by @a res bytes
	 *  @param msg Error to display if res is less than or equal to zero
	 *  @return true if everything worked, false otherwise */
	static bool ChkAdvance(int res, const char** buf, unsigned int* len, const char* msg);

	//! Handy for checking results from functions which manipulate the buffer
	/*! This really should be rewritten to use variable number of arguments which can be passed
	 *  directly on to printf...
	 *  @param res number of bytes used
	 *  @param buf pointer to address of current place in buffer, will be incremented by @a res bytes
	 *  @param len number of bytes remain between current place and end of buffer, will be decremented by @a res bytes
	 *  @param msg Error to display if res is less than or equal to zero
	 *  @param arg1 An additional argument to be displayed (using %d in the @a msg); intended for things such as line number of error in file being read
	 *  @return true if everything worked, false otherwise */
	static bool ChkAdvance(int res, const char** buf, unsigned int* len, const char* msg, int arg1);

	//! These are for putting creator codes at the beginning of your data to check for sanity, just optional
	//!@name Creator Utilities

	/*!@brief Returns size of the creator code
	 * @param creator a string to use for the creator
	 * @return the size to leave for the creator code */
	virtual unsigned int creatorSize(const char creator[]) const { return strlen(creator)+stringpad; }
	//! Compares the creator code in the buffer to the one given
	/*!@param creator what the creator should be
	 * @param buf the buffer to check
	 * @param len the size remaining in the buffer
	 * @param isLoading set this to true if you want to output a warning if it doesn't match
	 * @return the number of bytes used by the creator, or 0 if it didn't match */
	virtual unsigned int checkCreator(const char* creator, const char buf[], unsigned int len, bool isLoading=true) const;
	//! Compares the creator code in the file to the one given, will attempt to reset the file position if fails (so you can check for one of several types)
	/*!@param creator what the creator should be
	 * @param f the file pointer to check
	 * @param isLoading set this to true if you want to output a warning if it doesn't match
	 * @return the number of bytes consumed by the creator code, or 0 if it didn't match */
	virtual unsigned int checkCreator(const char* creator, FILE* f, bool isLoading=true) const;
	//! Saves a creator code to a buffer
	/*!@param creator the string to use for the creator code
	 * @param buf the buffer to save the code into
	 * @param len the space available in the buffer
	 * @return the number of bytes consumed */
	virtual unsigned int saveCreator(const char* creator, char buf[], unsigned int len) const;
	//! Saves a creator code directly to a file
	/*!@param creator the string to use for the creator code
	 * @param f the file to save the code into
	 * @return the number of bytes consumed */
	virtual unsigned int saveCreator(const char* creator, FILE* f) const;
	//@}

	/* //if you want to have a default behavior template like this (look up template specialization) (i thought i needed this, nevermind)
		template<class T> inline static unsigned int encode(const T x, char buf[], unsigned int cap) { cout << "*** WARNING attempted to encode non-primitive and non-LoadSave" << endl; return 0; }
		template<class T> inline static unsigned int decode(T& x, const char buf[], unsigned int cap) { cout << "*** WARNING attempted to decode non-primitive and non-LoadSave" << endl; return 0; }
		template<class T> inline static unsigned int encode(const T x, FILE* f) { cout << "*** WARNING attempted to encode non-primitive and non-LoadSave" << endl; return 0; }
		template<class T> inline static unsigned int decode(T& x, FILE* f) { cout << "*** WARNING attempted to decode non-primitive and non-LoadSave" << endl; return 0; }
	*/

	//!encode/decode cross-platform compatable (byte order consistancy)
	//!@name Encode/Decode Utils
	/*!@brief encode or decode with byte order consistancy*/
	inline static unsigned int encode(const LoadSave& x, char buf[], unsigned int cap)  { return x.SaveBuffer(buf,cap); }
	inline static unsigned int decode(LoadSave& x, const char buf[], unsigned int cap)  { return x.LoadBuffer(buf,cap); }
	inline static unsigned int encode(const LoadSave& x, FILE* f)                  { return x.SaveFileStream(f); }
	inline static unsigned int decode(LoadSave& x, FILE* f)                        { return x.LoadFileStream(f); }
	
#if LOADSAVE_SWAPBYTES
	
	inline static unsigned int encode(const double x, char buf[], unsigned int cap)          { if(cap<sizeof(x)) return 0; byteswap(*(double*)buf,x); return sizeof(x); }
	inline static unsigned int decode(double& x, const char buf[], unsigned int cap)         { if(cap<sizeof(x)) return 0; byteswap(x,*(const double*)buf); return sizeof(x);}
	inline static unsigned int encode(const double x, FILE* f)                          { double t=0; byteswap(t,x); return sizeof(x)*fwrite(&t,sizeof(x),1,f); }
	inline static unsigned int decode(double& x, FILE* f)                               { double t=0; if(fread(&t,sizeof(x),1,f)==0) return 0; byteswap(x,t); return sizeof(x);}
	
	inline static unsigned int encode(const float x, char buf[], unsigned int cap)           { if(cap<sizeof(x)) return 0; byteswap(*(float*)buf,x); return sizeof(x); }
	inline static unsigned int decode(float& x, const char buf[], unsigned int cap)          { if(cap<sizeof(x)) return 0; byteswap(x,*(const float*)buf); return sizeof(x);}
	inline static unsigned int encode(const float x, FILE* f)                           { float t=0; byteswap(t,x); return sizeof(x)*fwrite(&t,sizeof(x),1,f); }
	inline static unsigned int decode(float& x, FILE* f)                                { float t=0; if(fread(&t,sizeof(x),1,f)==0) return 0; byteswap(x,t); return sizeof(x);}
	
	inline static unsigned int encode(const long x, char buf[], unsigned int cap)            { if(cap<sizeof(x)) return 0; byteswap(*(long*)buf,x); return sizeof(x); }
	inline static unsigned int decode(long& x, const char buf[], unsigned int cap)           { if(cap<sizeof(x)) return 0; byteswap(x,*(const long*)buf); return sizeof(x);}
	inline static unsigned int encode(const long x, FILE* f)                            { long t=0; byteswap(t,x); return sizeof(x)*fwrite(&t,sizeof(x),1,f); }
	inline static unsigned int decode(long& x, FILE* f)                                 { long t=0; if(fread(&t,sizeof(x),1,f)==0) return 0; byteswap(x,t); return sizeof(x);}
	inline static unsigned int encode(const unsigned long x, char buf[], unsigned int cap)   { if(cap<sizeof(x)) return 0; byteswap(*(unsigned long*)buf,x); return sizeof(x); }
	inline static unsigned int decode(unsigned long& x, const char buf[], unsigned int cap)  { if(cap<sizeof(x)) return 0; byteswap(x,*(const unsigned long*)buf); return sizeof(x);}
	inline static unsigned int encode(const unsigned long x, FILE* f)                   { unsigned long t=0; byteswap(t,x); return sizeof(x)*fwrite(&t,sizeof(x),1,f); }
	inline static unsigned int decode(unsigned long& x, FILE* f)                        { unsigned long t=0; if(fread(&t,sizeof(x),1,f)==0) return 0; byteswap(x,t); return sizeof(x);}
	inline static unsigned int encode(const int x, char buf[], unsigned int cap)             { if(cap<sizeof(x)) return 0; byteswap(*(int*)buf,x); return sizeof(x); }
	inline static unsigned int decode(int& x, const char buf[], unsigned int cap)            { if(cap<sizeof(x)) return 0; byteswap(x,*(const int*)buf); return sizeof(x);}
	inline static unsigned int encode(const int x, FILE* f)                             { int t=0; byteswap(t,x); return sizeof(x)*fwrite(&t,sizeof(x),1,f); }
	inline static unsigned int decode(int& x, FILE* f)                                  { int t=0; if(fread(&t,sizeof(x),1,f)==0) return 0; byteswap(x,t); return sizeof(x);}
	inline static unsigned int encode(const unsigned int x, char buf[], unsigned int cap)    { if(cap<sizeof(x)) return 0; byteswap(*(unsigned int*)buf,x); return sizeof(x); }
	inline static unsigned int decode(unsigned int& x, const char buf[], unsigned int cap)   { if(cap<sizeof(x)) return 0; byteswap(x,*(const unsigned int*)buf); return sizeof(x);}
	inline static unsigned int encode(const unsigned int x, FILE* f)                    { unsigned int t=0; byteswap(t,x); return sizeof(x)*fwrite(&t,sizeof(x),1,f); }
	inline static unsigned int decode(unsigned int& x, FILE* f)                         { unsigned int t=0; if(fread(&t,sizeof(x),1,f)==0) return 0; byteswap(x,t); return sizeof(x);}
	
	inline static unsigned int encode(const short x, char buf[], unsigned int cap)           { if(cap<sizeof(x)) return 0; byteswap(*(short*)buf,x); return sizeof(x); }
	inline static unsigned int decode(short& x, const char buf[], unsigned int cap)          { if(cap<sizeof(x)) return 0; byteswap(x,*(const short*)buf); return sizeof(x);}
	inline static unsigned int encode(const short x, FILE* f)                           { short t; byteswap(t,x); return sizeof(x)*fwrite(&t,sizeof(x),1,f); }
	inline static unsigned int decode(short& x, FILE* f)                                { short t=0; if(fread(&t,sizeof(x),1,f)==0) return 0; byteswap(x,t); return sizeof(x);}
	inline static unsigned int encode(const unsigned short x, char buf[], unsigned int cap)  { if(cap<sizeof(x)) return 0; byteswap(*(unsigned short*)buf,x); return sizeof(x); }
	inline static unsigned int decode(unsigned short& x, const char buf[], unsigned int cap) { if(cap<sizeof(x)) return 0; byteswap(x,*(const unsigned short*)buf); return sizeof(x);}
	inline static unsigned int encode(const unsigned short x, FILE* f)                  { unsigned short t; byteswap(t,x); return sizeof(x)*fwrite(&t,sizeof(x),1,f); }
	inline static unsigned int decode(unsigned short& x, FILE* f)                       { unsigned short t=0; if(fread(&t,sizeof(x),1,f)==0) return 0; byteswap(x,t); return sizeof(x);}

#else
	
	inline static unsigned int encode(const double x, char buf[], unsigned int cap)         { if(cap<sizeof(x)) return 0; memcpy(buf,&x,sizeof(x)); return sizeof(x); }
	inline static unsigned int decode(double& x, const char buf[], unsigned int cap)        { if(cap<sizeof(x)) return 0; memcpy(&x,buf,sizeof(x)); return sizeof(x); }
	inline static unsigned int encode(const double x, FILE* f)                         { return sizeof(x)*fwrite(&x,sizeof(x),1,f); }
	inline static unsigned int decode(double& x, FILE* f)                              { return sizeof(x)*fread(&x,sizeof(x),1,f); }
	
	inline static unsigned int encode(const float x, char buf[], unsigned int cap)          { if(cap<sizeof(x)) return 0; memcpy(buf,&x,sizeof(x)); return sizeof(x); }
	inline static unsigned int decode(float& x, const char buf[], unsigned int cap)         { if(cap<sizeof(x)) return 0; memcpy(&x,buf,sizeof(x)); return sizeof(x); }
	inline static unsigned int encode(const float x, FILE* f)                          { return sizeof(x)*fwrite(&x,sizeof(x),1,f); }
	inline static unsigned int decode(float& x, FILE* f)                               { return sizeof(x)*fread(&x,sizeof(x),1,f); }

	inline static unsigned int encode(const long x, char buf[], unsigned int cap)           { if(cap<sizeof(x)) return 0; memcpy(buf,&x,sizeof(x)); return sizeof(x); }
	inline static unsigned int decode(long& x, const char buf[], unsigned int cap)          { if(cap<sizeof(x)) return 0; memcpy(&x,buf,sizeof(x)); return sizeof(x); }
	inline static unsigned int encode(const long x, FILE* f)                           { return sizeof(x)*fwrite(&x,sizeof(x),1,f); }
	inline static unsigned int decode(long& x, FILE* f)                                { return sizeof(x)*fread(&x,sizeof(x),1,f); }
	inline static unsigned int encode(const unsigned long x, char buf[], unsigned int cap)  { if(cap<sizeof(x)) return 0; memcpy(buf,&x,sizeof(x)); return sizeof(x); }
	inline static unsigned int decode(unsigned long& x, const char buf[], unsigned int cap) { if(cap<sizeof(x)) return 0; memcpy(&x,buf,sizeof(x)); return sizeof(x); }
	inline static unsigned int encode(const unsigned long x, FILE* f)                  { return sizeof(x)*fwrite(&x,sizeof(x),1,f); }
	inline static unsigned int decode(unsigned long& x, FILE* f)                       { return sizeof(x)*fread(&x,sizeof(x),1,f); }
	inline static unsigned int encode(const int x, char buf[], unsigned int cap)            { if(cap<sizeof(x)) return 0; memcpy(buf,&x,sizeof(x)); return sizeof(x); }
	inline static unsigned int decode(int& x, const char buf[], unsigned int cap)           { if(cap<sizeof(x)) return 0; memcpy(&x,buf,sizeof(x)); return sizeof(x); }
	inline static unsigned int encode(const int x, FILE* f)                            { return sizeof(x)*fwrite(&x,sizeof(x),1,f); }
	inline static unsigned int decode(int& x, FILE* f)                                 { return sizeof(x)*fread(&x,sizeof(x),1,f); }
	inline static unsigned int encode(const unsigned int x, char buf[], unsigned int cap)   { if(cap<sizeof(x)) return 0; memcpy(buf,&x,sizeof(x)); return sizeof(x); }
	inline static unsigned int decode(unsigned int& x, const char buf[], unsigned int cap)  { if(cap<sizeof(x)) return 0; memcpy(&x,buf,sizeof(x)); return sizeof(x); }
	inline static unsigned int encode(const unsigned int x, FILE* f)                   { return sizeof(x)*fwrite(&x,sizeof(x),1,f); }
	inline static unsigned int decode(unsigned int& x, FILE* f)                        { return sizeof(x)*fread(&x,sizeof(x),1,f); }
	
	inline static unsigned int encode(const short x, char buf[], unsigned int cap)          { if(cap<sizeof(x)) return 0; memcpy(buf,&x,sizeof(x)); return sizeof(x); }
	inline static unsigned int decode(short& x, const char buf[], unsigned int cap)         { if(cap<sizeof(x)) return 0; memcpy(&x,buf,sizeof(x)); return sizeof(x); }
	inline static unsigned int encode(const short x, FILE* f)                          { return sizeof(x)*fwrite(&x,sizeof(x),1,f); }
	inline static unsigned int decode(short& x, FILE* f)                               { return sizeof(x)*fread(&x,sizeof(x),1,f); }
	inline static unsigned int encode(const unsigned short x, char buf[], unsigned int cap) { if(cap<sizeof(x)) return 0; memcpy(buf,&x,sizeof(x)); return sizeof(x); }
	inline static unsigned int decode(unsigned short& x, const char buf[], unsigned int cap){ if(cap<sizeof(x)) return 0; memcpy(&x,buf,sizeof(x)); return sizeof(x); }
	inline static unsigned int encode(const unsigned short x, FILE* f)                 { return sizeof(x)*fwrite(&x,sizeof(x),1,f); }
	inline static unsigned int decode(unsigned short& x, FILE* f)                      { return sizeof(x)*fread(&x,sizeof(x),1,f); }

#endif //end of big/little endian differences
	
	inline static unsigned int encode(const std::string& x, char buf[], unsigned int cap)  { if(cap<sizeof(unsigned int)+x.size()+1) return 0; memcpy(buf+encode(x.size(),buf,cap),x.c_str(),x.size()+1); return x.size()+stringpad; }
	inline static unsigned int decode(std::string& x, const char buf[], unsigned int cap)  { if(cap<sizeof(unsigned int)) return 0; unsigned int sz=0; decode(sz,buf,cap); if(cap<sizeof(unsigned int)+sz+1) return 0; x.assign(buf+sizeof(unsigned int),sz); return x.size()+stringpad; }
	inline static unsigned int encode(const std::string& x, FILE* f)                  { encode(x.size(),f); return sizeof(unsigned int)+fwrite(x.c_str(),1,sizeof(x)+1,f); }
	inline static unsigned int decode(std::string& x, FILE* f)                        { unsigned int sz=0; decode(sz,f); char *tmp=new char[sz+1]; if(fread(tmp,1,sz+1,f)!=sz+1) { delete [] tmp; return 0; } x.assign(tmp,sz); delete [] tmp; return sz+stringpad; }
	
	inline static unsigned int encode(const char* x, char buf[], unsigned int cap)      { unsigned int sz=strlen(x); if(cap<sizeof(unsigned int)+sz+1) return 0; memcpy(buf+encode(sz,buf,cap),x,sz+1); return sz+stringpad; }
	inline static unsigned int decode(char*& x, const char buf[], unsigned int cap)     { if(cap<sizeof(unsigned int)) return 0; unsigned int sz=0; decode(sz,buf,cap); if(cap<sizeof(unsigned int)+sz+1) return 0; x=new char[sz+1]; strncpy(x,buf+sizeof(unsigned int),sz+1); return sz+stringpad; }
	inline static unsigned int encode(const char* x, FILE* f)                      { unsigned int sz=strlen(x); encode(sz,f); return sizeof(unsigned int)+fwrite(x,1,sz+1,f); }
	inline static unsigned int decode(char*& x, FILE* f)                           { unsigned int sz=0; decode(sz,f); x=new char[sz+1]; if(fread(x,1,sz+1,f)!=sz+1) { delete [] x; x=NULL; return 0; } return sz+stringpad; }
	
	inline static unsigned int encode(const char x, char buf[], unsigned int cap)          { if(cap<sizeof(x)) return 0; buf[0]=x; return sizeof(x); }
	inline static unsigned int decode(char& x, const char buf[], unsigned int cap)         { if(cap<sizeof(x)) return 0; x=buf[0]; return sizeof(x);}
	inline static unsigned int encode(const char x, FILE* f)                          { return sizeof(x)*fwrite(&x,sizeof(x),1,f); }
	inline static unsigned int decode(char& x, FILE* f)                               { return sizeof(x)*fread(&x,sizeof(x),1,f); }
	inline static unsigned int encode(const unsigned char x, char buf[], unsigned int cap) { if(cap<sizeof(x)) return 0; buf[0]=(char)x; return sizeof(x); }
	inline static unsigned int decode(unsigned char& x, const char buf[], unsigned int cap){ if(cap<sizeof(x)) return 0; x=(unsigned char)buf[0]; return sizeof(x);}
	inline static unsigned int encode(const unsigned char x, FILE* f)                 { return sizeof(x)*fwrite(&x,sizeof(x),1,f); }
	inline static unsigned int decode(unsigned char& x, FILE* f)                      { return sizeof(x)*fread(&x,sizeof(x),1,f); }
	
	inline static unsigned int encode(const bool x, char buf[], unsigned int cap)          { if(cap<sizeof(char)) return 0; buf[0]=(char)(x?1:0); return sizeof(char); }
	inline static unsigned int decode(bool& x, const char buf[], unsigned int cap)         { if(cap<sizeof(char)) return 0; x=(buf[0]!=(char)0); return sizeof(char);}
	inline static unsigned int encode(const bool x, FILE* f)                          { char t=(char)(x?1:0); return sizeof(char)*fwrite(&t,sizeof(char),1,f); }
	inline static unsigned int decode(bool& x, FILE* f)                               { char t='\0'; fread(&t,sizeof(char),1,f); x=(t!=(char)0); return sizeof(char); }
	//@}	
protected:
	//!templated function to swap byte ordering, should allow compiler to unroll the loop; warning don't use this if src==dst!!!
	template<class T> inline static void byteswap(T& dstc, const T& srcc) {
		char* dst=(char*)&dstc;
		const char* src=(const char*)&srcc;
		for(unsigned int i=0; i<sizeof(T); i++)
			dst[sizeof(T)-1-i]=src[i];
	}
};


/*! @file
 * @brief Describes LoadSave, inherit from this to use a standard interface for loading and saving
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.13 $
 * $State: Exp $
 * $Date: 2005/08/07 04:11:04 $
 */

#endif
