//-*-c++-*-
#ifndef INCLUDED_Config_h
#define INCLUDED_Config_h

#include <vector>
#include <string>
#include "RobotInfo.h"
extern "C" {
#include <jpeglib.h>
}

//!provides global access to system configuration information
class Config {
 public:
	//!constructor
	Config()
		: wireless(), vision(), main(), behaviors(), controller(), motion(this),
		sound(this), fsRoot()
		{setFileSystemRoot("");}
	//!constructor
	Config(const std::string& filename)
		: wireless(), vision(), main(), behaviors(), controller(), motion(this),
		sound(this), fsRoot()
		{ setFileSystemRoot(""); readConfig(filename); }
	//!destructor
	~Config() {}

	void setFileSystemRoot(const std::string& fsr); //!< sets #fsRoot
	const std::string& getFileSystemRoot() const { return fsRoot; } //!< returns #fsRoot
	std::string portPath(const std::string& path) const; //!< returns a portable version of @a path which should be usable on either the simulator or the robot
	
	//!section IDs
	enum section_t {
		sec_wireless=0,  //!< denotes wireless section of config file
		sec_vision,      //!< denotes vision section of config file
		sec_main,        //!< denotes main section of config file, for misc. settings
		sec_behaviors,   //!< denotes behaviors section of config file
		sec_controller,  //!< denotes controller section of config file
		sec_motion,      //!< denotes motion section of config file
		sec_sound,       //!< denotes sound section of config file
		sec_invalid      //!< denotes an invalid section of config file
	};

	//!wirless information
	struct wireless_config {
		int id; //!< id number (in case you have more than one AIBO)
      
		wireless_config () : id(1) {} //!< constructor
	} wireless;

	//!vision information
	struct vision_config {
		int white_balance;    //!< white balance
		int gain;             //!< gain
		int shutter_speed;    //!< shutter speed
		int resolution;       //!< resolution
		std::vector<std::string> thresh;      //!< thresholds
		char colors[50];      //!< colors
		int rawcam_port;      //!< port to send raw frames on
		int rawcam_transport; //!< transport protocol: 0 for udp, 1 for tcp
		unsigned int rawcam_interval;  //!< interval between images: 0 for fast-as-possible, 100 for 10 FPS, 200 for 5 FPS, etc.
		int rle_port;         //!< port to send RLE frames on
		int rle_transport;    //!< transport protocol: 0 for udp, 1 for tcp
		unsigned int rle_interval;     //!< interval between images: 0 for fast-as-possible, 100 for 10 FPS, 200 for 5 FPS, etc.
		int region_port;      //!< port to send Region information on
		int region_transport; //!< transport protocol: 0 for udp, 1 for tcp
		int obj_port;         //!< port to send object info on
		bool restore_image;   //!< if true, replaces pixels holding image info with actual image pixels (as much as possible anyway)
		J_DCT_METHOD jpeg_dct_method;  //!< pick between dct methods for jpeg compression
		float aspectRatio;    //!< ratio of width to height (x_res/y_res); this is *not* read from configuration file, but set from most recent camera image (or RobotInfo namespace values if no camera image has been received)
		float x_range;        //!< range of values for the x axis when using generalized coordinates; this is *not* read from configuration file, but set from most recent camera image (or RobotInfo namespace values if no camera image has been received)
		float y_range;        //!< range of values for the x axis when using generalized coordinates; this is *not* read from configuration file, but set from most recent camera image (or RobotInfo namespace values if no camera image has been received)

		//! type of information to send, stored in Config::vision_config::rawcam_encoding
		enum encoding_t {
			ENCODE_COLOR, //!< send Y, U, and V channels
			ENCODE_SINGLE_CHANNEL, //!< send only a single channel (which channel to send is stored in Config::vision_config::rawcam_channel) This is also used for all seg cam images
		};
		encoding_t rawcam_encoding; //!< holds whether to send color or single channel
		int rawcam_channel;    //!< RawCameraGenerator::channel_id_t, if raw_encoding is single channel, this holds the channel to send (computed from rawcam_encoding, not set directly)
		
		//! compression format to use, stored in Config::vision_config::rawcam_compression
		enum compression_t {
			COMPRESS_NONE, //!< no compression (other than subsampling)
			COMPRESS_JPEG, //!< JPEG compression
			COMPRESS_RLE   //!< RLE compression
		};
		compression_t rawcam_compression;//!< holds whether to send jpeg compression

		int rawcam_compress_quality;//!< 0-100, compression quality (currently only used by jpeg)
		int rawcam_y_skip;     //!< resolution level to transmit y channel at
		int rawcam_uv_skip;    //!< resolution level to transmit uv channel at
		int rlecam_skip;       //!< resolution level to transmit segmented images at
		int rlecam_channel;    //!< channel of RLEGenerator to send
		compression_t rlecam_compression; //!< what compression to use on the segmented image
    int regioncam_skip;    //!< resolution level to transmit segmented images at 

		//!These values represent a "Plumb Bob" model introduced by Brown in 1966
		/*!Lens Distortion for Close-Range Photogrammetry -  D.C. Brown, Photometric Engineering, pages 855-866, Vol. 37, No. 8, 1971.
		 * 
		 * Can be computated by 'Camera Calibration Toolbox for Matlab', by Jean-Yves Bouguet:
		 * http://www.vision.caltech.edu/bouguetj/calib_doc/ */
		//!@name Intrinsic Parameters
		float focal_len_x; //!< focal length of camera, in pixels, K11
		float focal_len_y; //!< focal length of camera, in pixels, K22
		float principle_point_x; //!< center of optical projection, K13
		float principle_point_y; //!< center of optical projection, K23
		float skew; //!< CCD skew, K12 = skew*focal_len_x
		float kc1_r2; //!< r-squared radial distortion
		float kc2_r4; //!< r to the 4 radial distortion
		float kc5_r6; //!< r to the 6 radial distortion
		float kc3_tan1; //!< first tangential correction term
		float kc4_tan2; //!< second tangential correctiont term
		unsigned int calibration_res_x; // x resolution of images used during calibration
		unsigned int calibration_res_y; // y resolution of images used during calibration

		//!provides a ray from camera through pixel in image; where possible, use computePixel for better accuracy (i.e. try to always move from world to camera instead of the other way around)
		/*! We can't undo some terms of the distortion model -- this is an estimate.
		 *  @param[in] x x position in range [-1,1]
		 *  @param[in] y y position in range [-1,1]
		 *  @param[out] r_x x value of the ray
		 *  @param[out] r_y y value of the ray
		 *  @param[out] r_z z value of the ray (always 1) */
		void computeRay(float x, float y, float& r_x, float& r_y, float& r_z) {
			x=(x+1)*calibration_res_x/2;
			y=(y+1)*calibration_res_y/2;
			float yd=(y-principle_point_y)/focal_len_y;
			float xd=(x-principle_point_x)/focal_len_x-skew*yd;
			float r2=xd*xd+yd*yd; //estimate of original
			float radial=(1 + kc1_r2*r2 + kc2_r4*r2*r2 + kc5_r6*r2*r2*r2);
			r_x=(xd - 2*kc3_tan1*x*y - kc4_tan2*(r2+2*x*x))/radial; //estimating tangential component
			r_y=(yd - kc3_tan1*(r2+2*y*y) - 2*kc4_tan2*x*y)/radial; //estimating tangential component
			r_z=1;
		}
      
		//!provides a pixel hit in image by a ray going through the camera frame
		/*! Hopefully we'll eventually upgrade this to account for lens distortion
		 *  @param[in] r_x x value of the ray
		 *  @param[in] r_y y value of the ray
		 *  @param[in] r_z z value of the ray
		 *  @param[out] x x position in range [-1,1]
		 *  @param[out] y y position in range [-1,1] */
		void computePixel(float r_x, float r_y, float r_z, float& x, float& y) {
			if(r_z==0) {
				x=y=0;
				return;
			}
			r_x/=r_z;
			r_y/=r_z;
			float r2 = r_x*r_x + r_y*r_y; //'r2' for 'radius squared', not 'ray number 2'
			float radial=(1 + kc1_r2*r2 + kc2_r4*r2*r2 + kc5_r6*r2*r2*r2);
			float xd = radial*r_x + 2*kc3_tan1*r_x*r_y + kc4_tan2*(r2+2*r_x*r_x);
			float yd = radial*r_y + kc3_tan1*(r2+2*r_y*r_y) + 2*kc4_tan2*r_x*r_y;
			x=focal_len_x*(xd+skew*yd)+principle_point_x;
			y=focal_len_y*yd+principle_point_y;
			x=2*x/calibration_res_x-1;
			y=2*y/calibration_res_y-1;
		}
		//@}
		
		//!constructor
		vision_config()
			: white_balance(3), gain(2), shutter_speed(2), resolution(2),
				thresh(), colors(), rawcam_port(10011), rawcam_transport(0), rawcam_interval(0),
				rle_port(10012), rle_transport(0), rle_interval(0), region_port(0), region_transport(0), obj_port(0), restore_image(true), 
				jpeg_dct_method(JDCT_IFAST), aspectRatio(CameraResolutionX/(float)CameraResolutionY),
				x_range(aspectRatio>1?1:aspectRatio), y_range(aspectRatio>1?1/aspectRatio:1), 
				rawcam_encoding(ENCODE_COLOR), rawcam_channel(0),
				rawcam_compression(COMPRESS_NONE), rawcam_compress_quality(75), rawcam_y_skip(0),
				rawcam_uv_skip(0), rlecam_skip(1), rlecam_channel(0), rlecam_compression(COMPRESS_RLE), regioncam_skip(1),
				focal_len_x(CameraResolutionX),focal_len_y(CameraResolutionX),principle_point_x(CameraResolutionX/2),
				principle_point_y(CameraResolutionY/2),skew(0),kc1_r2(0),kc2_r4(0),kc5_r6(0),kc3_tan1(0),kc4_tan2(0),
				calibration_res_x(CameraResolutionX), calibration_res_y(CameraResolutionY)
		{}
	} vision;
	
	//!core functionality information
	struct main_config {
		bool seed_rng;     //!< if true, will call srand with timestamp data, mangled by current sensor data
		int console_port;  //!< port to send/receive "console" information on (separate from system console)
		int stderr_port;   //!< port to send error information to
		int error_level;   //!< controls amount of info to error port
		int debug_level;   //!< controls amount of debug info
		int verbose_level; //!< controls verbosity of info
		int wsjoints_port; //!< port to send joint positions on
		int wspids_port;   //!< port to send pid info on
		int headControl_port;	   //!< port for receiving head commands
		int walkControl_port;	   //!< port for receiving walk commands
		int estopControl_port;	   //!< port for receiving walk commands
		int stewart_port;  //!< port for running a stewart platform style of control
		int aibo3d_port;   //!< port for send/receive of joint positions from Aibo 3D GUI
		int wmmonitor_port; //!< port for monitoring Watchable Memory
		bool use_VT100;    //!< if true, enables VT100 console codes (currently only in Controller menus - 1.3)
		unsigned int worldState_interval; //!< time (in milliseconds) to wait between sending WorldState updates over wireless
		//!constructor
		main_config()
			: seed_rng(true), console_port(10001), stderr_port(10002), error_level(0), debug_level(0),
				verbose_level(0),wsjoints_port(10031),wspids_port(10032),headControl_port(10052),
				walkControl_port(10050),estopControl_port(10053),stewart_port(10055),aibo3d_port(10051),
				wmmonitor_port(10061), use_VT100(true), worldState_interval(0)
		{ }
	} main;

	//!placeholder
	struct behaviors_config {
		unsigned int flash_bytes; //!< how many bytes of the IP to flash
		bool flash_on_start;      //!< whether or not to trigger flashing when initially started
		behaviors_config() : flash_bytes(4), flash_on_start(true) {} //!< constructor
	} behaviors;
    
	//!controller information
	struct controller_config {
		int gui_port;        //!< port to listen for the GUI to connect to aibo on
		char select_snd[50]; //!< sound file to use for "select" action
		char next_snd[50];   //!< sound file to use for "next" action
		char prev_snd[50];   //!< sound file to use for "prev" action
		char read_snd[50];   //!< sound file to use for "read from std-in" action
		char cancel_snd[50]; //!< sound file to use for "cancel" action
		char error_snd[50]; //!< sound file to use to signal errors

		//!constructor
		controller_config() : gui_port(10020) {
			select_snd[0]=next_snd[0]=prev_snd[0]=read_snd[0]=cancel_snd[0]=error_snd[0]='\0';
		}
	} controller;
    
	//!motion information
	struct motion_config {
		Config* thisconfig;  //!<pointer back to the containing config object
		std::string root;       //!< path on memory stick to "motion" files - for instance, position (.pos) and motion sequence (.mot)
		std::string walk;       //!< the walk parameter file to load by default for new WalkMC's
		std::string kinematics;  //!< the kinematics description file to load
		std::vector<std::string> kinematic_chains; //!< list of chains to load from #kinematics
		float calibration[NumPIDJoints]; //!< multiplier from desired to command for PID joints
		char estop_on_snd[50];  //!< sound file to use when e-stop turned on
		char estop_off_snd[50]; //!< sound file to use when e-stop turned off
		float max_head_tilt_speed; //!< max speed for the head joints, used by HeadPointerMC; rad/s
		float max_head_pan_speed; //!< max speed for the head joints, used by HeadPointerMC; rad/s
		float max_head_roll_speed; //!< max speed for the head joints, used by HeadPointerMC; rad/s
		bool inf_walk_accel; //!< if true, walks should attempt to switch directions immediately; otherwise they should do some kind of software acceleration to more smoothly switch direction
		int console_port;  //!< port to send/receive "console" information on (separate from system console)
		int stderr_port;   //!< port to send error information to

		//!returns an absolute path if @a is relative (to root), otherwise just @a name
		std::string makePath(const std::string& name) { 
			if(name[0]=='/')
				return thisconfig->portPath(name);
			if(root[root.size()-1]=='/')
				return thisconfig->portPath(root+name);
			else
				return thisconfig->portPath(root+"/"+name);
		}

		//!constructor
		motion_config(Config* c)
			: thisconfig(c), root(), walk(), kinematics(), kinematic_chains(), max_head_tilt_speed(0),
				max_head_pan_speed(0), max_head_roll_speed(0), inf_walk_accel(false), console_port(10003), stderr_port(10004)
		{
			estop_on_snd[0]=estop_off_snd[0]='\0';
			for(unsigned int i=0; i<NumPIDJoints; i++)
				calibration[i]=1;
		}
	private:
		motion_config(const motion_config&); //!< don't call
		motion_config& operator=(const motion_config&); //!< don't call
	} motion;

	//!sound information
	struct sound_config {
		Config* thisconfig;  //!<pointer back to the containing config object
		std::string root;         //!< path to sound clips
		unsigned int volume;      //!< volume in decibels - the value is interpreted as a signed short, where 0 is full volume, 0x8000 is mute
		unsigned int sample_rate; //!< sample rate to send to system, currently only 8000 or 16000 supported
		unsigned int sample_bits; //!< sample bit depth, either 8 or 16
		std::vector<std::string> preload; //!< list of sounds to preload at boot
			
		//!returns an absolute path if @a is relative (to root), otherwise just @a name
		std::string makePath(const std::string& name) { 
			if(name[0]=='/')
				return thisconfig->portPath(name);
			if(root[root.size()-1]=='/')
				return thisconfig->portPath(root+name);
			else
				return thisconfig->portPath(root+"/"+name);
		}
		
		//! audio streaming configuration
		struct streaming_config {
			unsigned int mic_port;        //!< port for streaming microphone samples
			unsigned int mic_sample_rate; //!< sample rate from the microphone
			unsigned int mic_sample_bits; //!< sample bit depth from the microphone (either 8 or 16)
			bool mic_stereo; //!< whether to stream stereo or mono from the microphone
      
			unsigned int speaker_port;    //!< port for streaming speaker samples
			unsigned int speaker_frame_length; //!< length of frame sent to the speaker (ms)
			unsigned int speaker_max_delay; //!< maximum delay (ms) during playback

			//! constructor
			streaming_config() : mic_port(10070), mic_sample_rate(16000),
			mic_sample_bits(16), mic_stereo(true),
			speaker_port(10071), speaker_frame_length(64),
			speaker_max_delay(1000) {}
		} streaming;

		//!constructor
		sound_config(Config* c) : thisconfig(c), root(), volume(0xF600), sample_rate(0), sample_bits(0), preload(), streaming() {}
	private:
		sound_config(const sound_config&); //!< don't call
		sound_config& operator=(const sound_config&); //!< don't call
	} sound;

	//! call this function when it's time to read the configuration file
	void readConfig(const std::string& filename);
	//! returns the section structure corresponding to the section name given
	section_t parseSection(const char* key);
	//! pass the section, item name string, item value string - sets the value and returns pointer to the item changed
	void* setValue(section_t section, const char *key, const char *value, bool updated=false);


protected:
	//! returns true if pattern matches model - pattern may have up to 1 '*', case insensitive
	bool matchNoCase(const std::string& model, const std::string& pattern);

	//! returns bool value corresponding to a @a value of "t", "f", "true", "false", "y", "n", "yes", "no", or zero/nonzero number
	static bool extractBool(const char* value);
	
	//! a prefix representing the file system root, usually indicating the robot's storage root.
	/*! When running in the simulator, this is used to pretend that a subdirectory in the project folder (e.g. 'ms') is the root file system */
	std::string fsRoot;
};

//!allows global access to current settings
extern Config* config;

/*! @file
 * @brief Describes Config, which provides global access to system configuration information
 * @author alokl (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.49 $
 * $State: Exp $
 * $Date: 2005/08/07 04:11:03 $
 */

#endif
