#include "PostureMC.h"
#include "Shared/Config.h"
#include "MotionManager.h"
#include "Shared/WorldState.h"

PostureMC& PostureMC::setDirty(bool d/*=true*/) {
	dirty=d;
	targetReached=false;
	for(unsigned int i=0; i<NumOutputs; i++)
		curPositions[i]=motman->getOutputCmd(i).value; //not state->outputs[i]; - see function documentation
	return *this;
} 

void PostureMC::defaultMaxSpeed(float x/*=1*/) {
	for(unsigned int i=0; i<NumOutputs; i++)
		maxSpeeds[i]=MaxOutputSpeed[i]*FrameTime*x; //MaxOutputsSpeed is rad/ms
	//respect the config values for the neck joints (which are stored as rad/sec)
	maxSpeeds[HeadOffset+TiltOffset]=config->motion.max_head_tilt_speed*FrameTime*x/1000; 
	maxSpeeds[HeadOffset+PanOffset]=config->motion.max_head_pan_speed*FrameTime*x/1000;
	maxSpeeds[HeadOffset+RollOffset]=config->motion.max_head_roll_speed*FrameTime*x/1000;
}

int PostureMC::updateOutputs() {
	int tmp=isDirty();
	if(tmp || hold) {
		dirty=false;
		for(unsigned int i=0; i<NumOutputs; i++) {
			if(maxSpeeds[i]<=0) {
				curPositions[i]=cmds[i].value;
				motman->setOutput(this,i,cmds[i]);
			} else { // we may be trying to exceeded maxSpeed
				unsigned int f=0;
				while(cmds[i].value>curPositions[i]+maxSpeeds[i] && f<NumFrames) {
					curPositions[i]+=maxSpeeds[i];
					motman->setOutput(this,i,OutputCmd(curPositions[i],cmds[i].weight),f);
					f++;
				}
				while(cmds[i].value<curPositions[i]-maxSpeeds[i] && f<NumFrames) {
					curPositions[i]-=maxSpeeds[i];
					motman->setOutput(this,i,OutputCmd(curPositions[i],cmds[i].weight),f);
					f++;
				}
				if(f<NumFrames) { //we reached target value, fill in rest of frames
					curPositions[i]=cmds[i].value;
					for(;f<NumFrames;f++)
						motman->setOutput(this,i,cmds[i],f);
				} else // we didn't reach target value, still dirty
					dirty=true;
			}
		}
		if(!dirty && !targetReached) {
			postEvent(EventBase(EventBase::motmanEGID,getID(),EventBase::statusETID));
			targetReached=true;
			targetTimestamp=get_time();
		}
	}
	return tmp;
}

int PostureMC::isAlive() {
	if(dirty || !targetReached)
		return true;
	if(targetReached && get_time()-targetTimestamp>timeout) { //prevents a conflicted PostureMC's from fighting forever
		if(getAutoPrune())
			serr->printf("WARNING: posture timed out - possible joint conflict or out-of-range target\n");
		return false;
	}
	PostureEngine tmp; 
	tmp.takeSnapshot(); 
	return (maxdiff(tmp)>tolerance);
}

void PostureMC::init() {
	defaultMaxSpeed();
	setDirty();
}

/*! @file
 * @brief Implements PostureMC, a MotionCommand shell for PostureEngine
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.2 $
 * $State: Exp $
 * $Date: 2005/01/07 21:12:35 $
 */
