#ifndef PLATFORM_APERIOS
#include "Thread.h"

#include <pthread.h>
#include <string.h>
#include <iostream>
#include <signal.h>
#include <unistd.h>

using namespace std;

struct Threadstorage_t {
	Threadstorage_t() : threadInfo(), threadAttr() {
		if(int err=pthread_attr_init(&threadAttr))
			cerr << "Threadstorage_t constructor, pthread_attr_init: " << strerror(err) << endl;;
	}
	~Threadstorage_t() {
		if(int err=pthread_attr_destroy(&threadAttr))
			cerr << "Threadstorage_t destructor, pthread_attr_destroy: " << strerror(err) << endl;
	}
	pthread_t threadInfo;
	pthread_attr_t threadAttr;
private:
	Threadstorage_t(const Threadstorage_t& r); //!< don't call
	Threadstorage_t& operator=(const Threadstorage_t& r); //!< don't call
};

Thread::Thread() : pt(new Threadstorage_t), running(false), returnValue(NULL) {}

Thread::~Thread() {
	//can only happen externally
	if(running) {
		stop();
		join();
	}
	if(pt==NULL) {
		std::cerr << "Thread storage already deleted!?!?!" << std::endl;
		*(int*)NULL=0xDEADDEAD;
	}
	delete pt;
	pt=NULL;
}

void Thread::start() {
	if(running) {
		std::cerr << "Thread::start() -- thread is already running!" << std::endl;
		std::cerr << "   make another instance if you want to run another copy of this thread" << std::endl;
		return;
	}
	running=true;
	if(int err=pthread_create(&pt->threadInfo, &pt->threadAttr, launch, this))
		cerr << "Thread start(), pthread_create: " << strerror(err) << endl;
}

void * Thread::run() {
	for(;;) {
		unsigned int sleeptime=runloop();
		if(sleeptime==-1U)
			return returnValue;
		usleep(sleeptime);
		testCancel();
	}
	return NULL; //never happens
}

void Thread::stop() {
	if(int err=pthread_cancel(pt->threadInfo))
		cerr << "Thread cancel(), pthread_cancel: " << strerror(err) << endl;
}

void Thread::kill() {
	if(int err=pthread_kill(pt->threadInfo,SIGALRM))
		cerr << "Thread stop(), pthread_kill(SIGALRM): " << strerror(err) << endl;
}

void Thread::murder() {
	if(int err=pthread_detach(pt->threadInfo))
		cerr << "Thread kill(), thread_detach: " << strerror(err) << endl;
	if(int err=pthread_kill(pt->threadInfo,SIGSTOP))
		cerr << "Thread kill(), pthread_kill(SIGSTOP): " << strerror(err) << endl;
	running=false;
}

void * Thread::join() {
	void * ans=NULL;
	if(int err=pthread_join(pt->threadInfo, &ans))
		cerr << "Thread join(), pthread_join: " << strerror(err) << endl;
	return ans;
}

void Thread::testCancel() {
	pthread_testcancel();
}

void * Thread::launch(void * msg) {
	if(int err=pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,NULL))
		cerr << "Thread launch(), pthread_setcanceltype: " << strerror(err) << endl;
	if(int err=pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED,NULL))
		cerr << "Thread launch(), pthread_setcanceltype: " << strerror(err) << endl;
	signal(SIGALRM,Thread::handle_signal);
	void * ans=NULL;
	{
		//These pthread functions actually define a scope between them
		//I've added braces of my own to make this explicitly clear
		pthread_cleanup_push(Thread::handle_exit,msg);
		static_cast<Thread*>(msg)->run();
		pthread_cleanup_pop(true);
	}
	return ans;
}

void Thread::handle_signal(int /*sig*/) {
	pthread_exit(NULL);
}

void Thread::handle_exit(void * th) {
	static_cast<Thread*>(th)->running=false;
}

namespace ThreadNS {
		
	struct LockStorage {
		LockStorage() : mutex(), attr() {
			pthread_mutexattr_init(&attr);
			pthread_mutexattr_settype(&attr,PTHREAD_MUTEX_RECURSIVE);
			pthread_mutex_init(&mutex,&attr);
		}
		~LockStorage() {
			pthread_mutexattr_destroy(&attr);
			pthread_mutex_destroy(&mutex);
		}
		LockStorage(const LockStorage& ls) : mutex(ls.mutex), attr(ls.attr) {}
		LockStorage& operator=(const LockStorage& ls) { mutex=ls.mutex; attr=ls.attr; return *this; }
		void lock() {
			pthread_mutex_lock(&mutex);
		}
		bool trylock() {
			return !pthread_mutex_trylock(&mutex);
		}
		void unlock() {
			pthread_mutex_unlock(&mutex);
		}
		pthread_mutex_t mutex;
		pthread_mutexattr_t attr;
	};

	map<int,struct LockStorage*> Lock::ids;
	bool Lock::issetup=false;

	Lock::Lock() : my_id() {
		if(!issetup)
			setup();
		ids[-1]->lock();
		my_id=(--ids.end())->first+1;
		ids[my_id]=new LockStorage;
		ids[-1]->unlock();
	}
	/*
	Lock::Lock(int ident, bool autolock/ *=true* /) : my_id(ident) {
		if(!issetup)
			setup();
		if(ids.find(my_id)==ids.end()) {
			ids[-1]->lock();
			ids[my_id]=new LockStorage;
			ids[-1]->unlock();
		}
		if(autolock)
			ids[my_id]->lock();
	}
	*/
	Lock::~Lock() {
		ids[my_id]->unlock();
	}

	void Lock::lock() {
		ids[my_id]->lock();
	}
	bool Lock::trylock() {
		return ids[my_id]->trylock();
	}
	void Lock::unlock() {
		ids[my_id]->unlock();
	}
	void Lock::setup() {
		issetup=true;
		ids[-1]=new LockStorage;
	}

}

#endif // PLATFORM check

/*! @file
* @brief Describes the Thread class and its AutoThread templated subclass
* @author ejt (Creator)
*
* $Author: ejt $
* $Name: tekkotsu-2_4_1 $
* $Revision: 1.6 $
* $State: Exp $
* $Date: 2005/08/01 23:17:59 $
*/
