//-*-c++-*-
#ifndef INCLUDED_RegionRegistry_h_
#define INCLUDED_RegionRegistry_h_

#ifdef PLATFORM_APERIOS
#  warning RegionRegistry is not Aperios compatable
#else

#include "ListMemBuf.h"
#include "RCRegion.h"
#include "LockScope.h"
#include "ProcessID.h"

//! Keeps track of currently available shared memory regions
template<unsigned int MAX_REGIONS, unsigned int NAME_LEN=0>
class RegionRegistry {
protected:
	mutable MutexLock<ProcessID::NumProcesses> lock;
	typedef LockScope<ProcessID::NumProcesses> AutoLock;
	
	//! Holds information regarding a shared memory region available for listening
	struct entry {
		//! constructor
		entry() : id() { name[0]='\0'; }
		entry(const char n[], RCRegion* r) : id(r->ID()) {
			strncpy(name,n,NAME_LEN);
			name[NAME_LEN]='\0';
		}
		char name[NAME_LEN+1]; //!< the name for the region
		RCRegion::Identifier id;
	};
	typedef ListMemBuf<entry,MAX_REGIONS> registry_t;
	registry_t avail;

public:
	static const unsigned int CAPACITY=MAX_REGIONS;
	static const unsigned int REGION_NAME_LEN=NAME_LEN;
	typedef typename registry_t::index_t index_t;

	RegionRegistry() : lock(), avail() {}
	
	~RegionRegistry() {
		AutoLock autolock(lock,ProcessID::getID());
		avail.clear();
	}
	
	index_t findRegion(const std::string& name) const {
		AutoLock autolock(lock,ProcessID::getID());
		if(name.size()>NAME_LEN)
			std::cerr << "WARNING: RegionRegistry::attach("<<name<<") is too long, max is " << NAME_LEN << std::endl;
		for(index_t it=begin(); it!=end(); it=next(it))
			if(name==avail[it].name)
				return it;
		return avail.end();
	}

	index_t registerRegion(const std::string& name, const RCRegion * region) {
		AutoLock autolock(lock,ProcessID::getID());
		index_t it=findRegion(name);
		if(it!=end()) { //found, already registered
			if(avail[it].regions[ProcessID::getID()]==region)
				return it; //same one, just return it
			return end(); // conflict, return invalid
		}
		//not found, make a new one
		return avail.push_back(entry(name.c_str(),region));
	}
	
	RCRegion * registerRegion(const std::string& name, size_t size) {
		AutoLock autolock(lock,ProcessID::getID());
		index_t it=findRegion(name);
		if(it!=end()) {
			//found, already registered
			return RCRegion::attach(avail[it].id);
		} else {
			//not found:
			RCRegion * region = new RCRegion(name,size);
			avail.push_back(entry(name.c_str(),region));
			return region;
		}
	}
	
	RCRegion * operator[](index_t it) const {
		AutoLock autolock(lock,ProcessID::getID());
		if(it==end())
			return NULL;
		return RCRegion::attach(avail[it].id);
	}
	
	void erase(index_t it) {
		AutoLock autolock(lock,ProcessID::getID());
		avail.erase(it);
	}
	
	index_t begin() const { return avail.begin(); }
	index_t next(index_t it) const { return avail.next(it); }
	index_t end() const { return avail.end(); }
};

/*! @file
 * @brief 
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.2 $
 * $State: Exp $
 * $Date: 2005/06/01 05:47:46 $
 */

#endif //Aperios check

#endif //INCLUDED

