//-*-c++-*-
#ifdef PLATFORM_APERIOS
#  include <OPENR/RCRegion.h>
#else
#  ifndef INCLUDED_RCRegion_h_
#  define INCLUDED_RCRegion_h_

#include "Shared/ReferenceCounter.h"
#include "MutexLock.h"
#include "ProcessID.h"
#include "Shared/plist.h"
#include <sys/types.h>
#include <map>
#include <exception>

//do you want to use the SysV shared memory call interface
//or the POSIX shared memory interface?  They have different
//strengths/weaknesses on different platforms... :-/
#ifndef POSIX_SHM
#  define POSIX_SHM 1
#endif
#ifndef SYSV_SHM
#  define SYSV_SHM 2
#endif
#ifndef TEKKOTSU_SHM_STYLE
#  define TEKKOTSU_SHM_STYLE POSIX_SHM
#endif

#if TEKKOTSU_SHM_STYLE!=SYSV_SHM && TEKKOTSU_SHM_STYLE!=POSIX_SHM
#  error Unknown TEKKOTSU_SHM_STYLE setting
#endif


//! provides compatability with the OPEN-R type of the same name
class RCRegion {
public:
#if TEKKOTSU_SHM_STYLE==SYSV_SHM
	//! contains all information needed to attach this region from a different process
	struct Identifier {
		Identifier() : key(), shmid(), size(0) {}
		key_t key;
		int shmid;
		size_t size;
	};
	//! constructor (OPEN-R compatability)
	explicit RCRegion(size_t sz)
		: id(), base(NULL), references(NULL), lock(NULL)
	{ init(sz,nextKey,true); }
	//! constructor, name isn't used for sysv
	/*! could hash the name to generate key...? */
	RCRegion(const std::string&, size_t sz)
		: id(), base(NULL), references(NULL), lock(NULL)
	{ init(sz,nextKey,true); }
#elif TEKKOTSU_SHM_STYLE==POSIX_SHM
	//! maximum guaranteed length for users' region names (might have a little leeway depending on process ID prefix or tmp path prefix)
	static const unsigned int MAX_NAME_LEN=64;
#  ifndef USE_UNBACKED_SHM
	static plist::Primitive<std::string> shmRoot;
#  endif
	static plist::Primitive<bool> useUniqueMemoryRegions;
	static pid_t rootPID; //!< this is the pid of the original process, used for unique names of memory regions; pid_t is from sys/types.h
	struct Identifier {
		Identifier() : size(0) {}
		char key[MAX_NAME_LEN];
		size_t size;
	};
	//! constructor (OPEN-R compatability, name is autogenerated)
	explicit RCRegion(size_t sz)
		: id(), base(NULL), references(NULL), lock(NULL)
	{
		char name[RCRegion::MAX_NAME_LEN];
		static unsigned int sn=0;
		snprintf(name,RCRegion::MAX_NAME_LEN,"Rgn.%d.%d",ProcessID::getID(),++sn);
		name[RCRegion::MAX_NAME_LEN-1]='\0';
		init(sz,name,true);
	}
	//! constructor, specify your own name for better debugging accountability (not OPEN-R compatable)
	RCRegion(const std::string& name, size_t sz)
		: id(), base(NULL), references(NULL), lock(NULL)
	{ init(sz,name,true); }
#endif

	static RCRegion * attach(const Identifier& rid);
	
	char * Base() const { return base; }
	size_t Size() const { return id.size; }
	static void setNextKey(key_t k) { nextKey=k; }
	const Identifier& ID() const { return id; }
	
	int NumberOfReference() const { return references[ProcessID::NumProcesses]; }
	int NumberOfLocalReference() const { return references[ProcessID::getID()]; }
	void AddReference();
	void RemoveReference();
	void AddSharedReference();
	void RemoveSharedReference();
	
	static void aboutToFork(ProcessID::ProcessID_t newID);
	static void faultShutdown();
	static unsigned int NumberOfAttach() { return attachedRegions.size(); }
	
	//! Different methods of handling regions with conflicting keys
	enum ConflictResolutionStrategy {
		RENAME,  //!< try another key until we find one that works (better for SYSV, maybe not so smart for POSIX)
		REPLACE, //!< delete the other region and try again (better for POSIX, maybe not so smart for SYSV)
		EXIT //!< go home and cry about it
	};
	
	static void setConflictResolution(ConflictResolutionStrategy crs) { conflictStrategy=crs; }
	static ConflictResolutionStrategy getConflictResolution() { return conflictStrategy; }
	
protected:
	RCRegion(const Identifier& rid)
		: id(), base(NULL), references(NULL), lock(NULL)
	{ init(rid.size,rid.key,false); }

	~RCRegion();

	static const unsigned int align=sizeof(unsigned int);
	static const unsigned int extra=sizeof(unsigned int)*(ProcessID::NumProcesses+1)
	                                +sizeof(MutexLock<ProcessID::NumProcesses>);
	static unsigned int calcRealSize(unsigned int size);

#if TEKKOTSU_SHM_STYLE==SYSV_SHM
	void init(size_t sz, key_t sug_key, bool create);
#elif TEKKOTSU_SHM_STYLE==POSIX_SHM
	std::string getQualifiedName() const { return getQualifiedName(id.key); }
	static std::string getQualifiedName(const std::string& key);
	int openRegion(int mode) const;
	bool unlinkRegion() const;
	void init(size_t sz, const std::string& name, bool create);
#endif

	static ConflictResolutionStrategy conflictStrategy;
	static bool isFaultShutdown;
									  
	static key_t nextKey;
#if TEKKOTSU_SHM_STYLE==SYSV_SHM
	typedef std::map<key_t,RCRegion*> attachedRegions_t;
#elif TEKKOTSU_SHM_STYLE==POSIX_SHM
	typedef std::map<std::string,RCRegion*> attachedRegions_t;
#endif
	static attachedRegions_t attachedRegions;
	
	Identifier id;
	char * base;
	unsigned int * references;
	MutexLock<ProcessID::NumProcesses> * lock;

private:
	RCRegion(const RCRegion& r); //!< don't call
	RCRegion& operator=(const RCRegion& r); //!< don't call
};

/*! @file
 * @brief Describes RCRegion, which provides compatability with the OPEN-R type of the same name
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.2 $
 * $State: Exp $
 * $Date: 2005/06/01 05:47:46 $
 */

#  endif
#endif
