//-*-c++-*-
#ifndef INCLUDED_MessageQueue_h_
#define INCLUDED_MessageQueue_h_

#ifdef PLATFORM_APERIOS
#  warning MessageQueue is not Aperios compatable
#else

#include "ListMemBuf.h"
#include "RCRegion.h"
#include "LockScope.h"
#include <exception>

//! Defines the interface for sending new shared memory regions between processes
/*! This base class holds all of the template-independent code to allow general
 *  operations on MessageQueues.  The templated version of MessageQueue provides
 *  concrete implementation, which is what you would instantiate.
 *  
 *  Each message entails its own shared memory region, as compared to
 *  SharedQueue, where a single large buffer is maintained, and all messages are
 *  copied into the common buffer.  This class is better for large regions since
 *  it can avoid copying data around. */
class MessageQueueBase {
public:

	//!constructor
	MessageQueueBase()
		: lock(), overflowPolicy(THROW_BAD_ALLOC), isClosed(false), reportDroppings(false), numMessages(0),
			numReceivers(0), messagesRead()
	{}
	//!destructor
	virtual ~MessageQueueBase() {}
	
	
	//!An interface to allow you to receive callbacks when a message has been read from a MessageQueue, subscribed via an external calss which is monitoring the queue's MessageQueueBase::pollStatus() (e.g. LoadFileThread)
	class StatusListener {
	public:
		//! destructor -- does nothing
		virtual ~StatusListener() {}
		
		//! Called after a message has been read by all receivers, and thus has been removed from the queue
		/*! In order to allow this to happen, a thread must repeatedly call
		 *  MessageQueueBase::pollStatus() in order to check if other processes have
		 *  read their messages since the last pollStatus call.
		 *
		 *  Don't assume that because you receive this callback there is space in
		 *  the queue -- an earlier listener may have already added a message, or
		 *  the queue might have been already waiting to send a message if
		 *  #overflowPolicy is #WAIT
		 *
		 *  @param which The MessageQueueBase which has had message(s) read
		 *  @param howmany The number of message which have been cleared */
		virtual void messagesRead(MessageQueueBase& which, unsigned int howmany)=0;
	};

	//! Checks to see how many messages have been processed (read by all receivers and removed from queue) since the last call to pollStatus()
	virtual unsigned int pollStatus() {
		AutoLock autolock(lock);
		unsigned int read=messagesRead;
		messagesRead=0;
		return read;
	}

	
	//! The storage type for message entry indicies
	/*! This index is to be used with accessor functions, but may be recycled for
	 *  a new message after all receivers have read the previous message.  If you
	 *  wish to have a unique message identifier, see getMessageSN() */
	typedef unsigned short index_t;
	
	
	//!< add one to the receiver reference count
	virtual void addReceiver()=0;
	//!< remove one from the receiver reference count
	virtual void removeReceiver()=0;
	//! return the receiver reference count
	virtual unsigned int getNumReceivers() const { return numReceivers; }


	//! post a message into the queue -- a shared reference is added, the caller retains control current reference
	/*! Thus, if you are sending a region and do not intend to use it again, call
	 *  RCRegion::RemoveReference() after sending to free the sender's memory.
	 *  Otherwise, you can continue to access the region, even as the receiver
	 *  accesses it as well.  If both sides retain references, you can use the
	 *  region as a shared memory area for future communication.  (beware of race
	 *  conditions!) */
	virtual void sendMessage(RCRegion * rcr)=0;
	//! request access to a particular message, increments read counter -- do not call more than once per receiver!
	/*! The message is marked read and will be popped from the queue if all
	 *  receivers have read the message as well.  The caller inherits a reference
	 *  to the returned region -- call RemoveReference when you are done with
	 *  it */
	virtual RCRegion * readMessage(index_t msg)=0;
	//! request access to a particular message, does not mark message -- call as often as you like
	/*! The caller inherits a reference to the returned region -- call
	 *  RemoveReference when you are done with it */
	virtual RCRegion * peekMessage(index_t msg)=0;
	//! increments read counter -- do not call more than once per receiver!
	virtual void markRead(index_t msg)=0;
	//! do not allow any new messages to be posted
	virtual void close() { AutoLock autolock(lock); isClosed=true; }

	virtual void setReportDroppings(bool report) { reportDroppings=report; }
	virtual bool getReportDroppings() const { return reportDroppings; }
	
	
	//! Each message gets a unique, monotonically increasing serial number; this function returns that number (#serialNumber)
	virtual unsigned int getMessageSN(index_t msg)=0;
	
	
	//! a typedef to make it easier to obtain a lock on the queue for the extent of a scope
	typedef LockScope<ProcessID::NumProcesses> AutoLock;
	//! returns a lock on the queue for the scope of the returned storage
	AutoLock getLock() const { return AutoLock(lock); }

	
	virtual index_t oldest() const=0;          //!< return oldest message still in the queue (may or may not have been read by this process)
	virtual index_t newer(index_t it) const=0; //!< return the next message in the queue (may or may not have been read by this process)
	virtual index_t older(index_t it) const=0; //!< return the previous message in the queue (may or may not have been read by this process)
	virtual index_t newest() const=0;          //!< return most recent message added to the queue (may or may not have been read by this process)
	virtual bool isEnd(index_t it) const=0;    //!< returns true if @a it is the one-past-the-end of the queue
	
	//! an enumerations of policies for dealing with overflow, pass to setOverflowPolicy()
	enum OverflowPolicy_t {
		DROP_OLDEST,     //!< the oldest unread message is dropped
		DROP_NEWEST,     //!< the most recently added message is dropped (i.e. the overflowing message is ignored)
		WAIT,            //!< the adding process/thread polls until space is available
		THROW_BAD_ALLOC  //!< throw a std::bad_alloc exception (falls through to abort() if you don't catch it)
	};
	//! allows you to pick how to handle running out of space in the queue, see OverflowPolicy_t
	void setOverflowPolicy(OverflowPolicy_t op) { overflowPolicy=op; }
	//! returns the current overflow policy, see OverflowPolicy_t
	OverflowPolicy_t getOverflowPolicy() const { return overflowPolicy; }
	
protected:
	//! data storage needed for each message
	struct entry {
		entry() : id(), sn(), numRead(0) {} //!< constructor
		entry(unsigned int serialNumber, RCRegion* r)
			: id(r->ID()), sn(serialNumber), numRead(0) {} //!< constructor, pass message info
		RCRegion::Identifier id; //! the identifier for the shared memory region so that other regions can attach it
		unsigned int sn; //!< serial number for this message (not the same as its index in the queue -- indicies are reused, this id is unique to this message
		unsigned int numRead; //!< a count of the number of receivers which have read this message
	};
	mutable MutexLock<ProcessID::NumProcesses> lock; //!< a lock to grant serial access to the queue
	OverflowPolicy_t overflowPolicy; //!< the choice of how to handle message overflow -- see OverflowPolicy_t
	bool isClosed; //!< if true, new messages will be rejected
	bool reportDroppings; //!< if true, output will be sent on cerr when overflow occurs
	unsigned int numMessages; //!< number of messages which have been sent (serial number of next message)
	unsigned int numReceivers; //!< how many receivers to expect
	unsigned int messagesRead; //!< number of messages which have been read and removed from queue since last call to pollStatus()
};

//! An implementation of MessageQueueBase, which provides mechanisms for sending shared memory regions between processes
/*! @see MessageQueueBase */
template<unsigned int MAX_UNREAD>
class MessageQueue : public MessageQueueBase {
public:
	//! total number of messages which can be backed up in the queue
	static const unsigned int CAPACITY=MAX_UNREAD;
	
	//! constructor
	MessageQueue() : MessageQueueBase(), mq() {}
	
	virtual ~MessageQueue() {
		//lock shouldn't be necessary -- refcount should ensure the containing
		//region isn't deleted until only one process has access anyway
		//AutoLock autolock(lock);
		while(!mq.empty()) {
			RCRegion * rcr = RCRegion::attach(mq.front().id);
			rcr->RemoveSharedReference();
			rcr->RemoveReference();
			mq.pop_front();
		}
	}
	
	virtual void addReceiver() {
		AutoLock autolock(lock);
		numReceivers++;
	}
	
	virtual void removeReceiver() {
		AutoLock autolock(lock);
		numReceivers--;
		for(index_t it=mq.begin(); it!=mq.end(); it=mq.next(it)) {
			if(mq[it].numRead==numReceivers) {
				//all *remaining* processes have gotten a look, remove the neutral MessageQueue reference
				RCRegion * rcr = RCRegion::attach(mq[it].id);
				rcr->RemoveSharedReference();
				rcr->RemoveReference();
				it=mq.prev(it);
				mq.erase(mq.next(it));
				messagesRead++;
			}
		}
	}
	
	virtual void sendMessage(RCRegion * rcr) {
		AutoLock autolock(lock);
		if(numReceivers==0) {
			//if(reportDroppings)
			//std::cerr << "Warning: MessageQueue dropping " << rcr->ID().key << " because there are no receivers" << std::endl;
			return;
		}
		if(isClosed) {
			if(reportDroppings)
				std::cerr << "Warning: MessageQueue dropping " << rcr->ID().key << " because queue is closed" << std::endl;
			return;
		}
		rcr->AddSharedReference();
		if(mq.size()==mq.getMaxCapacity()) {
			switch(overflowPolicy) {
				case DROP_OLDEST: {
					if(reportDroppings)
						std::cerr << "WARNING: MessageQueue full, dropping oldest unread message (" << mq.front().id.key << ")" << std::endl;
					RCRegion * eldest = RCRegion::attach(mq.front().id);
					eldest->RemoveSharedReference();
					mq.pop_front();
					eldest->RemoveReference();
				} break;
				case DROP_NEWEST:
					if(reportDroppings)
						std::cerr << "WARNING: MessageQueue full, dropping newest unread message (" << rcr->ID().key << ")" << std::endl;
					rcr->RemoveSharedReference();
					return;
				case WAIT:
					if(reportDroppings)
						std::cerr << "WARNING: MessageQueue full, waiting for readers to catch up" << std::endl;
					while(mq.size()==mq.getMaxCapacity()) {
						//have to release locks so readers can get access
						unsigned int ll=lock.get_lock_level();
						lock.releaseAll();
						usleep(MutexLockBase::usleep_granularity*15);
						for(unsigned int i=0; i<ll; i++)
							lock.lock(ProcessID::getID());
					}
					break;
				case THROW_BAD_ALLOC:
					if(reportDroppings)
						std::cerr << "WARNING: MessageQueue full, throwing bad_alloc exception" << std::endl;
					rcr->RemoveSharedReference();
					throw std::bad_alloc();
					break;
			}
		}
		if(mq.push_back(entry(numMessages++,rcr))==mq.end()) {
			//our overflow policy should've prevented this
			std::cerr << "ERROR: MessageQueue unable to add message; buggy overflow policy?" << std::endl;
			exit(EXIT_FAILURE);
		}
	}
	
	virtual RCRegion * readMessage(index_t msg) {
		AutoLock autolock(lock);
		RCRegion * rcr = RCRegion::attach(mq[msg].id);
		mq[msg].numRead++;
		if(mq[msg].numRead==numReceivers) {
			//all processes have gotten a look, remove the neutral MessageQueue reference
			rcr->RemoveSharedReference();
			mq.erase(msg);
			messagesRead++;
		}
		return rcr;
	}
	
	virtual RCRegion * peekMessage(index_t msg) {
		//AutoLock autolock(lock); //I don't think a lock is necessary here
		return RCRegion::attach(mq[msg].id);
	}
	
	virtual void markRead(index_t msg) {
		AutoLock autolock(lock);
		mq[msg].numRead++;
		if(mq[msg].numRead==numReceivers) {
			//all processes have gotten a look, remove the neutral MessageQueue reference
			RCRegion * rcr = RCRegion::attach(mq[msg].id);
			rcr->RemoveSharedReference();
			rcr->RemoveReference();
			mq.erase(msg);
			messagesRead++;
		}
	}

	virtual unsigned int getMessageSN(index_t msg) { /*AutoLock autolock(lock);*/ return mq[msg].sn; }
	
	virtual index_t oldest() const { AutoLock autolock(lock); return mq.begin(); }
	virtual index_t newer(index_t it) const { AutoLock autolock(lock); return mq.next(it); }
	virtual index_t older(index_t it) const { AutoLock autolock(lock); return mq.prev(it); }
	virtual index_t newest() const { AutoLock autolock(lock); return mq.prev(mq.end()); }
	virtual bool isEnd(index_t it) const { AutoLock autolock(lock); return it==mq.end() || it>=mq_t::MAX_ENTRIES; }
	
protected:
	//! shorthand for the type of data storage of message entries
	typedef ListMemBuf<entry,MAX_UNREAD,index_t> mq_t;
	
	//! the data storage of message entries
	mq_t mq;
};

/*! @file
 * @brief Defines MessageQueue, which provides mechanisms for sending shared memory regions between processes
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.12 $
 * $State: Exp $
 * $Date: 2005/08/04 21:32:16 $
 */

#endif //APERIOS check

#endif //INCLUDED
