//-*-c++-*-
#ifndef INCLUDED_VisionObjectEvent_h
#define INCLUDED_VisionObjectEvent_h

#include "EventBase.h"

//! Extends EventBase to also include location in the visual field and distance (though distance is not implimented yet)
class VisionObjectEvent : public EventBase {
public:
	//! Constructor, pass a source id and type id -- mainly useful for deactivate events since all object parameters are going to be set to 0
	/*! @param sid The source ID for the object being detected -- you can define your own values, some are already set in ProjectInterface, but can be reassigned during your project's startup
	 *  @param tid The type ID for the event
	 */
  explicit VisionObjectEvent(unsigned int sid=0,EventTypeID_t tid=EventBase::deactivateETID)
		: EventBase(EventBase::visObjEGID,sid,tid,0),
			_x1(0),_x2(0),_y1(0),_y2(0), _area(0),
			_clipLeft(false), _clipRight(false), _clipTop(false), _clipBottom(false),
			_xRange(0), _yRange(0), _frame(0)
	{}
		
	//! Constructor, pass the type id, source id, left, right, top, bottom, x range, and y range
	/*! @param sid The source ID for the object being detected -- you can define your own values, some are already set in ProjectInterface, but can be reassigned during your project's startup
	 *  @param tid The type ID for the event
	 *  @param x1 The leftmost extent of the object, in generalized coordinates (see #_x1)
	 *  @param x2 The rightmost extent of the object in generalized coordinates (see #_x2)
	 *  @param y1 The topmost extent of the object, in generalized coordinates (see #_y1)
	 *  @param y2 The bottommost extent of the object, in generalized coordinates (see #_y2)
	 *  @param objarea The area of the object being detected, in squared generalized coordinates (see #_area)
	 *  @param rx The plus/minus range of the x coordinates (generally xres/xres for cameras which are wider than they are high)
	 *  @param ry The plus/minus range of the y coordinates (camera yres/xres for cameras which are wider than they are high)
	 */
  VisionObjectEvent(unsigned int sid, EventTypeID_t tid, float x1, float x2, float y1, float y2,float objarea, float rx, float ry)
		: EventBase(EventBase::visObjEGID,sid,tid,0),
			_x1(x1),_x2(x2),_y1(y1),_y2(y2), _area(objarea),
			_clipLeft(_x1<=-rx), _clipRight(_x2>=rx), _clipTop(_y1<=-ry), _clipBottom(_y2>=ry),
			_xRange(rx), _yRange(ry), _frame(0)
	{}
	
 	//! Constructor, pass the type id, source id, left, right, top, bottom, x range, y range, and frame_number
	/*! @param sid The source ID for the object being detected -- you can define your own values, some are already set in ProjectInterface, but can be reassigned during your project's startup
	 *  @param tid The type ID for the event
	 *  @param x1 The leftmost extent of the object, in generalized coordinates (see #_x1)
	 *  @param x2 The rightmost extent of the object in generalized coordinates (see #_x2)
	 *  @param y1 The topmost extent of the object, in generalized coordinates (see #_y1)
	 *  @param y2 The bottommost extent of the object, in generalized coordinates (see #_y2)
	 *  @param objarea The area of the object being detected, in squared generalized coordinates (see #_area)
	 *  @param rx The plus/minus range of the x coordinates (generally xres/xres for cameras which are wider than they are high)
	 *  @param ry The plus/minus range of the y coordinates (camera yres/xres for cameras which are wider than they are high)
	 *  @param frame The camera frame number the object was detected in (see #_frame)
	 */
  VisionObjectEvent(unsigned int sid, EventTypeID_t tid, float x1, float x2, float y1, float y2, float objarea, float rx, float ry,unsigned int frame)
		: EventBase(EventBase::visObjEGID,sid,tid,0),
			_x1(x1),_x2(x2),_y1(y1),_y2(y2), _area(objarea),
			_clipLeft(_x1<=-rx), _clipRight(_x2>=rx), _clipTop(_y1<=-ry), _clipBottom(_y2>=ry),
			_xRange(rx),_yRange(ry), _frame(frame)
	{} 
  
	//! destructor
	virtual ~VisionObjectEvent() {}
  
	virtual EventBase* clone() const { return new VisionObjectEvent(*this); }

	virtual unsigned int getClassTypeID() const { return makeClassTypeID("VISO"); }

	//!@name Attribute Accessors
  float getLeft() const { return _x1;} //!< returns the initial x (#_x1) coordinate of the Bounding Box (inclusive value)
  VisionObjectEvent& setLeft(float x1) { _x1=x1; return *this;} //!< sets the initial x (#_x1) coordinate of the Bounding Box
  
  float getRight() const { return _x2;} //!< returns the final x (#_x2) coordinate of the Bounding Box (inclusive value)
  VisionObjectEvent& setRight(float x2) { _x2=x2; return *this;} //!< sets the final x (#_x2) coordinate of the Bounding Box
  
	float getTop() const { return _y1;} //!< returns the initial y (#_y1) coordinate of the Bounding Box (inclusive value)
  VisionObjectEvent& setTop(float y1) { _y1=y1; return *this;} //!< sets the initial y (#_y1) coordinate of the Bounding Box
  
  float getBottom() const { return _y2;} //!< returns the final y (#_y2) coordinate of the Bounding Box (inclusive value)
  VisionObjectEvent& setBottom(float y2) { _y2=y2; return *this;} //!< sets the final y (#_y2) coordinate of the Bounding Box

	float getObjectArea() const { return _area; } //!< returns the object's #_area within the camera, in squared generalized coordinates (multiply by the major camera resolution to get pixel area)
	VisionObjectEvent& setObjectArea(float objarea) { _area=objarea; return *this; } //!< returns the object's #_area within the camera, in squared generalized coordinates (multiply by the major camera resolution to get pixel area)
	//@}
 
	//!@name Calculated Attributes
  float getCenterX() const { return (_x1+_x2)/2; } //!< returns the center along x
  float getCenterY() const { return (_y1+_y2)/2; } //!< returns the center along y
  float getWidth() const { return _x2-_x1; } //!< return width along x
  float getHeight() const { return _y2-_y1; } //!< return height along y
  float getBoundaryArea() const { return (_x2-_x1)*(_y2-_y1); } //!< returns the area of the bounding box, just multiplication of width*height, (multiply by the major camera resolution to get pixel area)
  float getXrange() const{ return  _xRange;}//!< returns the maximum x value
  float getYrange() const{return _yRange;}//!< returns the maximum y value
  unsigned int getFrame() const{return _frame;}//!< returns number of frame when the event was generated
	//@}

  //!@name Object out of bounds Detection Functions
  bool isClippedLeft() const { return _clipLeft; } //!< returns #_clipLeft
  bool isClippedRight() const { return _clipRight; }  //!< returns #_clipRight
  bool isClippedTop() const { return _clipTop; } //!< returns #_clipTop
  bool isClippedBottom() const {return _clipBottom; } //!< returns #_clipBottom
	void setClipping(bool left, bool right, bool top, bool bottom) { _clipLeft=left; _clipRight=right; _clipTop=top; _clipBottom=bottom; } //!< sets clipping boundaries
	//@}
      
	virtual std::string getDescription(bool showTypeSpecific=true, unsigned int verbosity=0) const;
	
	virtual unsigned int getBinSize() const;
	virtual unsigned int LoadBinaryBuffer(const char buf[], unsigned int len);
	virtual unsigned int SaveBinaryBuffer(char buf[], unsigned int len) const;
	virtual void LoadXML(xmlNode* node);
	virtual void SaveXML(xmlNode * node) const;

protected:
	float _x1;  //!< a value representing location in visual field - from -1 if on the left edge to 1 if it's on the right edge
	float _x2;  //!< a value representing location in visual field - from -1 if on the left edge to 1 if it's on the right edge
	float _y1;  //!< top boundary, in range of @f$ \pm\frac{160}{208} @f$) for ERS-7; actual values vary depending on aspect ratio to keep square coordinates
	float _y2;  //!< bottom boundary, in approximately @f$ \pm\frac{160}{208} @f$ for ERS-7; actual values vary depending on aspect ratio to keep square coordinates
	float _area; //!< area of the actual object within bounding box as set by generator, in same units as getBoundaryArea() (squared generalized coordinates -- multiply by the major camera resolution to get pixel area)
	bool _clipLeft;   //!< flag to indicate left boundary is on or beyond the camera image's boundary
	bool _clipRight;  //!< flag to indicate right boundary is on or beyond the camera image's boundary
	bool _clipTop;    //!< flag to indicate top boundary is on or beyond the camera image's boundary
	bool _clipBottom; //!< flag to indicate bottom boundary is on or beyond the camera image's boundary
	float _xRange; //!< Max range of X dimension
	float _yRange; //!< Max range of Y dimension
	unsigned int _frame; //!< Number of frame when the event was generated.
};

/*! @file
 * @brief Describes VisionObjectEvent, which provides information about objects recognized in the camera image
 * @author alokl (Creator)
 * @author Ignacio Herrero Reder &lt; nhr at dte uma es &gt; (VisionObjectInfo Boundary Box - bug 74, frame number - bug 143)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.15 $
 * $State: Exp $
 * $Date: 2005/08/07 04:11:03 $
 */

#endif
